/**
 * 2018, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.flightrecorder;

import org.json.JSONObject;
import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.controller.objects.ClientDescriptor;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.exception.DIFException;
import pt.digitalis.dif.model.utils.AbstractBeanAttributes;
import pt.digitalis.dif.utils.ObjectFormatter.Format;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.common.DateUtils;
import pt.digitalis.utils.common.NumericUtils;
import pt.digitalis.utils.common.StringUtils;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * The Class RecorderEntry.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Jul 31, 2019
 */
public class RecorderEntry extends AbstractBeanAttributes
{

    /** The id counter. */
    private static long idCounter = 1;

    /** The was added to queue. */
    boolean wasAddedToQueue = false;

    /** The client address. */
    private String clientAddress;

    /** The client descriptor. */
    private JSONObject clientDescriptor;

    /** The elapsed time. */
    private Long elapsedTime = null;

    /** The final exception. */
    private Throwable finalException;

    /** The id. */
    private Long id = idCounter++;

    /** The dif requests. */
    private List<FlightRecorderLogEntry> logEntries = new ArrayList<FlightRecorderLogEntry>();

    /** The session ID. */
    private String sessionID;

    /** The time stamp. */
    private Date timeStamp = new Date();

    /** The user. */
    private JSONObject user;

    /** The user ID. */
    private String userID;

    /** The user ID. */
    private String userName;

    /**
     * Adds the exception.
     *
     * @param difException the dif exception
     */
    public void addException(DIFException difException)
    {
        logEntries.add(new FlightRecorderLogEntry(difException));
    }

    /**
     * Adds the exception.
     *
     * @param exception the exception
     */
    public void addException(Throwable exception)
    {
        logEntries.add(new FlightRecorderLogEntry(exception));
    }

    /**
     * Adds the log.
     *
     * @param className the class name
     * @param level     the level
     * @param message   the message
     */
    public void addLog(String className, LogLevel level, Object message)
    {
        logEntries.add(new FlightRecorderLogEntry(level, message, className));
    }

    /**
     * Adds the request.
     *
     * @param context the context
     */
    public void addRequest(IDIFContext context)
    {
        logEntries.add(new FlightRecorderLogEntry(context));
    }

    /**
     * Adds the log.
     *
     * @param log the log
     */
    public void addSQLLog(SQLExecutionLog log)
    {
        logEntries.add(new FlightRecorderLogEntry(log));
    }

    /**
     * Gets the attribute no graph navigation.
     *
     * @param attributeName the attribute name
     *
     * @return the attribute no graph navigation
     *
     * @see pt.digitalis.dif.model.utils.AbstractBeanAttributes#getAttributeNoGraphNavigation(java.lang.String)
     */
    @Override
    protected Object getAttributeNoGraphNavigation(String attributeName)
    {
        if (Fields.ID.equals(attributeName))
            return this.id;
        else if (Fields.CLIENTIP.equals(attributeName))
            return this.clientAddress;
        else if (Fields.SESSIONID.equals(attributeName))
            return this.sessionID;
        else if (Fields.TIMESTAMP.equals(attributeName))
            return timeStamp;
        else if (Fields.USERID.equals(attributeName))
            return this.userID;
        else if (Fields.USERNAME.equals(attributeName))
            return this.userName;
        else
            return null;
    }

    /**
     * Inspector for the 'clientAddress' attribute.
     *
     * @return the clientAddress value
     */
    public String getClientAddress()
    {
        return clientAddress;
    }

    /**
     * Modifier for the 'clientAddress' attribute.
     *
     * @param clientAddress the new clientAddress value to set
     */
    public void setClientAddress(String clientAddress)
    {
        this.clientAddress = clientAddress;
    }

    /**
     * Inspector for the 'clientDescriptor' attribute.
     *
     * @return the clientDescriptor value
     */
    public JSONObject getClientDescriptor()
    {
        return clientDescriptor;
    }

    /**
     * Modifier for the 'clientDescriptor' attribute.
     *
     * @param clientDescriptor the new clientDescriptor value to set
     */
    public void setClientDescriptor(ClientDescriptor clientDescriptor)
    {
        this.clientDescriptor =
                clientDescriptor == null ? null : clientDescriptor.toObjectFormatter(Format.JSON, null).getJsonObject();
        this.clientAddress = clientDescriptor == null ? null : StringUtils
                .toStringOrNull(clientDescriptor.getAttribute(HTTPConstants.CLIENT_REMOTE_ADDR));
    }

    /**
     * Inspector for the 'elapsedTime' attribute.
     *
     * @return the elapsedTime value
     */
    public Long getElapsedTime()
    {
        return elapsedTime;
    }

    /**
     * Modifier for the 'elapsedTime' attribute.
     *
     * @param elapsedTime the new elapsedTime value to set
     */
    public void setElapsedTime(Long elapsedTime)
    {
        this.elapsedTime = elapsedTime;
    }

    /**
     * Inspector for the 'finalException' attribute.
     *
     * @return the finalException value
     */
    public Throwable getFinalException()
    {
        return finalException;
    }

    /**
     * Modifier for the 'finalException' attribute.
     *
     * @param finalException the new finalException value to set
     */
    public void setFinalException(Throwable finalException)
    {
        this.finalException = finalException;
    }

    /**
     * Inspector for the 'id' attribute.
     *
     * @return the id value
     */
    public Long getId()
    {
        return id;
    }

    /**
     * Modifier for the 'id' attribute.
     *
     * @param id the new id value to set
     */
    public void setId(Long id)
    {
        this.id = id;
    }

    /**
     * Inspector for the 'logEntries' attribute.
     *
     * @return the logEntries value
     */
    public List<FlightRecorderLogEntry> getLogEntries()
    {
        return logEntries;
    }

    /**
     * Inspector for the 'sessionID' attribute.
     *
     * @return the sessionID value
     */
    public String getSessionID()
    {
        return sessionID;
    }

    /**
     * Modifier for the 'sessionID' attribute.
     *
     * @param sessionID the new sessionID value to set
     */
    public void setSessionID(String sessionID)
    {
        this.sessionID = sessionID;
    }

    /**
     * Inspector for the 'timeStamp' attribute.
     *
     * @return the timeStamp value
     */
    public Date getTimeStamp()
    {
        return timeStamp;
    }

    /**
     * Modifier for the 'timeStamp' attribute.
     *
     * @param timeStamp the new timeStamp value to set
     */
    public void setTimeStamp(Date timeStamp)
    {
        this.timeStamp = timeStamp;
    }

    /**
     * Inspector for the 'user' attribute.
     *
     * @return the user value
     */
    public JSONObject getUser()
    {
        return user;
    }

    /**
     * Modifier for the 'user' attribute.
     *
     * @param user the new user value to set
     */
    public void setUser(IDIFUser user)
    {
        this.user = user == null ? null : user.toObjectFormatter(Format.JSON, null).getJsonObject();
        this.userID = user == null ? null : user.getID();
        this.userName = user == null ? null : user.getName();
    }

    /**
     * Inspector for the 'userID' attribute.
     *
     * @return the userID value
     */
    public String getUserID()
    {
        return userID;
    }

    /**
     * Modifier for the 'userID' attribute.
     *
     * @param userID the new userID value to set
     */
    public void setUserID(String userID)
    {
        this.userID = userID;
    }

    /**
     * Inspector for the 'userName' attribute.
     *
     * @return the userName value
     */
    public String getUserName()
    {
        return userName;
    }

    /**
     * Modifier for the 'userName' attribute.
     *
     * @param userName the new userName value to set
     */
    public void setUserName(String userName)
    {
        this.userName = userName;
    }

    /**
     * Inspector for the 'wasAddedToQueue' attribute.
     *
     * @return the wasAddedToQueue value
     */
    public boolean isWasAddedToQueue()
    {
        return wasAddedToQueue;
    }

    /**
     * Modifier for the 'wasAddedToQueue' attribute.
     *
     * @param wasAddedToQueue the new wasAddedToQueue value to set
     */
    public void setWasAddedToQueue(boolean wasAddedToQueue)
    {
        this.wasAddedToQueue = wasAddedToQueue;
    }

    /**
     * Sets the attribute.
     *
     * @param attributeName  the attribute name
     * @param attributeValue the attribute value
     *
     * @see pt.digitalis.utils.common.IBeanAttributes#setAttribute(java.lang.String, java.lang.Object)
     */
    @Override
    public void setAttribute(String attributeName, Object attributeValue)
    {
        // This method will only be used by the DataSet filtering process. So we assume we can create dummy objects for
        // the filtering values and that being dummy objects they will always be empty. If we call these methods on real
        // objects they will not change the forbiden objects like user or client descriptor

        if (Fields.ID.equals(attributeName) && this.id == null)
            this.id = (Long) attributeValue;
        else if (Fields.CLIENTIP.equals(attributeName) && this.clientAddress == null)
            this.clientAddress = StringUtils.toStringOrNull(attributeValue);
        else if (Fields.SESSIONID.equals(attributeName) && this.sessionID == null)
            this.sessionID = StringUtils.toStringOrNull(attributeValue);
        else if (Fields.TIMESTAMP.equals(attributeName) && this.timeStamp == null)
            this.timeStamp = (Date) attributeValue;
        else if (Fields.USERID.equals(attributeName) && this.userID == null)
            this.userID = StringUtils.toStringOrNull(attributeValue);
        else if (Fields.USERNAME.equals(attributeName) && this.userName == null)
            this.userName = StringUtils.toStringOrNull(attributeValue);
    }

    /**
     * Sets the attribute from string.
     *
     * @param attributeName  the attribute name
     * @param attributeValue the attribute value
     *
     * @see pt.digitalis.utils.common.IBeanAttributes#setAttributeFromString(java.lang.String, java.lang.String)
     */
    @Override
    public void setAttributeFromString(String attributeName, String attributeValue)
    {
        // This method will only be used by the DataSet filtering process. So we assume we can create dummy objects for
        // the filtering values and that being dummy objects they will always be empty. If we call these methods on real
        // objects they will not change the forbiden objects like user or client descriptor

        if (Fields.ID.equals(attributeName) && this.id == null)
            this.id = NumericUtils.toLong(attributeValue);
        else if (Fields.CLIENTIP.equals(attributeName) && this.clientAddress == null)
            this.clientAddress = attributeName;
        else if (Fields.SESSIONID.equals(attributeName) && this.sessionID == null)
            this.sessionID = attributeValue;
        else if (Fields.TIMESTAMP.equals(attributeName) && this.timeStamp == null)
            try
            {
                this.timeStamp = DateUtils.stringToDate(attributeValue);
            }
            catch (ParseException e)
            {
                e.printStackTrace();
            }
        else if (Fields.USERID.equals(attributeName) && this.userID == null)
            this.userID = attributeName;
        else if (Fields.USERNAME.equals(attributeName) && this.userName == null)
            this.userName = attributeName;
    }

    /**
     * Fan attributes enumeration.
     */
    public static class Fields
    {

        /** The Constant CLIENTIP. */
        public static final String CLIENTIP = "clientIP";

        /** The Constant ID. */
        public static final String ID = "id";

        /** The Constant SESSIONID. */
        public static final String SESSIONID = "sessionID";

        /** The Constant TIMESTAMP. */
        public static final String TIMESTAMP = "timeStamp";

        /** The Constant USERID. */
        public static final String USERID = "userID";

        /** The Constant USERNAME. */
        public static final String USERNAME = "userName";

        /**
         * List of existing fields.
         *
         * @return list of field names
         */
        public static List<String> values()
        {
            ArrayList<String> list = new ArrayList<String>();
            list.add(ID);
            list.add(USERID);
            list.add(USERNAME);
            list.add(CLIENTIP);
            list.add(SESSIONID);
            list.add(TIMESTAMP);

            return list;
        }
    }
}
