/**
 * 2018, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.flightrecorder;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.hibernate.jdbc.util.BasicFormatterImpl;
import pt.digitalis.utils.common.DateUtils;
import pt.digitalis.utils.common.IBeanAttributes;
import pt.digitalis.utils.common.StringUtils;

import java.text.ParseException;
import java.util.Date;

/**
 * The Class SQLExecutionLog.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Mar 5, 2019
 */
public class SQLExecutionLog implements IBeanAttributes
{

    /** The last ID generated. */
    private static long lastID = 0;

    /** The execution time. */
    private Long averageExecutionTime;

    /** The database ID. */
    private String databaseID;

    /** The exception. */
    private Exception exception;

    /** The executions. */
    private Long executions = 1L;

    /** The date. */
    private Date firstDate;

    /** The id. */
    private long id;

    /** The date. */
    private Date lastDate;

    /** The last stack trace. */
    private String lastStackTrace;

    /** The sql. */
    private String sql;

    /** The success. */
    private boolean success;

    /** The execution time. */
    private Long totalExecutionTime;

    /**
     * Instantiates a new SQL execution log.
     *
     * @param date          the date
     * @param databaseID    the database ID
     * @param sql           the sql
     * @param executionTime the execution time
     */
    public SQLExecutionLog(Date date, String databaseID, String sql, Long executionTime)
    {
        this(date, databaseID, sql, executionTime, null);
    }

    /**
     * Instantiates a new SQL execution log.
     *
     * @param date          the date
     * @param databaseID    the database ID
     * @param sql           the sql
     * @param executionTime the execution time
     * @param exception     the exception
     */
    public SQLExecutionLog(Date date, String databaseID, String sql, Long executionTime, Exception exception)
    {
        super();
        this.id = ++lastID;
        this.firstDate = date;
        this.lastDate = date;
        this.databaseID = databaseID;
        this.sql = sql;
        this.averageExecutionTime = executionTime;
        this.totalExecutionTime = executionTime;
        this.exception = exception;
        this.success = (exception == null);
        this.lastStackTrace = ExceptionUtils.getFullStackTrace(new Throwable());
    }

    /**
     * @see pt.digitalis.utils.common.IBeanAttributes#getAttribute(java.lang.String)
     */
    @Override
    public Object getAttribute(String attributeName)
    {
        if ("averageExecutionTime".equalsIgnoreCase(attributeName))
            return averageExecutionTime;
        else if ("databaseID".equals(attributeName))
            return databaseID;
        if ("exception".equalsIgnoreCase(attributeName))
            return exception;
        if ("executions".equalsIgnoreCase(attributeName))
            return executions;
        if ("firstDate".equalsIgnoreCase(attributeName))
            return firstDate;
        if ("id".equalsIgnoreCase(attributeName))
            return id;
        if ("lastDate".equalsIgnoreCase(attributeName))
            return lastDate;
        if ("lastStackTrace".equalsIgnoreCase(attributeName))
            return lastStackTrace;
        if ("sql".equalsIgnoreCase(attributeName))
            return sql;
        if ("success".equalsIgnoreCase(attributeName))
            return success;
        if ("totalExecutionTime".equalsIgnoreCase(attributeName))
            return totalExecutionTime;
        else
            return null;
    }

    /**
     * @see pt.digitalis.utils.common.IBeanPropertyInspector#getAttributeAsString(java.lang.String)
     */
    @Override
    public String getAttributeAsString(String attributeName)
    {
        if ("averageExecutionTime".equalsIgnoreCase(attributeName))
            return StringUtils.toStringOrNull(averageExecutionTime);
        else if ("databaseID".equals(attributeName))
            return databaseID;
        if ("exception".equalsIgnoreCase(attributeName))
            return StringUtils.toStringOrNull(exception);
        if ("executions".equalsIgnoreCase(attributeName))
            return StringUtils.toStringOrNull(executions);
        if ("firstDate".equalsIgnoreCase(attributeName))
            return StringUtils.toStringOrNull(firstDate);
        if ("id".equalsIgnoreCase(attributeName))
            return StringUtils.toStringOrNull(id);
        if ("lastDate".equalsIgnoreCase(attributeName))
            return StringUtils.toStringOrNull(lastDate);
        if ("lastStackTrace".equalsIgnoreCase(attributeName))
            return lastStackTrace;
        if ("sql".equalsIgnoreCase(attributeName))
            return sql;
        if ("success".equalsIgnoreCase(attributeName))
            return StringUtils.toStringOrNull(success);
        if ("totalExecutionTime".equalsIgnoreCase(attributeName))
            return StringUtils.toStringOrNull(totalExecutionTime);
        else
            return null;
    }

    /**
     * Inspector for the 'averageExecutionTime' attribute.
     *
     * @return the averageExecutionTime value
     */
    public Long getAverageExecutionTime()
    {
        return averageExecutionTime;
    }

    /**
     * Modifier for the 'averageExecutionTime' attribute.
     *
     * @param averageExecutionTime the new averageExecutionTime value to set
     */
    public void setAverageExecutionTime(Long averageExecutionTime)
    {
        this.averageExecutionTime = averageExecutionTime;
    }

    /**
     * Inspector for the 'databaseID' attribute.
     *
     * @return the databaseID value
     */
    public String getDatabaseID()
    {
        return databaseID;
    }

    /**
     * Modifier for the 'databaseID' attribute.
     *
     * @param databaseID the new databaseID value to set
     */
    public void setDatabaseID(String databaseID)
    {
        this.databaseID = databaseID;
    }

    /**
     * Inspector for the 'exception' attribute.
     *
     * @return the exception value
     */
    public Exception getException()
    {
        return exception;
    }

    /**
     * Modifier for the 'exception' attribute.
     *
     * @param exception the new exception value to set
     */
    public void setException(Exception exception)
    {
        this.exception = exception;
    }

    /**
     * Inspector for the 'executions' attribute.
     *
     * @return the executions value
     */
    public Long getExecutions()
    {
        return executions;
    }

    /**
     * Modifier for the 'executions' attribute.
     *
     * @param executions the new executions value to set
     */
    public void setExecutions(Long executions)
    {
        this.executions = executions;
    }

    /**
     * Inspector for the 'firstDate' attribute.
     *
     * @return the firstDate value
     */
    public Date getFirstDate()
    {
        return firstDate;
    }

    /**
     * Modifier for the 'firstDate' attribute.
     *
     * @param firstDate the new firstDate value to set
     */
    public void setFirstDate(Date firstDate)
    {
        this.firstDate = firstDate;
    }

    /**
     * Inspector for the 'id' attribute.
     *
     * @return the id value
     */
    public long getId()
    {
        return id;
    }

    /**
     * Modifier for the 'id' attribute.
     *
     * @param id the new id value to set
     */
    public void setId(long id)
    {
        this.id = id;
    }

    /**
     * Inspector for the 'lastDate' attribute.
     *
     * @return the lastDate value
     */
    public Date getLastDate()
    {
        return lastDate;
    }

    /**
     * Modifier for the 'lastDate' attribute.
     *
     * @param lastDate the new lastDate value to set
     */
    public void setLastDate(Date lastDate)
    {
        this.lastDate = lastDate;
    }

    /**
     * Inspector for the 'lastStackTrace' attribute.
     *
     * @return the lastStackTrace value
     */
    public String getLastStackTrace()
    {
        return lastStackTrace;
    }

    /**
     * Modifier for the 'lastStackTrace' attribute.
     *
     * @param lastStackTrace the new lastStackTrace value to set
     */
    public void setLastStackTrace(String lastStackTrace)
    {
        this.lastStackTrace = lastStackTrace;
    }

    /**
     * Inspector for the 'sql' attribute.
     *
     * @return the sql value
     */
    public String getSql()
    {
        try
        {
            return new BasicFormatterImpl().format(sql);
        }
        catch (Exception e)
        {
            //Do nothing and return the original sql

            return sql;
        }
    }

    /**
     * Modifier for the 'sql' attribute.
     *
     * @param sql the new sql value to set
     */
    public void setSql(String sql)
    {
        this.sql = sql;
    }

    /**
     * Inspector for the 'totalExecutionTime' attribute.
     *
     * @return the totalExecutionTime value
     */
    public Long getTotalExecutionTime()
    {
        return totalExecutionTime;
    }

    /**
     * Modifier for the 'totalExecutionTime' attribute.
     *
     * @param totalExecutionTime the new totalExecutionTime value to set
     */
    public void setTotalExecutionTime(Long totalExecutionTime)
    {
        this.totalExecutionTime = totalExecutionTime;
    }

    /**
     * Increment executions.
     *
     * @param newExecutionTime the new execution time
     *
     * @return the SQL execution log
     */
    public SQLExecutionLog incrementExecutions(Long newExecutionTime)
    {
        this.executions++;
        this.totalExecutionTime += newExecutionTime;

        this.averageExecutionTime = totalExecutionTime / executions;

        return this;
    }

    /**
     * Inspector for the 'success' attribute.
     *
     * @return the success value
     */
    public boolean isSuccess()
    {
        return success;
    }

    /**
     * Modifier for the 'success' attribute.
     *
     * @param success the new success value to set
     */
    public void setSuccess(boolean success)
    {
        this.success = success;
    }

    /**
     * @see pt.digitalis.utils.common.IBeanAttributes#setAttribute(java.lang.String, java.lang.Object)
     */
    @Override
    public void setAttribute(String attributeName, Object attributeValue)
    {
        if ("averageExecutionTime".equalsIgnoreCase(attributeName))
            setAverageExecutionTime((Long) attributeValue);
        if ("databaseID".equalsIgnoreCase(attributeName))
            setDatabaseID((String) attributeValue);
        if ("exception".equalsIgnoreCase(attributeName))
            setException((Exception) attributeValue);
        if ("executions".equalsIgnoreCase(attributeName))
            setExecutions((Long) attributeValue);
        if ("firstDate".equalsIgnoreCase(attributeName))
            setFirstDate((Date) attributeValue);
        if ("id".equalsIgnoreCase(attributeName))
            setId((Long) attributeValue);
        if ("lastDate".equalsIgnoreCase(attributeName))
            setLastDate((Date) attributeValue);
        if ("lastStackTrace".equalsIgnoreCase(attributeName))
            setLastStackTrace((String) attributeValue);
        if ("sql".equalsIgnoreCase(attributeName))
            setSql((String) attributeValue);
        if ("success".equalsIgnoreCase(attributeName))
            setSuccess((Boolean) attributeValue);
        if ("totalExecutionTime".equalsIgnoreCase(attributeName))
            setTotalExecutionTime((Long) attributeValue);
    }

    /**
     * @see pt.digitalis.utils.common.IBeanAttributes#setAttributeFromString(java.lang.String, java.lang.String)
     */
    @Override
    public void setAttributeFromString(String attributeName, String attributeValue)
    {
        try
        {
            if ("averageExecutionTime".equalsIgnoreCase(attributeName))
                setAverageExecutionTime(Long.parseLong(attributeValue));
            if ("databaseID".equalsIgnoreCase(attributeName))
                setDatabaseID(attributeValue);
            if ("exception".equalsIgnoreCase(attributeName))
                ;
            if ("executions".equalsIgnoreCase(attributeName))
                setExecutions(Long.parseLong(attributeValue));
            if ("firstDate".equalsIgnoreCase(attributeName))
                setFirstDate(DateUtils.stringToSimpleDate(attributeValue));
            if ("id".equalsIgnoreCase(attributeName))
                setId(Long.parseLong(attributeValue));
            if ("lastDate".equalsIgnoreCase(attributeName))
                setLastDate(DateUtils.stringToSimpleDate(attributeValue));
            if ("lastStackTrace".equalsIgnoreCase(attributeName))
                setLastStackTrace(attributeValue);
            if ("sql".equalsIgnoreCase(attributeName))
                setSql(attributeValue);
            if ("success".equalsIgnoreCase(attributeName))
                setSuccess(Boolean.parseBoolean(attributeValue));
            if ("totalExecutionTime".equalsIgnoreCase(attributeName))
                setTotalExecutionTime(Long.parseLong(attributeValue));
        }
        catch (ParseException e)
        {
            e.printStackTrace();
        }
    }

    /**
     * @see pt.digitalis.utils.common.IBeanAttributes#setNestedAttribute(java.lang.String, java.lang.Object)
     */
    @Override
    public void setNestedAttribute(String attributeName, Object attributeValue)
    {
        setAttribute(attributeName, attributeValue);
    }

    /**
     * @see pt.digitalis.utils.common.IBeanAttributes#setNestedAttributeFromString(java.lang.String,
     *         java.lang.String)
     */
    @Override
    public void setNestedAttributeFromString(String attributeName, String attributeValue)
    {
        setAttributeFromString(attributeName, attributeValue);
    }
}
