/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.sanitycheck.manager;

import pt.digitalis.dif.sanitycheck.ISanityCheckTestSuite;
import pt.digitalis.dif.sanitycheck.SanityCheckResult;
import pt.digitalis.dif.sanitycheck.annotations.SanityCheckDependency;
import pt.digitalis.dif.sanitycheck.annotations.SanityCheckFinalize;
import pt.digitalis.dif.sanitycheck.annotations.SanityCheckSetup;
import pt.digitalis.dif.sanitycheck.annotations.SanityCheckTest;
import pt.digitalis.dif.sanitycheck.exceptions.WrongTestMethodSignature;
import pt.digitalis.utils.common.StringUtils;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Defines a given test suite for usage by the {@link SanityCheckManager} internals
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 15 de Jul de 2011
 */
public class SanityCheckTestSuiteDefinition
{

    /** the list of dependencies for execution of the current test suite */
    private List<String> dependencies;

    /** the last execution result */
    private SanityCheckResult executionResult;

    /** The tests finalize method */
    private Method finalizeMethod;

    /** Unique identifier */
    private String id;

    /** the test suite name */
    private String name;

    /** The tests setup method */
    private Method setupMethod;

    /** the test methods */
    private List<TestMethodDefinition> testMethods;

    /** The test suite instance */
    private ISanityCheckTestSuite testSuiteInstance;

    /**
     * @param id
     * @param testSuiteInstance
     */
    public SanityCheckTestSuiteDefinition(String id, ISanityCheckTestSuite testSuiteInstance)
    {
        super();
        this.id = id;
        this.name = StringUtils.camelCaseToString(testSuiteInstance.getClass().getSimpleName());
        this.testSuiteInstance = testSuiteInstance;

        // Dependencies will be lazy initialized from the annotation in the first access.
        this.dependencies = null;

        // Initialize the execution methods
        this.testMethods = new ArrayList<TestMethodDefinition>();

        // Parse all methods...
        for (Method method : testSuiteInstance.getClass().getMethods())
        {
            if (method.isAnnotationPresent(SanityCheckSetup.class))
                setupMethod = method;

            else if (method.isAnnotationPresent(SanityCheckFinalize.class))
                finalizeMethod = method;

            else if (method.isAnnotationPresent(SanityCheckTest.class))
                try
                {
                    testMethods.add(new TestMethodDefinition(testSuiteInstance, method));
                }
                catch (WrongTestMethodSignature e)
                {
                    e.printStackTrace();
                }
        }
    }

    /**
     * @return the dependent list of
     */
    public List<SanityCheckTestSuiteDefinition> getDependencies()
    {
        if (dependencies == null)
        {
            // First access. Parse dependencies
            dependencies = new ArrayList<String>();
            SanityCheckDependency dependenciesAnnotation =
                    testSuiteInstance.getClass().getAnnotation(SanityCheckDependency.class);

            if (dependenciesAnnotation != null)
            {
                String[] depencencyList = dependenciesAnnotation.value().split(",");
                dependencies = Arrays.asList(depencencyList);
            }
        }

        List<SanityCheckTestSuiteDefinition> result = new ArrayList<SanityCheckTestSuiteDefinition>();

        for (String dependencyID : dependencies)
        {
            result.add(SanityCheckManager.getTestSuites().get(dependencyID));
        }

        return result;
    }

    /**
     * Inspector for the 'executionResult' attribute.
     *
     * @return the executionResult value
     */
    public SanityCheckResult getExecutionResult()
    {
        return executionResult;
    }

    /**
     * Modifier for the 'executionResult' attribute.
     *
     * @param executionResult the new executionResult value to set
     */
    public void setExecutionResult(SanityCheckResult executionResult)
    {
        this.executionResult = executionResult;
    }

    /**
     * Inspector for the 'finalizeMethod' attribute.
     *
     * @return the finalizeMethod value
     */
    public Method getFinalizeMethod()
    {
        return finalizeMethod;
    }

    /**
     * Inspector for the 'id' attribute.
     *
     * @return the id value
     */
    public String getId()
    {
        return id;
    }

    /**
     * Inspector for the 'name' attribute.
     *
     * @return the name value
     */
    public String getName()
    {
        return name;
    }

    /**
     * Inspector for the 'setupMethod' attribute.
     *
     * @return the setupMethod value
     */
    public Method getSetupMethod()
    {
        return setupMethod;
    }

    /**
     * Inspector for the 'testMethods' attribute.
     *
     * @return the testMethods value
     */
    public List<TestMethodDefinition> getTestMethods()
    {
        return testMethods;
    }

    /**
     * Inspector for the 'testSuiteInstance' attribute.
     *
     * @return the testSuiteInstance value
     */
    public ISanityCheckTestSuite getTestSuiteInstance()
    {
        return testSuiteInstance;
    }

    /**
     * Runs the finalize method of the test suite
     *
     * @exception InvocationTargetException
     * @exception IllegalAccessException
     * @exception IllegalArgumentException
     */
    public void runTestsFinalize() throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
    {
        if (finalizeMethod != null)
            finalizeMethod.invoke(testSuiteInstance);
    }

    /**
     * Runs the setup method of the test suite
     *
     * @exception InvocationTargetException
     * @exception IllegalAccessException
     * @exception IllegalArgumentException
     */
    public void runTestsSetup() throws IllegalArgumentException, IllegalAccessException, InvocationTargetException
    {
        if (setupMethod != null)
            setupMethod.invoke(testSuiteInstance);
    }
}
