/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.sanitycheck.manager;

import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.dem.annotations.AnnotationTags;
import pt.digitalis.dif.sanitycheck.ExecutionResult;
import pt.digitalis.dif.sanitycheck.ISanityCheckTestSuite;
import pt.digitalis.dif.sanitycheck.TestResult;
import pt.digitalis.dif.sanitycheck.annotations.SanityCheckTest;
import pt.digitalis.dif.sanitycheck.exceptions.WrongTestMethodSignature;
import pt.digitalis.dif.utils.DIFConstants;
import pt.digitalis.utils.common.StringUtils;

import java.lang.reflect.Method;

/**
 * Defines a Sanity Check Test Method form a {@link ISanityCheckTestSuite}
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 15 de Jul de 2011
 */
public class TestMethodDefinition
{

    /** the name of the test */
    private String name;

    /** if true the test can only be ran if an administrator is logged in */
    private boolean onlyAdministrators;

    /** if T will pass the DIF context to the test method invocation */
    private boolean passContext;

    /** the method host test suite instance */
    private ISanityCheckTestSuite testSuiteInstance;

    /** the method to run */
    private Method theMethod;

    /**
     * @param testSuiteInstance
     * @param theMethod
     *
     * @exception WrongTestMethodSignature the methods can only have a single {@link IDIFContext} parameter, or none at
     *                                     all
     */
    public TestMethodDefinition(ISanityCheckTestSuite testSuiteInstance, Method theMethod)
            throws WrongTestMethodSignature
    {
        SanityCheckTest declarationAnnotation = theMethod.getAnnotation(SanityCheckTest.class);

        this.theMethod = theMethod;
        this.onlyAdministrators = declarationAnnotation.onlyAdministrators();
        this.testSuiteInstance = testSuiteInstance;

        if (declarationAnnotation.name().equals(AnnotationTags.NONE))
            this.name = StringUtils.camelCaseToString(theMethod.getName());
        else
            this.name = StringUtils.camelCaseToString(declarationAnnotation.name());

        // Method has parameters... analise them...
        if (theMethod.getParameterTypes().length > 0)
        {
            if (theMethod.getParameterTypes().length > 1)
                // More than 1 parameters... invalid!
                throw new WrongTestMethodSignature(theMethod);

            else if (theMethod.getParameterTypes()[0] != IDIFContext.class)
                // The single parameter is not of type IDIFContext... invalid!
                throw new WrongTestMethodSignature(theMethod);

            else
                // A single parameter of the type IDIFContext... Valid!
                this.passContext = true;
        }

        // Check the return type, as TestResult...
        if (theMethod.getReturnType() != TestResult.class)
            throw new WrongTestMethodSignature(theMethod);
    }

    /**
     * Inspector for the 'name' attribute.
     *
     * @return the name value
     */
    public String getName()
    {
        return name;
    }

    /**
     * Modifier for the 'name' attribute.
     *
     * @param name the new name value to set
     */
    public void setName(String name)
    {
        this.name = name;
    }

    /**
     * Inspector for the 'onlyAdministrators' attribute.
     *
     * @return the onlyAdministrators value
     */
    public boolean isOnlyAdministrators()
    {
        return onlyAdministrators;
    }

    /**
     * Modifier for the 'onlyAdministrators' attribute.
     *
     * @param onlyAdministrators the new onlyAdministrators value to set
     */
    public void setOnlyAdministrators(boolean onlyAdministrators)
    {
        this.onlyAdministrators = onlyAdministrators;
    }

    /**
     * Runs the test
     *
     * @param context the current DIF context
     *
     * @return the result of the test execution
     */
    public TestResult run(IDIFContext context)
    {
        TestResult result;

        try
        {
            if (onlyAdministrators && (!context.getSession().isLogged() || !context.getSession().getUser().getGroupIDs()
                    .contains(DIFConstants.ADMINISTRATORS_GROUP.toLowerCase())))
            {
                result = new TestResult(ExecutionResult.SKIPPED);
            }
            else
            {
                if (passContext)
                    result = (TestResult) theMethod.invoke(testSuiteInstance, context);
                else
                    result = (TestResult) theMethod.invoke(testSuiteInstance);
            }
        }
        catch (Exception e)
        {
            result = new TestResult(ExecutionResult.FAILED);
            result.setErrorMessage(e.getMessage());
        }

        return result;
    }
}
