/**
 * 2019, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 http://www.digitalis.pt
 */
package pt.digitalis.dif.servermanager;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.json.JSONObject;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.servermanager.jobs.KeepServerAliveJob;
import pt.digitalis.dif.servermanager.jobs.ProcessServerMessagesJob;
import pt.digitalis.dif.servermanager.jobs.RefreshServerListJob;
import pt.digitalis.dif.servermanager.messages.IServerMessage;
import pt.digitalis.dif.startup.DIFStartupConfiguration;
import pt.digitalis.dif.utils.jobs.DIFJobsManager;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.ConfigurationException;

import java.util.ArrayList;
import java.util.List;

// TODO: Auto-generated Javadoc

/**
 * The Class AbstractServerManager.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Apr 23, 2019
 */
public abstract class AbstractServerManager implements IServerManager
{

    /** The Constant the defined the mili-seconds a server can be without activity before is considered inactive. */
    public static final long INACTIVE_AFTER_MINUTES = 10; // 10 minutes

    /** The server node. */
    protected ServerApplicationNodeInstance serverNode = null;

    /** The port has been initialized. */
    boolean portHasBeenInitialized = false;

    /** The comunication port. */
    private String comunicationPort = null;

    /**
     * Builds the server application node instance.
     *
     * @return the server application node instance
     *
     * @exception ServerManagerException the server manager exception
     */
    protected ServerApplicationNodeInstance buildNewServerApplicationNodeInstanceObject() throws ServerManagerException
    {
        String serverURL;
        ServerApplicationNodeInstance serverNode;

        try
        {
            serverURL = ServerManagerConfigurations.getInstance().getServerURLForServerToServerSync();
        }
        catch (ConfigurationException e)
        {
            throw new ServerManagerException(e);
        }

        if (StringUtils.isBlank(serverURL))
            serverURL = this.getServerCommunicator()
                    .getServerURL(this.getServerCommunicator().getCurrentServerIPAddress(),
                            this.getCurrentServerCommunicationPort(),
                            this.getServerCommunicator().getCurrentServerContextRoot());

        serverNode = new ServerApplicationNodeInstance(DIFStartupConfiguration.getMachineIDForConfigurations(),
                this.getServerCommunicator().getCurrentServerIPAddress(),
                this.getServerCommunicator().getCurrentServerContextRoot(), this.getCurrentServerCommunicationPort(),
                serverURL);

        return serverNode;
    }

    /**
     * Gets the apps running on this server instance.
     *
     * @return the apps running on this server instance
     *
     * @exception ServerManagerException the server manager exception
     * @see pt.digitalis.dif.servermanager.IServerManager#getAppsRunningOnThisServerInstance()
     */
    public List<ServerApplicationNodeInstance> getAppsRunningOnThisServerInstance() throws ServerManagerException
    {
        ArrayList<ServerApplicationNodeInstance> result = new ArrayList<ServerApplicationNodeInstance>();

        for (ServerApplicationNodeInstance server : this.getAllServers())
        {
            if (this.getServerNode().getServerIP().equals(server.getServerIP()) &&
                this.getServerNode().getPort().equals(server.getPort()))
                result.add(server);
        }

        return result;
    }

    /**
     * Gets the comunication port.
     *
     * @return the comunicationPort
     */
    public String getComunicationPort()
    {
        return comunicationPort;
    }

    /**
     * Gets the current server communication port.
     *
     * @return the current server communication port
     *
     * @exception ServerManagerException the server manager exception
     * @see pt.digitalis.dif.servermanager.IServerManager#getCurrentServerCommunicationPort()
     */
    public String getCurrentServerCommunicationPort() throws ServerManagerException
    {
        if (StringUtils.isNotBlank(comunicationPort))
            return comunicationPort;
        else
            try
            {
                return StringUtils
                        .toStringOrNull(ServerManagerConfigurations.getInstance().getPreviousCommunicationPort());
            }
            catch (ConfigurationException e)
            {
                throw new ServerManagerException(e);
            }
    }

    /**
     * Gets the other instances of this app.
     *
     * @return the other instances of this app
     *
     * @exception ServerManagerException the server manager exception
     * @see pt.digitalis.dif.servermanager.IServerManager#getOtherInstancesOfThisApp()
     */
    public List<ServerApplicationNodeInstance> getOtherInstancesOfThisApp() throws ServerManagerException
    {
        ArrayList<ServerApplicationNodeInstance> result = new ArrayList<ServerApplicationNodeInstance>();

        for (ServerApplicationNodeInstance server : this.getAllServers())
        {
            if (this.getServerNode().getContextRootID().equals(server.getContextRootID()))
                result.add(server);
        }

        return result;
    }

    /**
     * Gets the server communicator.
     *
     * @return the server communicator
     */
    protected final IServerManagerCommunicator getServerCommunicator()
    {
        return DIFIoCRegistry.getRegistry().getImplementation(IServerManagerCommunicator.class);
    }

    /**
     * Gets the server node.
     *
     * @return the server node
     *
     * @see pt.digitalis.dif.servermanager.IServerManager#getServerNode()
     */
    public ServerApplicationNodeInstance getServerNode()
    {
        return serverNode;
    }

    /**
     * Initialize.
     *
     * @exception ServerManagerException the server manager exception
     * @see pt.digitalis.dif.servermanager.IServerManager#initialize()
     */
    public final void initialize() throws ServerManagerException
    {
        internalInitialize();

        this.serverNode = buildNewServerApplicationNodeInstanceObject();

        registerServer();
        discoverServers();

        try
        {
            startMaintenanceJobs();
        }
        catch (ConfigurationException e)
        {
            throw new ServerManagerException(e);
        }
    }

    /**
     * Internal initialize.
     *
     * @exception ServerManagerException the server manager exception
     */
    abstract void internalInitialize() throws ServerManagerException;

    /**
     * Checks if is port has been initialized.
     *
     * @return the portHasBeenInitialized
     */
    public boolean isPortHasBeenInitialized()
    {
        return portHasBeenInitialized;
    }

    /**
     * Migrate messages from one server to another.
     *
     * @param previousServerNode  the previous server node
     * @param newServerNode       the new server node
     * @param onlyPendingMessages if T will migrate only pending messages
     */
    protected abstract void migrateMessagesBetweenServerEntries(ServerApplicationNodeInstance previousServerNode,
            ServerApplicationNodeInstance newServerNode, boolean onlyPendingMessages);

    /**
     * Process server message.
     *
     * @param message the message
     * @param sender  the sender
     *
     * @return the server message execution result
     */
    protected ServerMessageExecutionResult processServerMessage(String message, ServerApplicationNodeInstance sender)
    {
        // Get the ID of the message type
        JSONObject jsonObj = new JSONObject(message);
        String messageTypeID = jsonObj.getString("messageTypeID");

        // Get the message class object (empty object just to get it's class)
        IServerMessage serverMessage =
                DIFIoCRegistry.getRegistry().getImplementation(IServerMessage.class, messageTypeID);

        // Convert JSON to it's class object representation
        ObjectMapper mapper = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        try
        {
            serverMessage = mapper.readValue(message, serverMessage.getClass());
            return serverMessage.execute(sender);
        }
        catch (Exception e)
        {
            return new ServerMessageExecutionResult(e);
        }
    }

    /**
     * Start maintenance jobs.
     *
     * @exception ServerManagerException the server manager exception
     * @exception ConfigurationException
     */
    protected void startMaintenanceJobs() throws ServerManagerException, ConfigurationException
    {
        if (this.allowServerCommunication())
        {
            DIFJobsManager.addJob(new KeepServerAliveJob(this));
            DIFJobsManager.addJob(new RefreshServerListJob(this));
            DIFJobsManager.addJob(new ProcessServerMessagesJob(this));
        }
    }

    /**
     * Update HTTP port.
     *
     * @param portNumber the port number
     *
     * @exception ServerManagerException the server manager exception
     * @see pt.digitalis.dif.servermanager.IServerManager#updateHTTPPort(java.lang.Integer)
     */
    public void updateHTTPPort(Integer portNumber) throws ServerManagerException
    {
        if (!isPortHasBeenInitialized())
        {
            try
            {
                ServerApplicationNodeInstance previousServerNode = serverNode;

                comunicationPort = portNumber.toString();
                serverNode = this.buildNewServerApplicationNodeInstanceObject();
                serverNode.setServerManagementObject(previousServerNode.getServerManagementObject());
                serverNode.setStartupTime(previousServerNode.getStartupTime());
                registerServer();

                ServerApplicationNodeInstance newServerNode = serverNode;

                migrateMessagesBetweenServerEntries(previousServerNode, newServerNode, true);

                ServerManagerConfigurations.getInstance().setPreviousCommunicationPort(portNumber.longValue());

                portHasBeenInitialized = true;
            }
            catch (ConfigurationException e)
            {
                throw new ServerManagerException(e);
            }
        }
    }
}
