/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.startup;

import pt.digitalis.dif.codegen.DIFCodeGenerator;
import pt.digitalis.dif.codegen.templates.TemplateResources;
import pt.digitalis.dif.controller.interfaces.IModelManager;
import pt.digitalis.dif.controller.security.managers.IAuthenticationManager;
import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.managers.impl.IdentityManagerStaticImpl;
import pt.digitalis.dif.dem.interfaces.IApplication;
import pt.digitalis.dif.dem.interfaces.IApplicationConfiguration;
import pt.digitalis.dif.dem.interfaces.IApplicationPrivate;
import pt.digitalis.dif.dem.interfaces.IDIFAPI;
import pt.digitalis.dif.dem.interfaces.IProvider;
import pt.digitalis.dif.dem.interfaces.IService;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.dem.managers.impl.UsageIssuesManagerImpl;
import pt.digitalis.dif.dem.objects.issues.UsageIssue;
import pt.digitalis.dif.exception.DIFUncaughtExceptionHandler;
import pt.digitalis.dif.exception.InternalFrameworkException;
import pt.digitalis.dif.exception.codegen.DIFCodeGenerationException;
import pt.digitalis.dif.flightrecorder.FlightRecorder;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.servermanager.ServerManager;
import pt.digitalis.dif.servermanager.ServerManagerException;
import pt.digitalis.dif.utils.logging.AuditContext;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.dif.utils.logging.DIFLoggerInterceptorImpl;
import pt.digitalis.dif.utils.multithreading.IMultiThreadExecutor;
import pt.digitalis.dif.utils.multithreading.MultiThreadExecuterUtils;
import pt.digitalis.dif.utils.multithreading.TaskExecutor;
import pt.digitalis.iss.ISSManager;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.bytecode.exceptions.CodeGenerationException;
import pt.digitalis.utils.common.Chronometer;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.ConfigurationException;
import pt.digitalis.utils.config.IConfigurations;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;
import pt.digitalis.utils.ioc.IIoCRegistry;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Stack;

final public class DIFInitializer
{

    final static public String REPLICA_PREFS = "replica.prefs";

    final static private String INDENTATION = "  ";

    static public Date bootTime = null;

    static private boolean initialized = false;

    static private void addIssuesToBuffer(String location, StringBuffer buffer, String indentPrefix)
    {
        // Reuse var
        List<UsageIssue> issues = null;

        issues = UsageIssuesManagerImpl.getInstance().getIssues(location);

        for (UsageIssue usageIssue : issues)
        {
            buffer.append(indentPrefix + "  => " + usageIssue.getIssueType() + ": " + usageIssue.getIssueDescription() +
                          "\n");
        }
    }

    static private void dumpDEM()
    {
        // When in debug dump the full DEM structure to the log.
        StringBuffer buffer = new StringBuffer();
        Stack<String> indentationsStack = new Stack<String>();
        String currentIndentation = null;

        buffer.append("\n\nDumping DEM structure:\n");
        buffer.append("-----------------------------------------\n");

        IDEMManager dem = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class);

        Iterator<IProvider> providers = dem.getProviders().values().iterator();

        while (providers.hasNext())
        {
            IProvider provider = providers.next();

            if (providers.hasNext())
            {
                currentIndentation = INDENTATION + "|   ";
                buffer.append(INDENTATION + "|-- Provider: " + provider.getName() + " (" + provider.getID() + ")\n");
            }
            else
            {
                currentIndentation = INDENTATION + "    ";
                buffer.append(INDENTATION + "`-- Provider: " + provider.getName() + " (" + provider.getID() + ")\n");
            }

            if (UsageIssuesManagerImpl.getInstance().hasIssues(provider.getOriginalClassName()))
                addIssuesToBuffer(provider.getOriginalClassName(), buffer, currentIndentation);

            // Applications iteration
            Iterator<IApplication> applications = provider.getApplications().values().iterator();

            while (applications.hasNext())
            {
                IApplication application = applications.next();
                indentationsStack.add(currentIndentation);

                if (applications.hasNext())
                {
                    buffer.append(
                            currentIndentation + INDENTATION + "|-- Application: " + application.getName() + " (" +
                            application.getID() + ")\n");
                    currentIndentation += INDENTATION + "|   ";
                }
                else
                {
                    buffer.append(
                            currentIndentation + INDENTATION + "`-- Application: " + application.getName() + " (" +
                            application.getID() + ")\n");
                    currentIndentation += INDENTATION + "    ";
                }

                if (UsageIssuesManagerImpl.getInstance().hasIssues(application.getOriginalClassName()))
                    addIssuesToBuffer(application.getOriginalClassName(), buffer, currentIndentation);

                // Services iteration
                Iterator<IService> services = application.getServices().values().iterator();

                while (services.hasNext())
                {
                    IService service = services.next();
                    indentationsStack.add(currentIndentation);

                    if (services.hasNext())
                    {
                        buffer.append(currentIndentation + INDENTATION + "|-- Service: " + service.getName() + " (" +
                                      service.getID() + ")\n");
                        currentIndentation += INDENTATION + "|   ";
                    }
                    else
                    {
                        buffer.append(currentIndentation + INDENTATION + "`-- Service: " + service.getName() + " (" +
                                      service.getID() + ")\n");
                        currentIndentation += INDENTATION + "    ";
                    }

                    if (UsageIssuesManagerImpl.getInstance().hasIssues(service.getOriginalClassName()))
                        addIssuesToBuffer(service.getOriginalClassName(), buffer, currentIndentation);

                    // Stages iteration
                    Iterator<IStage> stages = service.getStages().values().iterator();

                    while (stages.hasNext())
                    {
                        IStage stage = stages.next();
                        indentationsStack.add(currentIndentation);

                        if (stages.hasNext())
                        {
                            buffer.append(
                                    currentIndentation + INDENTATION + "|-- " + stage.getName() + " (" + stage.getID() +
                                    ")\n");
                            currentIndentation += INDENTATION + "|   ";
                        }
                        else
                        {
                            buffer.append(
                                    currentIndentation + INDENTATION + "`-- " + stage.getName() + " (" + stage.getID() +
                                    ")\n");
                            currentIndentation += INDENTATION + "    ";
                        }

                        if (UsageIssuesManagerImpl.getInstance().hasIssues(stage.getOriginalClassName()))
                            addIssuesToBuffer(stage.getOriginalClassName(), buffer, currentIndentation);

                        currentIndentation = indentationsStack.pop();
                    }

                    buffer.append(currentIndentation + "\n");
                    currentIndentation = indentationsStack.pop();
                }

                currentIndentation = indentationsStack.pop();
            }
        }

        buffer.append("-----------------------------------------\n\n");

        if (DIFStartupConfiguration.getLogLevel().equals(LogLevel.DEBUG))
            DIFLogger.getLogger().debug(buffer.toString());
        else
            DIFLogger.getLogger().info(buffer.toString());
    }

    synchronized static public boolean initialize(boolean initDEMInstances, boolean upgradeModelManagers)

    {
        if (!isInitialized())
        {
            if (DIFInitializer.bootTime == null)
                DIFInitializer.bootTime = new Date();

            // Run startup checks to do a quick exist in case of failure
            // In the future we may need to move this further down to the end of the initialization
            // if the tests require the DIF Infrastructure and business entities to be present.
            if (!startupChecks())
                return false;

            // Replica mode
            if (DIFStartupConfiguration.getReplicaMode())
            {
                System.out.println("\n\n");
                System.out.println("*********************************************************************");
                System.out.println("*                                                                   *");
                System.out.println("*                            REPLICA MODE                           *");
                System.out.println("*                            ============                           *");
                System.out.println("*                                                                   *");
                System.out.println("* - Add databases will use dif2.properties database configurations  *");
                System.out.println("* - Preferences from \"" + REPLICA_PREFS + "\" will override default ones     *");
                System.out.println("* - All \"replica.DBInstance.sql\" scripts will be also be executed *");
                System.out.println("* - Changes will be lost on next update                             *");
                System.out.println("*                                                                   *");
                System.out.println("*********************************************************************");
                System.out.println("\n\n");
            }

            try
            {
                AuditContext.setProcessNameForCurrentThread("DIFInitializer");
                AuditContext.setUserForCurrentThread("DIF");

                // IoC initialization (includes module discovery and repository migration if needed)
                IIoCRegistry iocRegistry = DIFIoCRegistry.getRegistry();

                try
                {
                    // Configuration implementation initialization prior to it's first use
                    IConfigurations configAPI = iocRegistry.getImplementation(IConfigurations.class);

                    configAPI.initialize(upgradeModelManagers);

                    // Set the correct config API on the ISSManager
                    ISSManager.setConfigurations(configAPI);

                    // Replica override
                    if (DIFStartupConfiguration.getReplicaMode())
                    {
                        InputStream replicaConfigsFile =
                                Thread.currentThread().getContextClassLoader().getResourceAsStream(REPLICA_PREFS);

                        if (replicaConfigsFile != null)
                        {
                            System.out.println("\n\n");
                            System.out.println(
                                    "********************************************************************************");
                            System.out.println("*");
                            System.out.println(
                                    "* REPLICA_MODE: Forcing configurations from \"" + REPLICA_PREFS + "\"...");

                            Chronometer crono = new Chronometer();
                            crono.start();

                            Properties replicaForcedParameters = new Properties();
                            try
                            {
                                replicaForcedParameters.load(replicaConfigsFile);

                                for (Object key : replicaForcedParameters.keySet())
                                {
                                    String keyAsString = key.toString();
                                    String value = replicaForcedParameters.getProperty(keyAsString);
                                    int index = keyAsString.indexOf('.');
                                    String[] parts =
                                            {keyAsString.substring(0, index), keyAsString.substring(index + 1)};
                                    String path = parts[0];
                                    String keyID = parts[1];

                                    Properties nodeProps = configAPI.readConfiguration(path, null);
                                    nodeProps.setProperty(keyID, value);
                                    configAPI.writeConfiguration(path, null, nodeProps);

                                    System.out.println("*                 - " + keyAsString + " = " + value);
                                }

                                System.out.println("* EXECUTED: Configurations imported in " +
                                                   crono.getTimePassedAsFormattedString());
                            }
                            catch (IOException e)
                            {
                                System.out.println("* ERROR: Reading configurations: " + e.getMessage());
                            }

                            System.out.println("*");
                            System.out.println(
                                    "********************************************************************************");
                            System.out.println("\n\n");
                        }
                    }
                }
                catch (Exception e)
                {
                    throw new InternalFrameworkException(e, null);
                }

                // TODO: Determine when to set bootstrap mode ON
                // DIFStartupConfiguration.setBootstrapMode(true);

                // Startup the Identity and Authorization since they will be called form the CG process of the DEM
                iocRegistry.getImplementation(IIdentityManager.class);
                iocRegistry.getImplementation(IAuthenticationManager.class);

                /*
                 * ATTENTION !!! ......................................................................................
                 * The model upgrade must be before rest of initializations, mainly because: ..........................
                 * - Server Manager uses database to sync to communicate with other servers ...........................
                 * - Code generation may need to invoke identity manager ..............................................
                 * - Application @Init methods usually initialize all models ..........................................
                 * ....................................................................................................
                 */

                if (upgradeModelManagers)
                {
                    // Upgrades all database model, declared to the current application
                    DIFLogger.getLogger().info("Initializing Model Manager upgrades...");
                    DIFLogger.getLogger().increaseIndentation();

                    /* First initialize the DIF Model Manager, for dependencies purposes */
                    IModelManager difModelManager =
                            DIFIoCRegistry.getRegistry().getImplementation(IModelManager.class, "DIF");
                    if (difModelManager != null)
                    {
                        processModelManager(difModelManager);
                    }

                    IMultiThreadExecutor executor = MultiThreadExecuterUtils
                            .newExecutionPool("DIF Model Managers initializer",
                                    DIFStartupConfiguration.getMultiThreadStartup() ? 8 : 1);

                    /* Then initializes the others */
                    for (Entry<String, IModelManager> modelManagerEntry : DIFIoCRegistry.getRegistry()
                            .getImplementationsMap(IModelManager.class).entrySet())
                    {
                        if (!"DIF".equalsIgnoreCase(modelManagerEntry.getKey()))
                        {
                            final IModelManager modelManager = modelManagerEntry.getValue();
                            executor.submitTask(new TaskExecutor()
                            {

                                @Override
                                public void executeLogic() throws Exception
                                {
                                    processModelManager(modelManager);
                                }
                            });
                        }
                    }

                    executor.waitForAllThreadsToFinish(true);

                    DIFLogger.getLogger().decreaseIndentation();
                }

                // initialize the server manager instance
                // Only initializes if the DEM was initialize since the ServerManager needs an AJAX event from a DIF stage to fully operate
                if (initDEMInstances)
                    ServerManager.getInstance();

                if (DIFStartupConfiguration.hasMachineIDNameChangePending())
                {
                    ServerManager.getInstance().applyNameChange();

                    // Configurations initialization has moved the configurations if needed as ServerManager has done.
                    // Nothing more to do

                    DIFLogger.getLogger().info("Apply new machineUID \"" +
                                               DIFStartupConfiguration.getNewMachineIDForConfigurationsToApply() +
                                               "\"...");
                    DIFStartupConfiguration.setMachineIDForConfigurations(
                            DIFStartupConfiguration.getNewMachineIDForConfigurationsToApply());
                    DIFStartupConfiguration.setNewMachineIDForConfigurationsToApply(null);
                    DIFStartupConfiguration.updateConfig();
                }

                // Create the DIF code generator and inject it's dependencies into it
                DIFCodeGenerator codeGen = new DIFCodeGenerator();
                iocRegistry.injectDependencies(codeGen);

                if (initDEMInstances)
                {
                    // Discover all registered DEM entities packages
                    codeGen.collectRegisteredPackages();

                    // Search registered entities
                    codeGen.searchRegisteredPackages();

                    // Enhance the registered entities
                    codeGen.enhanceDEMClasses();
                }

                // Log all issues if in debug
                if (DIFStartupConfiguration.getLogLevel().equals(LogLevel.DEBUG))
                    UsageIssuesManagerImpl.getInstance().logAllIssuesByType();

                // Clean up all temporary data
                codeGen.cleanUp();

                // Initialize the declared APIs
                initializeAPIs();

                if (initDEMInstances)
                {
                    // Run the applications initialization procedures
                    initializeApplications();

                    // Initialize the JavaScript template cache for all stage views
                    TemplateResources.initializeJavaScriptTemplatesForViews();
                }

                // If the identityManager is the static implementation, call the static users initialization
                IIdentityManager identityManager =
                        DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class);
                if (identityManager instanceof IdentityManagerStaticImpl)
                    ((IdentityManagerStaticImpl) identityManager).initializeStaticCustomUsers();

                DIFLoggerInterceptorImpl.reavaluateInterceptorState();

                // Dump DEM structure with associated issues (if any), if DiF is running in DEBUG mode or if there are
                // issues
                if (initDEMInstances && DIFStartupConfiguration.getLogLevel().equals(LogLevel.DEBUG) ||
                    UsageIssuesManagerImpl.getInstance().hasIssues() || DIFStartupConfiguration.getDeveloperMode() ||
                    DIFStartupConfiguration.getTestingMode())
                {
                    dumpDEM();
                }

                // Flight recorder infrastructure
                Thread.setDefaultUncaughtExceptionHandler(new DIFUncaughtExceptionHandler());
                FlightRecorder.startup();

                initialized = true;
            }
            catch (ResourceNotFoundException resourceNotFoundException)
            {
                resourceNotFoundException.printStackTrace();
            }
            catch (CodeGenerationException codeGenerationException)
            {
                codeGenerationException.printStackTrace();
            }
            catch (DIFCodeGenerationException e)
            {
                e.printStackTrace();
            }
            catch (InternalFrameworkException e)
            {
                throw new RuntimeException(e);
            }
            catch (ConfigurationException e)
            {
                throw new RuntimeException(e);
            }
            catch (ServerManagerException e)
            {
                throw new RuntimeException(e);
            }
            catch (InterruptedException e)
            {
                throw new RuntimeException(e);
            }
        }
        return initialized;
    }

    private static boolean startupChecks()
    {
        Chronometer crono = new Chronometer();
        crono.start();
        boolean success = true;

        DIFLogger.getLogger().debug("Starting Startup Checks...");

        StringBuilder tableBuffer = new StringBuilder();
        tableBuffer.append("\n");
        tableBuffer.append("Startup check results:\n");
        tableBuffer.append("==============================\n");
        tableBuffer.append("\n");

        for (IStartupCheck check : DIFIoCRegistry.getRegistry().getImplementations(IStartupCheck.class))
        {
            StartupCheckResult checkResult = check.validate();
            success = success && (!check.isMandatory() || checkResult.isSuccess());
            String status = checkResult.isSuccess() ? " OK " : (check.isMandatory() ? "FAIL" : "WARN");
            String message = checkResult.getErrors().isEmpty() ? "" : "\n    - " + CollectionUtils
                    .listToSeparatedString(checkResult.getErrors(), "\n    - ");

            tableBuffer.append("> [" + status + "] " + StringUtils.camelCaseToString(check.getClass().getSimpleName()) +
                               message + "\n");
            tableBuffer.append("\n");
        }

        crono.end();

        if (success)
        {
            DIFLogger.getLogger().info("\n" + tableBuffer.toString() + ".");
            DIFLogger.getLogger().info("Startup checked validated in " + crono.getTimePassedAsFormattedString());

            return true;
        }
        else
        {
            // Reduce log level to minimum to prevent the following critical information to be berried inside the log file...
            DIFLogger.getLogger().setFatalLogLevel();

            StringBuilder errorAlert = new StringBuilder();
            errorAlert.append("\n\n");
            errorAlert.append("************************************************************************************\n");
            errorAlert.append("  _____ _             _                 ______                     \n");
            errorAlert.append(" / ____| |           | |               |  ____|                    \n");
            errorAlert.append("| (___ | |_ __ _ _ __| |_ _   _ _ __   | |__   _ __ _ __ ___  _ __ \n");
            errorAlert.append(" \\___ \\| __/ _` | '__| __| | | | '_ \\  |  __| | '__| '__/ _ \\| '__|\n");
            errorAlert.append(" ____) | || (_| | |  | |_| |_| | |_) | | |____| |  | | | (_) | |   \n");
            errorAlert.append("|_____/ \\__\\__,_|_|   \\__|\\__,_| .__/  |______|_|  |_|  \\___/|_|   \n");
            errorAlert.append("                               | |\n");
            errorAlert.append("                               |_|\n");

            errorAlert.append(tableBuffer);

            errorAlert.append("************************************************************************************\n");

            // Log as system out to write "no mater what"...
            System.out.println(errorAlert.toString());

            return false;
        }
    }

    private static void initializeAPIs() throws InternalFrameworkException, InterruptedException
    {
        List<IDIFAPI> list = DIFIoCRegistry.getRegistry().getImplementations(IDIFAPI.class);

        Collections.sort(list, new Comparator<IDIFAPI>()
        {

            @Override
            public int compare(IDIFAPI a, IDIFAPI b)
            {
                return a.order().compareTo(b.order());
            }
        });

        IMultiThreadExecutor executor = MultiThreadExecuterUtils
                .newExecutionPool("DIF API Initializer", DIFStartupConfiguration.getMultiThreadStartup() ? 8 : 1);

        for (final IDIFAPI difApi : list)
        {
            DIFLogger.getLogger().info("Initializing API " + difApi.getClass().getCanonicalName());

            executor.submitTask(new TaskExecutor()
            {

                @Override
                public void executeLogic() throws Exception
                {
                    difApi.initialize();
                }
            });
        }

        executor.waitForAllThreadsToFinish(true);
    }

    synchronized static public void initializeApplications() throws InterruptedException
    {
        List<IApplicationConfiguration> appConfImpls =
                DIFIoCRegistry.getRegistry().getImplementations(IApplicationConfiguration.class);
        if (appConfImpls.size() > 0)
        {
            appConfImpls.get(0).processConfigurations();
            DIFLogger.getLogger().info("Initializing Applications Configurations using: " +
                                       appConfImpls.get(0).getClass().getSimpleName());
        }

        IDEMManager demManager = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class);
        DIFLogger.getLogger().info("Initializing Applications...");

        IMultiThreadExecutor executor = MultiThreadExecuterUtils.newExecutionPool("DIF Applications initializer",
                DIFStartupConfiguration.getMultiThreadStartup() ? 8 : 1);

        for (IProvider provider : demManager.getProviders().values())
        {
            for (final IApplication application : provider.getApplications().values())
            {
                executor.submitTask(new TaskExecutor()
                {

                    @Override
                    public void executeLogic() throws Exception
                    {
                        Chronometer crono = new Chronometer();
                        crono.start();

                        AuditContext.setUserForCurrentThread("DIF");
                        AuditContext.setProcessNameForCurrentThread("DIFInitialize:" + application.getName());

                        ((IApplicationPrivate) application).__CG__initialize();

                        crono.end();

                        DIFLogger.getLogger()
                                .log(DIFStartupConfiguration.getDeveloperMode() && crono.getTimePassedInSeconds() >= 1
                                     ? LogLevel.WARN : LogLevel.TRACE,
                                        "      > Aplication " + application.getName() + " (" + application.getID() +
                                        ") init ran in " + crono.getTimePassedAsFormattedString());
                    }
                });
            }
        }

        executor.waitForAllThreadsToFinish(true);
    }

    synchronized static public boolean isInitialized()
    {
        return initialized;
    }

    public static void processModelManager(IModelManager modelManager)
            throws InternalFrameworkException, ConfigurationException
    {
        if (modelManager.isEnabled())
        {
            DIFLogger.getLogger().info("Model '" + modelManager.getSchema() + "': Enabled");
            if (!modelManager.isUpToDate())
            {
                String currentVersion = modelManager.getCurrentVersion();
                DIFLogger.getLogger()
                        .info("Model '" + modelManager.getSchema() + "': Upgrading from " + currentVersion);

                modelManager.updateVersion();
                DIFLogger.getLogger().info("Model '" + modelManager.getSchema() + "': Upgraded to " +
                                           modelManager.getCurrentVersion());
            }
            else
            {
                DIFLogger.getLogger().info("Model '" + modelManager.getSchema() + "': Up to date");
            }
        }
    }
}
