/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.startup;

import pt.digitalis.dif.dem.objects.messages.MessagesLocation;
import pt.digitalis.dif.ioc.DIFDefaultModulesConfiguration;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.dif.utils.system.SystemUtils;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.common.collections.SortedProperties;
import pt.digitalis.utils.config.AbstractConfigurationsImpl;
import pt.digitalis.utils.config.ConfigurationsPreferencesImpl;
import pt.digitalis.utils.config.IConfigurations;
import pt.digitalis.utils.ioc.IoCImplementations;
import pt.digitalis.utils.ioc.ModuleParser;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.URL;
import java.util.Map.Entry;
import java.util.Properties;

// TODO: Auto-generated Javadoc

/**
 * This class represents the basic startup configuration DIF needs. In this phase it can't rely on several
 * features/resources since they have not yet been started Note: There is no support for changes in this configuration
 * in runtime. Only by updating the configuration file and then restarting the DIF will this be reflected. This is a
 * design decision since these parameters define several resources that are mandatory for the startup of the DIF itself
 * and set the ground for all its infrastructure, like, for instance the IoC controller.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Oct 2, 2007
 */
public class DIFStartupConfiguration
{

    /**
     * The file name of the properties file that contains the startup parameters. Note: It's called dif2 for the
     * eventuality of the existence of dif 1.x instances in the same App. Server. this way they will not collide.
     */
    static final public String CONFIGURATION_FILE = "dif2.properties";

    /** The Constant EMPTY_PARAM_SLOT. */
    static public final String EMPTY_PARAM_SLOT = "<none>";

    /** The bootstrap mode. */
    static private Boolean bootstrapMode = false;

    /** The db props. */
    static private Properties dbProps = new Properties();

    /** The DEMO Mode property. */
    static private Boolean demoMode;

    /** The developer Mode property. */
    static private Boolean developerMode;

    /** The IoC engine to use: Property "iocimplementation". */
    static private IoCImplementations iocImplementation;

    /** The Log Level: Property "loglevel". */
    static private LogLevel logLevel;

    /** The machine ID for configurations. */
    static private String machineIDForConfigurations = null;

    /** The Messages Location: Property "messagelocation". */
    static private MessagesLocation messagesLocation;

    /** The Module Parsing method: Property "moduleparser". */
    static private ModuleParser moduleParser;

    /** The modules props. */
    static private Properties modulesProps = new Properties();

    /** The multi thread startup. */
    static private Boolean multiThreadStartup;

    /** The new machine ID for configurations to apply. */
    static private String newMachineIDForConfigurationsToApply = null;

    /** The "print stack trace" property. */
    static private Boolean printStackTrace = true;

    /** The REPLICA Mode property. */
    static private Boolean replicaMode;

    /**
     * The Allow incompatible java to initialize dif.
     */
    static private Boolean allowIncompatibleJavaToInitializeDIF = false;

    /** The machine ID for configurations. */
    static private String replicaModeAlternateMachineIDForConfigurations = EMPTY_PARAM_SLOT;

    /** The testing Mode property. */
    static private Boolean testingMode;

    /** When in developer mode and this is false will not read the messages cache. */
    static private Boolean useMessagesCache = true;

    /** Static initializer */
    static
    {
        Properties parameters = new Properties();
        try
        {
            /*
             * IMPLEMENTATION NOTE: Must use the Thread context class loader since inside Surefire the system
             * classLoader does not have all classpath entries
             */

            // Load the configuration file from the classPath
            InputStream configsFile =
                    Thread.currentThread().getContextClassLoader().getResourceAsStream(CONFIGURATION_FILE);

            if (configsFile == null)
                configsFile = new FileInputStream(CONFIGURATION_FILE);

            if (configsFile != null)
            {
                parameters.load(configsFile);
                boolean savePending = false;

                iocImplementation = IoCImplementations.getImplementationByName(
                        loadProperty(parameters, "iocimplementation", IoCImplementations.GUICE.toString()));

                multiThreadStartup = loadProperty(parameters, "multiThreadStartup", false);
                logLevel = LogLevel.getLevelByName(loadProperty(parameters, "loglevel", LogLevel.WARN.toString()));
                developerMode = loadProperty(parameters, "developerMode", false);
                demoMode = loadProperty(parameters, "demoMode", false);
                testingMode = loadProperty(parameters, "testingMode", false);
                replicaMode = loadProperty(parameters, "replicaMode", false);
                allowIncompatibleJavaToInitializeDIF =
                        loadProperty(parameters, "allowIncompatibleJavaToInitializeDIF", false);
                replicaModeAlternateMachineIDForConfigurations =
                        loadProperty(parameters, "replicaModeAlternateMachineIDForConfigurations", EMPTY_PARAM_SLOT);

                moduleParser = ModuleParser
                        .getParsingMethodByName(loadProperty(parameters, "moduleparser", ModuleParser.FAST.toString()));
                messagesLocation = MessagesLocation.getMessageLocationByName(
                        loadProperty(parameters, "messagelocation", MessagesLocation.MESSAGES_FOLDER.toString()));
                printStackTrace = loadProperty(parameters, "printStackTrace", true);
                useMessagesCache = developerMode;

                machineIDForConfigurations = loadProperty(parameters, "machineIDForConfigurations", null);
                newMachineIDForConfigurationsToApply =
                        loadProperty(parameters, "newMachineIDForConfigurationsToApply", null);

                if (StringUtils.isBlank(machineIDForConfigurations))
                {
                    machineIDForConfigurations = "SERVER|" + SystemUtils.getServerIP();
                    savePending = true;
                }

                for (Entry<Object, Object> prop : parameters.entrySet())
                {
                    if (prop.toString().startsWith(AbstractConfigurationsImpl.generalConfigurationPrefix + ".db."))
                        dbProps.put(prop.getKey().toString()
                                        .substring(4 + AbstractConfigurationsImpl.generalConfigurationPrefix.length()),
                                prop.getValue());
                    else if (prop.toString()
                            .startsWith(AbstractConfigurationsImpl.generalConfigurationPrefix + ".module."))
                        modulesProps.put(prop.getKey().toString()
                                        .substring(8 + AbstractConfigurationsImpl.generalConfigurationPrefix.length()),
                                prop.getValue());
                }

                // No modules defined. Try to get them from the previous place the ConfigurationPreferencesImpl.
                if (modulesProps.isEmpty())
                {
                    DIFLogger.getLogger()
                            .info("Migrating DIF module preferences from PreferencesAPI to " + CONFIGURATION_FILE +
                                  "...");

                    IConfigurations oldConfigurationImpl = new ConfigurationsPreferencesImpl();
                    modulesProps = oldConfigurationImpl
                            .readConfiguration(DIFDefaultModulesConfiguration.DEFAULT_MODULE_CONFIG_ID,
                                    DIFDefaultModulesConfiguration.DEFAULT_MODULE_CONFIG_SECTION_ID);

                    DIFLogger.getLogger().info("Removing deprecated DIF module preferences from PreferencesAPI...");
                    // Purge old module configurations so that we won't use it anymore and create confusion
                    oldConfigurationImpl.removeConfiguration(DIFDefaultModulesConfiguration.DEFAULT_MODULE_CONFIG_ID,
                            DIFDefaultModulesConfiguration.DEFAULT_MODULE_CONFIG_SECTION_ID);

                    savePending = true;
                }

                // Save the configuration
                if (savePending)
                    updateConfig();
            }
            else
                setDefaultConfigurations();
        }
        catch (Exception e)
        {
            setDefaultConfigurations();
            DIFLogger.getLogger().warn("dif2.properties unavailable! Reverting to defaults...");
        }
    }

    /**
     * Gets allow incompatible java to initialize dif.
     *
     * @return the allow incompatible java to initialize dif
     */
    public static Boolean getAllowIncompatibleJavaToInitializeDIF()
    {
        return allowIncompatibleJavaToInitializeDIF;
    }

    /**
     * Sets allow incompatible java to initialize dif.
     *
     * @param allowIncompatibleJavaToInitializeDIF the allow incompatible java to initialize dif
     */
    public static void setAllowIncompatibleJavaToInitializeDIF(Boolean allowIncompatibleJavaToInitializeDIF)
    {
        DIFStartupConfiguration.allowIncompatibleJavaToInitializeDIF = allowIncompatibleJavaToInitializeDIF;
    }

    /**
     * Gets the bootstrap mode.
     *
     * @return the bootstrapMode
     */
    public static Boolean getBootstrapMode()
    {
        return bootstrapMode;
    }

    /**
     * Sets the bootstrap mode.
     *
     * @param bootstrapMode the bootstrapMode to set
     */
    public static void setBootstrapMode(Boolean bootstrapMode)
    {
        DIFStartupConfiguration.bootstrapMode = bootstrapMode;
    }

    /**
     * Gets the db props.
     *
     * @return the dbProps
     */
    public static Properties getDbProps()
    {
        return dbProps;
    }

    /**
     * Sets the db props.
     *
     * @param dbProps the dbProps to set
     */
    public static void setDbProps(Properties dbProps)
    {
        DIFStartupConfiguration.dbProps = dbProps;
    }

    /**
     * Inspector for the 'demoMode' attribute.
     *
     * @return the demoMode value
     */
    public static Boolean getDemoMode()
    {
        return demoMode;
    }

    /**
     * Modifier for the 'demoMode' attribute.
     *
     * @param demoMode the new demoMode value to set
     */
    public static void setDemoMode(Boolean demoMode)
    {
        DIFStartupConfiguration.demoMode = demoMode;
    }

    /**
     * Inspector for the 'developerMode' attribute.
     *
     * @return the developerMode value
     */
    public static Boolean getDeveloperMode()
    {
        return developerMode;
    }

    /**
     * Modifier for the 'developerMode' attribute.
     *
     * @param developerMode the new developerMode value to set
     */
    public static void setDeveloperMode(Boolean developerMode)
    {
        DIFStartupConfiguration.developerMode = developerMode;
    }

    /**
     * Gets the file in config path.
     *
     * @param desiredFile the desired file
     *
     * @return the file in config path
     */
    public static String getFileInConfigPath(String desiredFile)
    {
        String jbossHomeDir = System.getProperty("jboss.server.home.dir");
        String filePath;

        if (StringUtils.isBlank(jbossHomeDir))
        {
            // Not JBoss environment...
            // Get the file wherever it is...
            URL configsFileURL = Thread.currentThread().getContextClassLoader().getResource(desiredFile);
            filePath = configsFileURL.getFile();
        }
        else
        {
            filePath = jbossHomeDir.concat("/conf/").concat(desiredFile);
        }

        return filePath;
    }

    /**
     * Gets the io C implementation.
     *
     * @return the iocImplementation
     */
    static public IoCImplementations getIoCImplementation()
    {
        return iocImplementation;
    }

    /**
     * Gets the log level.
     *
     * @return the logLevel
     */
    static public LogLevel getLogLevel()
    {
        return logLevel;
    }

    /**
     * Modifier for the 'logLevel' attribute.
     *
     * @param logLevel the new logLevel value to set
     */
    public static void setLogLevel(LogLevel logLevel)
    {
        DIFStartupConfiguration.logLevel = logLevel;
    }

    /**
     * Gets the machine ID for configurations.
     *
     * @return the machineIDForConfigurations
     */
    public static String getMachineIDForConfigurations()
    {
        return machineIDForConfigurations;
    }

    /**
     * Sets the machine ID for configurations.
     *
     * @param machineIDForConfigurations the machineIDForConfigurations to set
     */
    public static void setMachineIDForConfigurations(String machineIDForConfigurations)
    {
        DIFStartupConfiguration.machineIDForConfigurations = machineIDForConfigurations;
    }

    /**
     * Gets the messages location.
     *
     * @return the messagesLocation
     */
    static public MessagesLocation getMessagesLocation()
    {
        return messagesLocation;
    }

    /**
     * Gets the module parser.
     *
     * @return the moduleParser
     */
    static public ModuleParser getModuleParser()
    {
        return moduleParser;
    }

    /**
     * Gets the modules props.
     *
     * @return the modulesProps
     */
    public static Properties getModulesProps()
    {
        return modulesProps;
    }

    /**
     * Sets the modules props.
     *
     * @param modulesProps the modulesProps to set
     */
    public static void setModulesProps(Properties modulesProps)
    {
        DIFStartupConfiguration.modulesProps = modulesProps;
    }

    /**
     * Gets multi thread startup.
     *
     * @return the multiThreadStartup
     */
    public static Boolean getMultiThreadStartup()
    {
        return multiThreadStartup && developerMode;
    }

    /**
     * Sets multi thread startup.
     *
     * @param multiThreadStartup the multiThreadStartup to set
     */
    public static void setMultiThreadStartup(Boolean multiThreadStartup)
    {
        DIFStartupConfiguration.multiThreadStartup = multiThreadStartup;
    }

    /**
     * Gets the new machine ID for configurations to apply.
     *
     * @return the newMachineIDForConfigurationsToApply
     */
    public static String getNewMachineIDForConfigurationsToApply()
    {
        return newMachineIDForConfigurationsToApply;
    }

    /**
     * Sets the new machine ID for configurations to apply.
     *
     * @param newMachineIDForConfigurationsToApply the newMachineIDForConfigurationsToApply to set
     */
    public static void setNewMachineIDForConfigurationsToApply(String newMachineIDForConfigurationsToApply)
    {
        DIFStartupConfiguration.newMachineIDForConfigurationsToApply = newMachineIDForConfigurationsToApply;
    }

    /**
     * Gets the replica mode.
     *
     * @return the replicaMode
     */
    public static Boolean getReplicaMode()
    {
        return replicaMode;
    }

    /**
     * Sets the replica mode.
     *
     * @param replicaMode the replicaMode to set
     */
    public static void setReplicaMode(Boolean replicaMode)
    {
        DIFStartupConfiguration.replicaMode = replicaMode;
    }

    /**
     * Gets the replica mode alternate machine ID for configurations.
     *
     * @return the replicaModeAlternateMachineIDForConfigurations
     */
    public static String getReplicaModeAlternateMachineIDForConfigurations()
    {
        return replicaModeAlternateMachineIDForConfigurations;
    }

    /**
     * Sets the replica mode alternate machine ID for configurations.
     *
     * @param replicaModeAlternateMachineIDForConfigurations the replicaModeAlternateMachineIDForConfigurations to set
     */
    public static void setReplicaModeAlternateMachineIDForConfigurations(
            String replicaModeAlternateMachineIDForConfigurations)
    {
        DIFStartupConfiguration.replicaModeAlternateMachineIDForConfigurations =
                replicaModeAlternateMachineIDForConfigurations;
    }

    /**
     * Inspector for the 'testingMode' attribute.
     *
     * @return the testingMode value
     */
    public static Boolean getTestingMode()
    {
        return testingMode;
    }

    /**
     * Modifier for the 'testingMode' attribute.
     *
     * @param testingMode the new testingMode value to set
     */
    public static void setTestingMode(Boolean testingMode)
    {
        DIFStartupConfiguration.testingMode = testingMode;
    }

    /**
     * Inspector for the 'useMessagesCache' attribute.
     *
     * @return the useMessagesCache value
     */
    public static Boolean getUseMessagesCache()
    {
        return useMessagesCache;
    }

    /**
     * Modifier for the 'useMessagesCache' attribute.
     *
     * @param useMessagesCache the new useMessagesCache value to set
     */
    public static void setUseMessagesCache(Boolean useMessagesCache)
    {
        DIFStartupConfiguration.useMessagesCache = useMessagesCache;
    }

    /**
     * Checks for machine ID name change pending.
     *
     * @return true, if successful
     */
    public static boolean hasMachineIDNameChangePending()
    {
        return StringUtils.isNotBlank(newMachineIDForConfigurationsToApply);
    }

    /**
     * Inspector for the 'printStackTrace' attribute.
     *
     * @return the printStackTrace attribute value
     */
    static public Boolean isPrintStackTrace()
    {
        return printStackTrace;
    }

    /**
     * Checks if is production mode.
     *
     * @return T is the instance is in production mode. Production mode is the default mode when none of the other modes
     *         are active (development, testing or demo)
     */
    public static boolean isProductionMode()
    {
        return !getDemoMode() && !getDeveloperMode() && !getTestingMode();
    }

    /**
     * Overloads the 'loadProperty' method for boolean values.
     *
     * @param prop         the Properties to search
     * @param key          the key to search
     * @param defaultValue the default value when not found
     *
     * @return the read value or the default if not found
     */
    static Boolean loadProperty(Properties prop, String key, boolean defaultValue)
    {
        String value = prop.getProperty(key);

        if (value == null)
            return defaultValue;
        else
        {
            if (value.toLowerCase().equals("false") || value.toLowerCase().equals("f"))
                return false;
            else if (value.toLowerCase().equals("true") || value.toLowerCase().equals("t"))
                return true;
            else
                throw new RuntimeException("Invalid configuration value for property '" + key +
                                           "'! Unable to proceed...\nValid values are: 'true', 'T', 'false' and 'F'.");
        }
    }

    /**
     * Searches properties for a given key.
     *
     * @param prop         the Properties to search
     * @param key          the key to search
     * @param defaultValue the default value when not found
     *
     * @return the read value or the default if not found
     */
    static String loadProperty(Properties prop, String key, String defaultValue)
    {
        String value = prop.getProperty(key);

        if (value == null)
            return defaultValue;
        else
            return value;
    }

    /**
     * Sets the default configurations.
     */
    static private void setDefaultConfigurations()
    {
        iocImplementation = IoCImplementations.GUICE;
        logLevel = LogLevel.WARN;
        developerMode = false;
        demoMode = false;
        testingMode = false;
        moduleParser = ModuleParser.FAST;
        messagesLocation = MessagesLocation.MESSAGES_FOLDER;
        printStackTrace = true;
        replicaMode = false;
    }

    /**
     * Update config to dif2.properties file
     *
     * @return true, if successful
     */
    public static boolean updateConfig()
    {
        // Build the properties object to save to file
        Properties props = new SortedProperties();

        // Load current file so that we don't override any other parameters
        InputStream configsFile =
                Thread.currentThread().getContextClassLoader().getResourceAsStream(CONFIGURATION_FILE);
        try
        {
            if (configsFile != null)
                props.load(configsFile);

            props.setProperty("iocimplementation", iocImplementation.toString());
            props.setProperty("loglevel", logLevel.toString());
            props.setProperty("developerMode", developerMode.toString());
            props.setProperty("demoMode", demoMode.toString());
            props.setProperty("testingMode", testingMode.toString());
            props.setProperty("replicaMode", replicaMode.toString());
            props.setProperty("multiThreadStartup", multiThreadStartup.toString());
            props.setProperty("allowIncompatibleJavaToInitializeDIF", allowIncompatibleJavaToInitializeDIF.toString());

            props.setProperty("moduleParser", moduleParser.toString());
            props.setProperty("messagesLocation", messagesLocation.toString());
            props.setProperty("printStackTrace", printStackTrace.toString());
            props.setProperty("machineIDForConfigurations", machineIDForConfigurations.toString());
            props.setProperty("replicaModeAlternateMachineIDForConfigurations",
                    StringUtils.nvl(replicaModeAlternateMachineIDForConfigurations, EMPTY_PARAM_SLOT));

            if (newMachineIDForConfigurationsToApply == null)
                props.remove("newMachineIDForConfigurationsToApply");
            else
                props.setProperty("newMachineIDForConfigurationsToApply", newMachineIDForConfigurationsToApply);

            for (Entry<Object, Object> prop : dbProps.entrySet())
            {
                props.setProperty(
                        AbstractConfigurationsImpl.generalConfigurationPrefix + ".db." + prop.getKey().toString(),
                        prop.getValue().toString());
            }

            for (Entry<Object, Object> prop : modulesProps.entrySet())
            {
                props.setProperty(
                        AbstractConfigurationsImpl.generalConfigurationPrefix + ".module." + prop.getKey().toString(),
                        prop.getValue().toString());
            }

            // Save to file in the JBOSS Conf directory
            String difPropertiesFilePath = getFileInConfigPath(CONFIGURATION_FILE);

            DIFLogger.getLogger().info("Saving updated preferences in " + CONFIGURATION_FILE + "...");
            FileOutputStream outStream = new FileOutputStream(difPropertiesFilePath);
            props.store(outStream, "DIF2 Startup properties");

            return true;
        }
        catch (Exception e)
        {
            e.printStackTrace();
            return false;
        }
    }
}
