/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.utils;

import org.json.JSONObject;
import pt.digitalis.utils.common.BeanInspector;
import pt.digitalis.utils.common.StringUtils;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Helps to format an object for printing in human readable form.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Dec 7, 2007
 */
public class ObjectFormatter
{

    /** The prefix to format the lines with. */
    final static private String PREFIX = "  | ";

    /** The prefix to format the lines with. */
    final static private String SEPARATOR = ": ";

    /** The generated buffer to return. Will be built incrementally */
    private StringBuffer buffer = new StringBuffer();

    /** the dumpd object list to keep track (avoid circular dump and stack overflow). */
    private List<Object> dumpedObjects = new ArrayList<Object>();

    /** The format. */
    private Format format;

    /** The generated buffer to return. Will be built incrementally */
    private JSONObject jsonBuffer = new JSONObject();

    /**
     * Instantiates a new Object formatter.
     *
     * @param format        the format
     * @param dumpedObjects the already dumped objects for inner objects to be rendered inside others
     */
    public ObjectFormatter(Format format, List<Object> dumpedObjects)
    {
        super();
        this.format = format;
        if (dumpedObjects != null)
            this.dumpedObjects = dumpedObjects;
    }

    /**
     * Adds an item to the buffer.
     *
     * @param key   the key that identifies the value
     * @param value the value
     *
     * @return the updated instance (fluent interface)
     */
    public ObjectFormatter addItem(String key, Object value)
    {
        if (!dumpedObjects.contains(value) ||
            (value != null && value.getClass().getPackage().getName().startsWith("java.lang")))
        {
            if (key.contains("pass"))
                value = "*****";

            if (format == Format.TEXT)
            {
                String line = PREFIX + key + SEPARATOR;
                line += toStringParser(value, line.length());

                if (buffer.length() > 0)
                    buffer.append("\n");

                buffer.append(line);
            }
            else
            {
                if (value == null || value.getClass().isEnum() ||
                    value.getClass().getPackage().getName().startsWith("java.lang"))
                {
                    jsonBuffer.put(key, StringUtils.toStringOrNull(value));
                }
                else
                {
                    Map<String, Object> objectValues = new HashMap<String, Object>();

                    try
                    {
                        for (Map.Entry<String, Object> valueEntry : BeanInspector.getObjectFieldsAsMap(value)
                                .entrySet())
                        {
                            if (!dumpedObjects.contains(valueEntry.getValue()))
                                objectValues.put(valueEntry.getKey(), valueEntry.getValue());
                        }
                    }
                    catch (IllegalAccessException e)
                    {
                        e.printStackTrace();
                    }
                    catch (InvocationTargetException e)
                    {
                        e.printStackTrace();
                    }
                    catch (NoSuchMethodException e)
                    {
                        e.printStackTrace();
                    }

                    jsonBuffer.put(key, objectValues);
                }
            }
        }

        return this;
    }

    /**
     * Adds an item to the buffer.
     *
     * @param key   the key that identifies the value
     * @param value the value
     *
     * @return the updated instance (fluent interface)
     */
    public ObjectFormatter addItem(String key, IObjectFormatter value)
    {
        if (!dumpedObjects.contains(value))
        {
            if (format == Format.TEXT)
                return addItem(key, (Object) value);
            else
            {
                Object valueToAdd = null;

                if (value != null)
                {
                    ObjectFormatter innerObjectFormater = value.toObjectFormatter(Format.JSON, dumpedObjects);
                    valueToAdd = innerObjectFormater.jsonBuffer;
                    dumpedObjects = innerObjectFormater.dumpedObjects;
                }

                jsonBuffer.put(key, valueToAdd);
            }
        }

        return this;
    }

    /**
     * Adds an item to the buffer if the value is not null.
     *
     * @param key   the key that identifies the value
     * @param value the value
     *
     * @return the updated instance (fluent interface)
     */
    public ObjectFormatter addItemIfNotNull(String key, Object value)
    {
        if (value != null)
            addItem(key, value);

        return this;
    }

    /**
     * Formats an iterator in human readable form.
     *
     * @param obj the list to convert
     *
     * @return the formatted list
     */
    private String formatIterator(Iterable<?> obj)
    {
        if (!dumpedObjects.contains(obj))
        {
            ArrayList objectToDump = new ArrayList();

            synchronized (obj)
            {
                Iterator<?> iterator = ((Iterable<?>) obj).iterator();
                while (iterator.hasNext())
                {
                    objectToDump.add(iterator.next());
                }
            }

            dumpedObjects.add(obj);

            if (objectToDump.isEmpty())
                return "[]";
            else
            {
                String temp = "[\n";
                int pos = 1;

                for (Object object : objectToDump)
                {
                    String ident = " #" + StringUtils.fillStringLeft(Integer.toString(pos++), 2, " ") + " ";
                    temp += ident + toStringParserWithPrefix(object, ident.length(), " ") + "\n";
                }

                return temp + "]";
            }
        }
        else
            return "see above";
    }

    /**
     * Formats a map in human readable form.
     *
     * @param map the map to convert
     *
     * @return the formatted map
     */
    private String formatMap(Map<?, ?> map)
    {
        if (!dumpedObjects.contains(map))
        {
            dumpedObjects.add(map);

            if (map.size() == 0)
                return "[]";
            else
            {
                String temp = "[\n";
                int pos = 1;

                // Freze keys to avoid java.util.ConcurrentModificationException
                Set keys = new LinkedHashSet(map.keySet());

                for (Object key : keys)
                {
                    String ident =
                            " " + StringUtils.fillStringLeft("#" + Integer.toString(pos++), 3, " ") + " " + key + " = ";
                    Object value;

                    if (key != null && key.toString().contains("pass"))
                        value = "*****";
                    else
                        value = map.get(key);

                    if (value != null)
                    {
                        if (value instanceof String)
                            value = "\"" + value + "\"";

                        temp += ident + toStringParserWithPrefix(value, ident.length(), " ") + "\n";
                    }
                }

                return temp + "]";
            }
        }
        else
            return "see above";
    }

    /**
     * Gets the formated object.
     *
     * @return the formatted object
     */
    public String getFormatedObject()
    {
        if (format == Format.TEXT)
            return buffer.toString();
        else
            return jsonBuffer.toString();
    }

    /**
     * Inspector for the 'jsonBuffer' attribute.
     *
     * @return the jsonBuffer value
     */
    public JSONObject getJsonObject()
    {
        if (format == Format.TEXT)
            throw new UnsupportedOperationException(
                    "The format specified for this formatter is text. No JSON content available.");
        else
            return jsonBuffer;
    }

    /**
     * Parses a string and tries to format it for including has a value in a line.
     *
     * @param obj        the object to format
     * @param prefixSize the size of the necessary prefix
     *
     * @return the formatted string
     */
    private String toStringParser(Object obj, int prefixSize)
    {
        return toStringParserWithPrefix(obj, prefixSize, PREFIX);
    }

    /**
     * Parses a string and tries to format it for including has a value in a line.
     *
     * @param obj        the object to format
     * @param prefixSize the size of the necessary prefix
     * @param prefix     the prefix to use to pad the string
     *
     * @return the formatted string
     */
    private String toStringParserWithPrefix(Object obj, int prefixSize, String prefix)
    {
        if (obj == null)
            return null;
        else
        {
            String temp;

            if (obj instanceof Map)
                temp = formatMap((Map<?, ?>) obj);

            else if (obj instanceof Iterable)
                temp = formatIterator(((Iterable<?>) obj));

            else
                temp = obj.toString();

            dumpedObjects.add(obj);

            return temp.replace("\n", "\n" + StringUtils.fillString(prefix, prefixSize, " "));
        }
    }

    /**
     * The Enum Format.
     */
    public enum Format
    {

        /** The json. */
        JSON,

        /** The text. */
        TEXT
    }
}
