package pt.digitalis.dif.utils.development;

import org.aspectj.lang.annotation.SuppressAjWarnings;

import pt.digitalis.dif.utils.logging.DIFLogAspect;
import pt.digitalis.log.LogLevel;

/**
 * Intercepts printStackTrace calls from exceptions thrown from unit tests.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created 2007/11/21
 */
privileged public aspect ExceptionStackTraceInterceptionAspect extends DIFLogAspect {

    /**
     * Captures calls to <code>printStackTrace(..)</code> within DIF's logging appliances. Exposes the thrown
     * exception.
     * 
     * @param exception
     *            the thrown exception
     */
    private pointcut printStackTrace(Throwable exception): call(* *.printStackTrace(..)) &&
    	within(pt.digitalis.dif.utils.logging.*) && target(exception) && if(false);

    /**
     * Advises DIFExceptionLogAspect to inhibit the printStackTrace to run for exceptions throws as part of unit tests.
     * 
     * @param exception
     *            the thrown exception
     */
    @SuppressAjWarnings({"adviceDidNotMatch"})
    void around(Throwable exception): printStackTrace(exception) 
    {
        if (wasThrownFromMockupOrTest(exception))
            super.getLogger().log(LogLevel.WARN, "[DEV] Exception stack trace omitted!");
        else
            proceed(exception);
    }

    /**
     * Returns T if the exception was thrown from a Mockup object or a test class. Checks if it was called from a
     * junit/surefire class also.
     * 
     * Just searches the stack top and bottom element to avoid performance penalty.
     * 
     * @param exception
     *            the thrown exception
     * @return T if exception was thrown from a Mockup object, F otherwise.
     */
    static private boolean wasThrownFromMockupOrTest(Throwable exception) {
        StackTraceElement[] stack = exception.getStackTrace();

        for (StackTraceElement element : stack) {
            String className = element.getClassName().toLowerCase();

            if (className.contains("mockup") || className.contains("test") || className.contains("surefire")
                    || className.contains("junit"))
                return true;
        }

        return false;
    }
}
