/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.extensions.document;

import pt.digitalis.dif.controller.interfaces.IDIFSession;

import java.util.List;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Jun 5, 2009
 */
public interface IDocumentRepositoryManager
{

    /**
     * Adds a new document to the repository
     *
     * @param document the document to add
     *
     * @return the added document
     *
     * @exception DocumentRepositoryException
     */
    public DocumentRepositoryEntry addDocument(DocumentRepositoryEntry document) throws DocumentRepositoryException;

    /**
     * Adds a new document to the repository
     *
     * @param document        the document to add
     * @param ignoreSizeLimit ignores the size limit
     *
     * @return the added document
     *
     * @exception DocumentRepositoryException
     */
    public DocumentRepositoryEntry addDocument(DocumentRepositoryEntry document, Boolean ignoreSizeLimit)
            throws DocumentRepositoryException;

    /**
     * Declares the requested document to be declared as authorized access for the current session scope. If not called
     * an access to this document will result in an Authorization Exception
     *
     * @param session    the current session
     * @param documentID the document ID to authorize
     */
    public void authorizeDocumentForCurrentSession(IDIFSession session, Long documentID);

    /**
     * Deletes an existing document from the repository
     *
     * @param id the document to delete
     *
     * @exception DocumentRepositoryException
     */
    public void deleteDocument(Long id) throws DocumentRepositoryException;

    /**
     * Searches an existing document by it's unique identifier
     *
     * @param id the document id
     *
     * @return the document
     *
     * @exception DocumentRepositoryException
     */
    public DocumentRepositoryEntry getDocument(Long id) throws DocumentRepositoryException;

    /**
     * Searches an existing document by it's unique identifier.
     *
     * @param id           the document id
     * @param lazyLoadFile the lazy load file bytes. This is mainly for the DB implementation and allows, if true, the
     *                     document
     *                     data to be read but the file bytes aren't. This fact alone improves the performance greatly
     *                     in
     *                     scenarios that the file content is not required. To use this feature a valid/open transaction
     *                     must
     *                     exists between the getDocument and getBytes invocation, otherwise a exception is throwed.
     *
     * @return the document
     *
     * @exception DocumentRepositoryException the document repository exception
     */
    public DocumentRepositoryEntry getDocument(Long id, boolean lazyLoadFile) throws DocumentRepositoryException;

    /**
     * Searches for a document by it's original file name
     *
     * @param originalFileName the file name
     *
     * @return the document
     *
     * @exception DocumentRepositoryException
     */
    public List<DocumentRepositoryEntry> getDocumentByOriginalFileName(String originalFileName)
            throws DocumentRepositoryException;

    /**
     * Searches an existing document by it's creator
     *
     * @param creatorUserID the user id that created the document
     *
     * @return the document
     *
     * @exception DocumentRepositoryException
     */
    public List<DocumentRepositoryEntry> getDocumentsByCreator(String creatorUserID) throws DocumentRepositoryException;

    /**
     * Get {@link DocumentRepositoryEntry} by ids
     *
     * @param documentIds the array of document ids
     *
     * @return List of {@link DocumentRepositoryEntry}
     *
     * @exception DocumentRepositoryException
     */
    public List<DocumentRepositoryEntry> getDocumentsByIds(List<Long> documentIds) throws DocumentRepositoryException;

    /**
     * Searches the authorization list in session for the document ID
     *
     * @param session    the current session
     * @param documentID the document ID to check for authorization
     *
     * @return T if the document is authorized
     */
    public boolean isDocumentAuthorizedInCurrentSession(IDIFSession session, Long documentID);

    /**
     * Updates an existing document with the new data
     *
     * @param document the document to update in the respository
     *
     * @return the updated document
     */
    public DocumentRepositoryEntry updateDocument(DocumentRepositoryEntry document);
}
