/**
 * 2015, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.jobs;

import org.apache.commons.collections4.queue.CircularFifoQueue;
import pt.digitalis.dif.controller.interfaces.IControllerCleanupTask;
import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.controller.objects.ControllerExecutionStep;
import pt.digitalis.dif.exception.BusinessException;
import pt.digitalis.dif.exception.controller.ControllerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFInitializer;
import pt.digitalis.dif.utils.logging.AuditContext;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.dif.utils.logging.IErrorLogManager;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.common.Chronometer;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.common.TimeUtils;
import pt.digitalis.utils.common.TimeUtils.Scale;
import pt.digitalis.utils.config.ConfigurationException;
import pt.digitalis.utils.config.IConfigurations;

import javax.management.RuntimeErrorException;
import java.util.Calendar;
import java.util.Date;
import java.util.Properties;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 24/11/2015
 */
public abstract class DIFJob extends Thread
{

    /**
     * If T the user will have access to encritped data (will force to true the has access)
     */
    protected boolean forceAccessToEncriptedData;

    /**
     *
     */
    protected boolean isActive = true;

    /**
     *
     */
    protected Date lastRunDate = null;

    /**
     * If T the user must have access to encripted data access to be allowed to execute the Job
     */
    protected boolean mustHaveEncriptedDataAccess;

    /**
     *
     */
    protected Long runIntervalInSeconds = null;

    /**
     *
     */
    protected TimeOfDay runTimeOfDay = null;

    /**
     *
     */
    protected Long totalExecutions = 0L;

    /**
     *
     */
    protected String userName = null;

    /**
     *
     */
    private CircularFifoQueue<JobExecution> executionLog = new CircularFifoQueue<JobExecution>(100);

    /**
     *
     */
    private Long jobID;

    /**
     *
     */
    private Worker runningWorker = null;

    /**
     * @param session                     the current dif session. Null if none available (in this case any DB
     *                                    operations will be audited like
     *                                    DIF)
     * @param mustHavaEncriptedDataAccess if the current job can only be executed by users that have access to encrypted
     *                                    personal data
     */
    public DIFJob(IDIFSession session, boolean mustHavaEncriptedDataAccess)
    {
        if (session == null || !session.isLogged())
            userName = "DIF";
        else
            userName = session.getUser().getID();

        this.mustHaveEncriptedDataAccess = mustHavaEncriptedDataAccess;

        // TODO: netPA: Viegas: RGPD - 1. Validate if the user has access to encrypted data
    }

    /**
     * Clean up tasks to execute after each run of the query
     */
    protected void cleanUpAfterRun()
    {
        // Perform cleanup for current thread since it will run in pair with DIF controller cycle as another thread
        ControllerException controllerException = null;
        IControllerCleanupTask taskWithException = null;

        for (IControllerCleanupTask task : DIFIoCRegistry.getRegistry()
                .getImplementations(IControllerCleanupTask.class))
        {
            try
            {
                task.doTask(null, true);
            }
            catch (Exception e)
            {
                if (controllerException == null)
                {
                    controllerException = new ControllerException(ControllerExecutionStep.DISPATCHER_CONCLUDE, e);
                    taskWithException = task;
                }
            }
        }

        if (controllerException != null)
            new BusinessException("Error while cleaning up Job after run" + (taskWithException == null ? ""
                                                                                                       : " on task " +
                                                                                                         taskWithException
                                                                                                                 .getClass()
                                                                                                                 .getSimpleName()),
                    controllerException).addToExceptionContext("Task", taskWithException).log(LogLevel.WARN);
    }

    /**
     * @return T to state that the run was successful
     *
     * @exception Exception
     */
    protected abstract boolean executeEachTime() throws Exception;

    /**
     * @return should provide the default run interval for when no custom configuration exists
     *
     * @exception ConfigurationException
     */
    protected abstract Long getDefaultRunIntervalInSeconds() throws ConfigurationException;

    /**
     * Inspector for the 'executionLog' attribute.
     *
     * @return the executionLog value
     */
    public CircularFifoQueue<JobExecution> getExecutionLog()
    {
        return executionLog;
    }

    /**
     * Inspector for the 'jobID' attribute.
     *
     * @return the jobID value
     */
    public Long getJobID()
    {
        return this.jobID;
    }

    /**
     * Modifier for the 'jobID' attribute.
     *
     * @param jobID the new jobID value to set
     */
    public void setJobID(Long jobID)
    {
        this.jobID = jobID;
    }

    /**
     * Override in each job to provide custom title
     *
     * @return the Job title
     */
    public String getJobName()
    {
        String result = this.getClass().getSimpleName();
        if (StringUtils.isBlank(result))
        {
            result = this.getClass().getName();
        }
        return result;
    }

    /**
     * Inspector for the 'jobType' attribute.
     *
     * @return the jobType value
     */
    public abstract JobType getJobType();

    /**
     * Inspector for the 'lastRunDate' attribute.
     *
     * @return the lastRunDate value
     */
    public Date getLastRunDate()
    {
        return lastRunDate;
    }

    /**
     * Modifier for the 'lastRunDate' attribute.
     *
     * @param lastRunDate the new lastRunDate value to set
     */
    public void setLastRunDate(Date lastRunDate)
    {
        this.lastRunDate = lastRunDate;
    }

    /**
     * Inspector for the 'runIntervalInSeconds' attribute.<br/>
     * run every 5 minutes (default)
     *
     * @return the runIntervalInSeconds value
     *
     * @exception ConfigurationException
     */
    public Long getRunIntervalInSeconds() throws ConfigurationException
    {
        if (runIntervalInSeconds == null)
        {
            if (getDefaultRunIntervalInSeconds() == null)
                return 300L;
            else
                return getDefaultRunIntervalInSeconds();
        }

        return runIntervalInSeconds;
    }

    /**
     * Modifier for the 'runIntervalInSeconds' attribute.
     *
     * @param runIntervalInSeconds the new runIntervalInSeconds value to set
     */
    public void setRunIntervalInSeconds(Long runIntervalInSeconds)
    {
        this.runIntervalInSeconds = runIntervalInSeconds;
    }

    /**
     * Inspector for the 'runTimeOfDay' attribute.
     *
     * @return the runTimeOfDay value
     *
     * @exception ConfigurationException
     */
    public TimeOfDay getRunTimeOfDay() throws ConfigurationException
    {
        return runTimeOfDay;
    }

    /**
     * Modifier for the 'runTimeOfDay' attribute.
     *
     * @param runTimeOfDay the new runTimeOfDay value to set (in the format of (HH:MM))
     */
    public void setRunTimeOfDay(String runTimeOfDay)
    {
        this.runTimeOfDay = new TimeOfDay(runTimeOfDay);
    }

    /**
     * Modifier for the 'runTimeOfDay' attribute.
     *
     * @param runTimeOfDay the new runTimeOfDay value to set
     */
    public void setRunTimeOfDay(TimeOfDay runTimeOfDay)
    {
        this.runTimeOfDay = runTimeOfDay;
    }

    /**
     * Inspector for the 'totalExecutions' attribute.
     *
     * @return the totalExecutions value
     */
    public Long getTotalExecutions()
    {
        return totalExecutions;
    }

    /**
     * Inspector for the 'isActive' attribute.
     *
     * @return the isActive value
     */
    public boolean isActive()
    {
        return isActive;
    }

    /**
     * Modifier for the 'isActive' attribute.
     *
     * @param isActive the new isActive value to set
     */
    public void setActive(boolean isActive)
    {
        this.isActive = isActive;
    }

    /**
     * @return the forceAccessToEncriptedData
     */
    public boolean isForceAccessToEncriptedData()
    {
        return forceAccessToEncriptedData;
    }

    /**
     * @param forceAccessToEncriptedData the forceAccessToEncriptedData to set
     */
    public void setForceAccessToEncriptedData(boolean forceAccessToEncriptedData)
    {
        this.forceAccessToEncriptedData = forceAccessToEncriptedData;
    }

    /**
     * Read configuration from persistence
     *
     * @exception ConfigurationException
     */
    public void readConfig() throws ConfigurationException
    {
        IConfigurations configManager = DIFIoCRegistry.getRegistry().getImplementation(IConfigurations.class);
        Properties props = configManager.readConfiguration("DIF2", "Jobs/" + this.getJobName());

        if (!props.isEmpty())
        {
            DIFLogger.getLogger()
                    .info("Restoring Job \"" + this.getJobName() + "\" from saved settings...\n" + props.toString());

            String activeProp = this.readProperty(props, "active");
            String runIntervalInSecondsProp = this.readProperty(props, "runIntervalInSeconds");
            String runTimeOfDayProp = this.readProperty(props, "runTimeOfDay");

            if (activeProp != null)
                this.setActive(Boolean.parseBoolean(activeProp));

            if (runIntervalInSecondsProp != null)
                this.setRunIntervalInSeconds(Long.parseLong(runIntervalInSecondsProp));

            if (runTimeOfDayProp == null)
                this.setRunTimeOfDay((TimeOfDay) null);
            else
                this.setRunTimeOfDay(new TimeOfDay(runTimeOfDayProp));
        }
    }

    /**
     * Read property.
     *
     * @param props        the props
     * @param propertyName the property name
     *
     * @return the string
     */
    private String readProperty(Properties props, String propertyName)
    {
        String propValue = StringUtils.toStringOrNull(props.get(propertyName));

        if (StringUtils.isBlank(propValue))
            return null;
        else
            return propValue;
    }

    /**
     * @see java.lang.Thread#run()
     */
    @Override
    public final void run()
    {
        while (true)
        {
            try
            {
                if (!DIFInitializer.isInitialized())
                {
                    synchronized (this)
                    {
                        wait(10000); // wait 10s until DIF initializes
                    }
                }
                else
                {
                    boolean isSingleRun = this.getJobType() == JobType.SINGLE_EXECUTION;
                    boolean isTimeOfDayScheduled = this.getRunTimeOfDay() != null;
                    boolean isIntervalScheduled = !isTimeOfDayScheduled;
                    boolean runIntervalSchedule =
                            isIntervalScheduled && !new Long(0L).equals(this.getRunIntervalInSeconds());

                    AuditContext.setProcessNameForCurrentThread("Job:" + this.getClass().getSimpleName());
                    AuditContext.setUserForCurrentThread(userName);

                    // This is started statically and so the DIF might not be initialized yet. Must not trigger an
                    // initialize, will wait for it to happen in due time The 0 check is to allow disabling of the job
                    // by setting this to 0
                    if (isSingleRun || isTimeOfDayScheduled || runIntervalSchedule)
                        try
                        {
                            if (isActive())
                            {
                                // start a new thread so that we can have isolation between separate runs of the
                                // specific business logic
                                Worker worker = new Worker(this);

                                this.runningWorker = worker;

                                worker.start();
                                worker.join();

                                this.runningWorker = null;

                                this.setLastRunDate(new Date());
                            }
                        }
                        catch (Exception e)
                        {
                            this.runningWorker = null;

                            BusinessException exception =
                                    new BusinessException("Error executing Job " + this.getName() + " run", e);

                            exception.log(LogLevel.WARN);
                            DIFIoCRegistry.getRegistry().getImplementation(IErrorLogManager.class)
                                    .logError("DIF", "DIFJob:" + this.getName(), exception);
                        }

                    if (isSingleRun)
                        break;
                    else
                    {
                        cleanUpAfterRun();

                        if (isIntervalScheduled)
                        {
                            // Wait
                            synchronized (this)
                            {
                                long waitTime = (new Long(0L).equals(this.getRunIntervalInSeconds()) ||
                                                 this.getRunIntervalInSeconds() == null ? 60 : this
                                                         .getRunIntervalInSeconds()) * 1000;

                                DIFLogger.getLogger()
                                        .debug(this.getJobName() + ": Wait for next recurrent execution (will wait " +
                                               TimeUtils.getTimePassed(waitTime, Scale.MILI_SECONDS) + ")...");

                                wait(waitTime);
                            }
                        }
                    }
                }
            }
            catch (Exception e)
            {
                BusinessException exception =
                        new BusinessException("Error executing Job " + this.getName() + " run", e);

                exception.log(LogLevel.WARN);
                DIFIoCRegistry.getRegistry().getImplementation(IErrorLogManager.class)
                        .logError("DIF", "DIFJob:" + this.getName(), exception);
            }
        }

        cleanUpAfterRun();
    }

    /**
     *
     */
    public void wakeupWorker()
    {
        if (this.runningWorker != null)
        {
            synchronized (this.runningWorker)

            {
                this.runningWorker.notify();
            }
        }

        synchronized (this)
        {
            this.notify();
        }

        DIFLogger.getLogger().info("Waked up Job \"" + this.getJobName() + "\" to read new settings");
    }

    /**
     * Write configuration to persistence
     *
     * @exception ConfigurationException
     */
    public void writeConfig() throws ConfigurationException
    {
        IConfigurations configManager = DIFIoCRegistry.getRegistry().getImplementation(IConfigurations.class);
        Properties props = new Properties();

        props.setProperty("active", Boolean.toString(this.isActive()));
        props.setProperty("runIntervalInSeconds", Long.toString(this.getRunIntervalInSeconds()));
        props.setProperty("runTimeOfDay", this.getRunTimeOfDay() == null ? "" : this.getRunTimeOfDay().asString());

        configManager.writeConfiguration("DIF2", "Jobs/" + this.getJobName(), props);

        DIFLogger.getLogger().info("Saving Job \"" + this.getJobName() + "\" settings...\n" + props.toString());
    }

    /**
     * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
     * @created 28/07/2016
     */
    private class Worker extends Thread
    {

        /**
         *
         */
        private DIFJob jobInstance;

        /**
         * @param jobInstance
         */
        public Worker(DIFJob jobInstance)
        {
            this.jobInstance = jobInstance;
        }

        /**
         * @see java.lang.Thread#run()
         */
        @Override
        public void run()
        {
            Chronometer crono = new Chronometer();
            Date startDate = new Date();
            jobInstance.totalExecutions++;

            try
            {
                AuditContext.setProcessNameForCurrentThread("Job:" + this.getClass().getSimpleName());
                AuditContext.setUserForCurrentThread(userName);

                // Time of day specified. Create a TimerTask
                if (getRunTimeOfDay() != null)
                {
                    // Determine next time to run (may have passed the desired hour today and if so calculate tommorow's
                    // time
                    Calendar now = Calendar.getInstance();
                    Calendar nextExecution = Calendar.getInstance();
                    nextExecution.set(Calendar.HOUR_OF_DAY, getRunTimeOfDay().getHour());
                    nextExecution.set(Calendar.MINUTE, getRunTimeOfDay().getMinutes());
                    nextExecution.set(Calendar.MILLISECOND, 0);

                    // In case the hour has passed, will increment 1 day
                    if (nextExecution.before(now))
                        nextExecution.add(Calendar.DATE, 1);

                    long timeToWait = nextExecution.getTimeInMillis() - now.getTimeInMillis();

                    synchronized (this)
                    {
                        DIFLogger.getLogger().debug(jobInstance.getJobName() +
                                                    ": Wait for next \"Time Of Day\" execution (will wait " +
                                                    TimeUtils.getTimePassed(timeToWait, Scale.MILI_SECONDS) + ")...");
                        wait(timeToWait); // wait until desired date/time
                        DIFLogger.getLogger()
                                .debug(jobInstance.getJobName() + ": Wait for next \"Time Of Day\" complete");
                    }
                }

                DIFLogger.getLogger().debug(jobInstance.getJobName() + ": Executing...");

                crono.start();
                boolean result = jobInstance.executeEachTime();

                crono.end();
                DIFLogger.getLogger().debug(jobInstance.getJobName() + ": Execution complete");
                jobInstance.executionLog
                        .add(new JobExecution(totalExecutions, startDate, crono.getTimePassedInMilisecs(), result));
            }
            catch (Exception e)
            {
                crono.end();

                jobInstance.executionLog
                        .add(new JobExecution(totalExecutions, startDate, crono.getTimePassedInMilisecs(), false)
                                .setFeedback(e.getMessage()));

                throw new RuntimeErrorException(new Error(e));
            }

            jobInstance.cleanUpAfterRun();
        }
    }
}
