package pt.digitalis.dif.utils.logging;

import java.util.List;

import pt.digitalis.dif.codegen.util.DEMLoaderEntityRegistry;
import pt.digitalis.dif.codegen.util.DEMLoaderHelper;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.bytecode.holders.ClassHolder;

/**
 * Contains the logging logic to be applied on the DEM building procedure.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007/08/21
 */
final public aspect DEMEntityLogAspect extends DIFLogAspect {

    /** Class counter */
    private long classCount = 0;

    /** First time the DEMAnnotatyions search is ran? */
    private boolean firstDEMAnnotationsSearch = true;

    /** Keeps the class enhancing start timestamp */
    private long classEnhanceStartTime;

    /**
     * Captures the construction of each new DEM Entity class class while performing the search for entities on a given
     * package. Exposes the javassist CtClass object used on the construction.
     *
     * @param className
     *            the class name
     */
    private pointcut buildClassHolder(String className) : withincode (* pt.digitalis.dif.codegen.util.DEMLoaderHelper.getDEMEntityClassesInPackage(..)) && call (pt.digitalis.utils.bytecode.holders.ClassHolder.new(String)) && args (className) && excludeLoggingAppliances();

    /**
     * Captures the search for existing DEM Entity classes, exposing the name of the package from where the classes are
     * read.
     *
     * @param packageName
     *            the name of the package
     */
    private pointcut searchForDEMEntities(String packageName) : execution (* pt.digitalis.dif.codegen.util.DEMLoaderHelper.getDEMEntityClassesInPackage(String)) && args (packageName) && excludeLoggingAppliances();

    /** Captures the existing package classes. */
    private pointcut existingDEMAnnotations() : execution (* pt.digitalis.dif.codegen.util.DEMLoaderHelper.getDEMEntitiesAnnontations()) && excludeLoggingAppliances();

    /** Captures the DEMLoaderHelper clean up point. */
    private pointcut theDEMLoaderHelperCleanUpPoint() : execution (* pt.digitalis.dif.codegen.util.DEMLoaderHelper.cleanUp()) && excludeLoggingAppliances();

    /** Captures the DEM Entity classes fetching for a package. */
    private pointcut DEMEntityClassesLoading() : execution (* pt.digitalis.dif.codegen.util.DEMLoaderHelper.getDEMEntityClassesInPackage(String)) && excludeLoggingAppliances();

    /** Captures the DEM Entity classes identifying process. */
    private pointcut DEMEntitiesEnhanceStart() : execution (* pt.digitalis.dif.codegen.DIFCodeGenerator+.enhanceDEMClasses(..)) && excludeLoggingAppliances();

    /** Captures the DEM Entity classes enhancing process end. */
    private pointcut DEMEntitiesEnhanceEnd() : execution (* pt.digitalis.dif.codegen.DIFCodeGenerator.cleanUp()) && excludeLoggingAppliances();

    /** Captures the registry of an app suite. */
    private pointcut appSuiteRegistry() : execution (* pt.digitalis.dif.dem.config.IDEMRegistrator+.registerAppSuiteBasePackage(String)) && excludeLoggingAppliances();

    /** Captures the registry of a single meta-model package. */
    private pointcut singleMetaModelPackageRegistry() : execution (* pt.digitalis.dif.dem.config.IDEMRegistrator+.registerMetaModelPackage(String)) && excludeLoggingAppliances();

    /**
     * Logs the number of analysed classes after searching a given package for DEM Entities.
     *
     * @param packageName
     *            the name of the package
     */
    after(String packageName) : searchForDEMEntities (packageName)
    {
        getLogger().info("Package \"" + packageName + "\": Analysed " + classCount + " classes.");

        // Reset class counter
        classCount = 0;
    }

    /**
     * Traces the each read class
     *
     * @param className
     *            the class name
     */
    after(String className) : buildClassHolder (className)
    {
        getLogger().trace("Analyzing class: " + className + "...");

        classCount++;
    }

    /**
     * Logs the existing DEM annotations.
     *
     * @param existingDEMAnnotations
     *            the exposed method return
     */
    after() returning (List<String> existingDEMAnnotations) : existingDEMAnnotations ()
    {
        if (firstDEMAnnotationsSearch) {

            firstDEMAnnotationsSearch = false;

            // Get defined log level
            LogLevel logLevel = LogLevel.DEBUG;

            // DIFInitializer message to log...
            String messageToLog = null;

            // If DEM Annotations could be read
            if (existingDEMAnnotations.size() != 0 && existingDEMAnnotations != null) {

                // Using StringBuffer to prevent multiple String copies while
                // appending annotations names
                StringBuffer annotationsAsString = new StringBuffer(" ");
                annotationsAsString.append("\n   \\");

                // Compose message
                for (String annotation : existingDEMAnnotations) {
                    annotationsAsString.append("\n   | " + annotation);
                }

                messageToLog = "Found the following DEM Entity annotations: " + annotationsAsString.toString() + "\n";

            } else {
                logLevel = LogLevel.WARN;
                messageToLog = "No DEM Entity annotations found!!";
            }

            // Log message
            this.log(logLevel, messageToLog);
        }
    }

    /** Logs the DEMLoaderHelper clean up. */
    after() : theDEMLoaderHelperCleanUpPoint()
    {
        getLogger().debug("Cleaned DEMLoaderHelper resources...");
    }

    /**
     * Logs DEM Entities that were identified
     */
    before() : DEMEntitiesEnhanceStart()
    {
        classEnhanceStartTime = System.currentTimeMillis();

        getLogger().info(
                "Loading DEM Entities from " + DEMLoaderHelper.getPackageList().size() + " registered packages...");
    }

    /**
     * Logs DEM Entities that were identified
     */
    before() : DEMEntitiesEnhanceEnd()
    {
        getLogger().info(
                "Loaded " + DEMLoaderEntityRegistry.getEntityCount() + " Entities in "
                        + (System.currentTimeMillis() - classEnhanceStartTime) + " ms.");
    }

    /**
     * Logs DEM Entity class information.
     *
     * @param packageName
     *            the name of the package
     * @param demClassNames
     *            the exposed method's return
     */
    after(String packageName) returning (List<ClassHolder> demClassNames) : DEMEntityClassesLoading() && args (packageName)
    {
        // Get defined log level
        LogLevel logLevel = LogLevel.INFO;

        // DIFInitializer message to log...
        StringBuffer messageToLog = new StringBuffer();

        // If classes were found...
        if (demClassNames.size() > 0) {

            if (super.getLogger().isDebugEnabled() || super.getLogger().isTraceEnabled()) {
                logLevel = LogLevel.DEBUG;
                messageToLog.append("Package \"" + packageName + "\": Found the following DEM Entity classes");
                messageToLog.append("\n   \\");

                for (ClassHolder clazz : demClassNames) {
                    messageToLog.append("\n   | " + clazz.getFQName());
                }

                messageToLog.append("\n");
            } else {
                logLevel = LogLevel.INFO;
                messageToLog.append("Package \"" + packageName + "\": Found " + demClassNames.size()
                        + " DEM Entity classes");
            }

            // Log message...
            this.log(logLevel, messageToLog.toString());
        }
        // ...if the package doesn't contain any DEM Entity
        else {
            // Change log level to WARN!!
            logLevel = LogLevel.WARN;
            // Compose error message...
            messageToLog.append("Package \"" + packageName + "\": Didn't find any DEM Entity classes in package!");
            // Log message...
            this.log(logLevel, messageToLog.toString());
        }

    }

    /**
     * Log information regarding the registry of an app suite.
     *
     * @param topLevelPackage
     *            the top-level package for the app suite
     * @param success
     *            the exposed method result
     */
    after(String topLevelPackage) returning (boolean success) : appSuiteRegistry() && args(topLevelPackage)
    {
        // DIFInitializer message to log
        String messageToLog = null;
        // Default log level...
        LogLevel level = LogLevel.DEBUG;

        if (success) {
            // Compose message
            messageToLog = "Successfully registered the app suite on: " + topLevelPackage + ".";

        } else {
            // Log message as a warn...
            level = LogLevel.WARN;
            // Compose error message
            messageToLog = "Could not register app suite on: " + topLevelPackage + "!";
        }

        // Log message...
        this.log(level, messageToLog);

    }

    /**
     * Log information regarding the registry of as single meta model package.
     *
     * @param packageName
     *            the package name
     * @param success
     *            the exposed method result
     */
    after(String packageName) returning (boolean success) : singleMetaModelPackageRegistry() && args(packageName)
    {
        // DIFInitializer message to log
        String messageToLog = null;
        // Default log level...
        LogLevel level = LogLevel.DEBUG;

        if (success) {
            // Compose message
            messageToLog = "Package \"" + packageName + "\": Successfully registered as a meta model package.";

        } else {
            // Log message as a warn...
            level = LogLevel.WARN;
            // Compose error message
            messageToLog = "Package \"" + packageName + "\": Could not register as a meta model package!";
        }

        // Log message...
        this.log(level, messageToLog);

    }

    /**
     * Helper method that logs a given message with a given level.
     *
     * @param level
     *            the log level
     * @param message
     *            the message to log
     */
    private void log(LogLevel level, String message) {
        super.getLogger().log(level, message);
    }
}
