/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.utils.logging;

import pt.digitalis.dif.startup.DIFStartupConfiguration;
import pt.digitalis.dif.exception.DIFException;
import pt.digitalis.dif.exception.InternalFrameworkException;
import pt.digitalis.dif.exception.controller.UserHasNoAccessToStage;
import pt.digitalis.log.LogLevel;

/**
 * This aspect logs all system-wide exceptions.
 *
 * @author Rodrigo Gon�alves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created 2007/05/23
 */
public aspect DIFExceptionLogAspect extends DIFLogAspect {

    /**
     * Captures all exception handlers, exposing the exception to the advice and excluding this aspect (avoid
     * recursion).
     *
     * @param exception
     *            the exception to treat
     */
    private pointcut exceptionHandlers(Exception exception) : handler(Exception+) && args(exception) && excludeSystemUtils() && !within(DIFExceptionLogAspect+) && !cflow(adviceexecution());

    /**
     * Logs all exceptions thrown outside the aspect. It treats the situation where the exception comes from a
     * framework-generated code snippet. In this situation the exception is logged with a lower logging level.
     *
     * @param exception
     *            the thrown exception.
     */
    before(Exception exception) : exceptionHandlers (exception)
    {
        // Prevent exception concurrency for the System.err
        synchronized (exception)
        {

            // Set log level accordingly to exception type
            LogLevel level = (exception instanceof InternalFrameworkException) ? LogLevel.WARN : LogLevel.FATAL;

            // Log exception
            printExceptionMessage(level, exception);

            if (DIFStartupConfiguration.isPrintStackTrace() && !(exception instanceof UserHasNoAccessToStage))
                exception.printStackTrace();
        }
    }

    /**
     * Prints a message with information regarding the exception and the throwing point.
     *
     * @param level
     *            the level to log the message
     * @param exception
     *            the thrown exception
     */
    private void printExceptionMessage(LogLevel level, Throwable exception)
    {

        // Initialize message to return
        String messageToReturn = null;

        if (exception instanceof UserHasNoAccessToStage)
            messageToReturn = exception.getMessage();
            
        else if (getLogger().equals(LogLevel.DEBUG) || getLogger().equals(LogLevel.TRACE)
                || DIFStartupConfiguration.getDeveloperMode()
                || DIFStartupConfiguration.getTestingMode())
        {

            // Add exception class to message
            messageToReturn = "Exception " + exception.getClass() + "\n";

            // Add context if it's a DiFException
            if (exception instanceof DIFException)
                messageToReturn += ((DIFException) exception).getRenderedExceptionContext();
            else
                messageToReturn += exception.toString();

        }
        else if (exception.getStackTrace().length > 0)
        {
            StackTraceElement element = exception.getStackTrace()[0];

            // Add exception class to message
            messageToReturn = element.getFileName() + ": " + exception.getClass().getSimpleName() + " ("
                    + element.getMethodName() + ":" + element.getLineNumber() + ")";
        }

        // Log message
        getLogger().log(level, messageToReturn);
    }
}
