/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica.
 * Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos num.9 -
 * Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21
 * 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.utils.logging;

import pt.digitalis.dif.startup.DIFStartupConfiguration;
import pt.digitalis.log.ILogWrapper;
import pt.digitalis.log.LogLevel;
import pt.digitalis.log.Logger;

/**
 * This is the framework's base logging aspect. It defines the logging appliance used as a protected member that can be
 * initialized on each sub-aspect constructor, an abstract pointcut - logPoints() - and an abstract logging method that
 * should be implemented by the sub-aspects. Access to the logger should be done through the supplied inspector. Logging
 * of all the Exceptions thrown by the framework or its client applications is also done by this aspect. Exception
 * logging is a top-level concern so it should be on the base aspect. This aspect also supplies some common pointcuts to
 * its children. These include capturing all methods and constructors execution and the exclusion of the logging
 * appliances (such as this aspects and its children as well as the Logger class and its children).
 *
 * @author Rodrigo Gonçalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created 2007/05/22
 */

public abstract aspect DIFLogAspect {

    /* --- Members --- */

    /** The logger instance */
    static private ILogWrapper logger;

    /* --- General pointcuts --- */

    /** Excluding all development aspects. */
    protected pointcut excludeAllDev() : ! within(pt.digitalis.dif.utils.development.*);

    /** Excluding all startup classes. */
    protected pointcut excludeAllStartup() : ! within(pt.digitalis.dif.startup.DIFStartupConfiguration);

    /** Excluding all startup classes. */
    protected pointcut excludeSystemUtils() : ! within(pt.digitalis.dif.utils.system.SystemUtils);

    /**
     * This pointcut excludes all the logging appliances such as this aspect and all its children as well as the
     * Logger class and any eventual children it will might to have.
     */
    protected pointcut excludeLoggingAppliances() : !within(DIFLogAspect+) && !within(DIFLoggerInterceptor+) && !within(DIFLoggerInterceptorImpl+) && ! within(ILogWrapper+) &&
        excludeAllDev() && excludeAllStartup();

    /**
     * Captures the execution of all methods that are not part of the aspect or the Logger class.
     */
    protected pointcut allMethods() : execution(* pt.digitalis.dif.*..*(..)) && excludeLoggingAppliances();

    /**
     * Captures the execution of all constructors that are not part of the aspect or the Logger class.
     */
    protected pointcut allConstructors() : execution(pt.digitalis.dif.*..*.new(..)) && excludeLoggingAppliances();

    /* --- Methods --- */

    /**
     * Inspector for the defined logger object.
     *
     * @return the active logger instance
     */
    static protected ILogWrapper getLogger() {

        // Lazy loading
        if (logger == null) {
            LogLevel level = null;

            // Try to get log level from DiF configurations
            try {
                level = DIFStartupConfiguration.getLogLevel();
            } catch (Exception e) {
                // Catch RTEs such as NPEs and do nothing. The code below will make the logging proceed in a transparent and convenient way.
            }

            // If level could not be fetched before, set it to WARN
            if(level == null)
                level = LogLevel.WARN;

            // Initialize logger with the appropriate level.
            logger = Logger.getLogger("DIF Core", level);
        }

        // Return the active logger instance
        return logger;
    }
}
