/**
 * 2016, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.logging;

import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.exception.BusinessException;
import pt.digitalis.dif.flightrecorder.FlightRecorder;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.log.ILogWrapper;
import pt.digitalis.log.ILoggerInterceptor;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.IConfigurations;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 27/12/2016
 */
public class DIFLoggerInterceptorImpl implements ILogWrapper
{

    /** list of classes to ignore when determining the class that issued the log action */
    public static List<String> classesToIgnore = null;

    /**
     *
     */
    static boolean inteceptClassCals = false;

    /**
     *
     */
    static LogLevel inteceptLogLevel = LogLevel.INFO;

    /** if T will intercept logging */
    static boolean interceptMode = false;

    /**
     *
     */
    private static List<String> classesToMonitor = new ArrayList<String>();

    /**
     *
     */
    private static List<String> packagesToMonitor = new ArrayList<String>();

    static
    {
        classesToIgnore = new ArrayList<String>();
        classesToIgnore.add(Thread.class.getCanonicalName());
        classesToIgnore.add(DIFLoggerInterceptorImpl.class.getCanonicalName());
    }

    /** the real logger class we intercept and manage */
    private ILogWrapper logger;

    /**
     * creates a new instance of the interceptor logger passing the real logger class instance that will be managed by
     * the interceptor
     *
     * @param logger the real logger to use
     */
    public DIFLoggerInterceptorImpl(ILogWrapper logger)
    {
        this.logger = logger;
    }

    /**
     * Adds a stage to the classes to intercept list
     *
     * @param stage the stage definition object
     */
    public static void addStageToInterpectList(IStage stage)
    {
        boolean alreadyPresent = classesToMonitor.contains(stage.getOriginalClassName().toLowerCase());

        if (!alreadyPresent)
        {
            classesToMonitor.add(stage.getOriginalClassName().toLowerCase());

            LoggingConfiguration logConfig = LoggingConfiguration.getInstance();

            logConfig.setCaptureLogForClasses(CollectionUtils.listToCommaSeparatedString(classesToMonitor));

            try
            {
                DIFIoCRegistry.getRegistry().getImplementation(IConfigurations.class).writeConfiguration(logConfig);
            }
            catch (Exception e)
            {
                new BusinessException("Error while saving logging configurations for stage intercept", e)
                        .addToExceptionContext("Stage", stage).log(LogLevel.ERROR);
            }

            DIFLoggerInterceptorImpl.reavaluateInterceptorState();
        }
    }

    /**
     * Adds a stage to the classes to intercept list
     *
     * @param stageID the stage ID
     */
    public static void addStageToInterpectList(String stageID)
    {
        addStageToInterpectList(DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class).getStage(stageID));
    }

    /**
     * Inspector for the 'interceptMode' attribute.
     *
     * @return the interceptMode value
     */
    public static boolean isInterceptMode()
    {
        return interceptMode;
    }

    /**
     * Modifier for the 'interceptMode' attribute.
     *
     * @param interceptMode the new interceptMode value to set
     */
    public static void setInterceptMode(boolean interceptMode)
    {
        DIFLoggerInterceptorImpl.interceptMode = interceptMode;
    }

    /**
     * Determines if a given class in being monitored by the log interceptor
     *
     * @param className the class to check (canonical name)
     *
     * @return T if the class is being monitored
     */
    public static boolean isMonitoredClass(String className)
    {
        return classesToMonitor.contains(className.toLowerCase());
    }

    /**
     * Determines if a given stage in being monitored by the log interceptor
     *
     * @param stage the stage to check
     *
     * @return T if the stage is being monitored
     */
    public static boolean isMonitoredStage(IStage stage)
    {
        return isMonitoredClass(stage.getOriginalClassName());
    }

    /**
     * Determines if a given stage in being monitored by the log interceptor
     *
     * @param stageID the stage to check
     *
     * @return T if the stage is being monitored
     */
    public static boolean isMonitoredStage(String stageID)
    {
        return isMonitoredStage(DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class).getStage(stageID));
    }

    /**
     * Reavaluates the logging configuration to determine of the interceptor should came to life of remain in pass
     * through mode
     */
    public static void reavaluateInterceptorState()
    {
        classesToMonitor.clear();
        if (StringUtils.isNotBlank(LoggingConfiguration.getInstance().getCaptureLogForClasses()))
            classesToMonitor.addAll(Arrays
                    .asList(LoggingConfiguration.getInstance().getCaptureLogForClasses().toLowerCase().split(",")));

        packagesToMonitor.clear();
        if (StringUtils.isNotBlank(LoggingConfiguration.getInstance().getCaptureLogForPackages()))
            packagesToMonitor.addAll(Arrays
                    .asList(LoggingConfiguration.getInstance().getCaptureLogForPackages().toLowerCase().split(",")));

        LogLevel inteceptLogLevel = null;
        inteceptLogLevel = LogLevel.valueOf(LoggingConfiguration.getInstance().getCaptureLogDebugLevel());
        inteceptClassCals = LoggingConfiguration.getInstance().getCaptureLogForClassCalls();

        setInterceptMode((!classesToMonitor.isEmpty() || !packagesToMonitor.isEmpty()) && inteceptLogLevel != null);
    }

    /**
     * Removes a stage from the classes to intercept list
     *
     * @param stage the stage definition object
     */
    public static void removeStageFromInterpectList(IStage stage)
    {
        boolean alreadyPresent = classesToMonitor.contains(stage.getOriginalClassName().toLowerCase());

        if (alreadyPresent)
        {
            classesToMonitor.remove(stage.getOriginalClassName().toLowerCase());

            LoggingConfiguration logConfig = LoggingConfiguration.getInstance();

            logConfig.setCaptureLogForClasses(CollectionUtils.listToCommaSeparatedString(classesToMonitor));

            try
            {
                DIFIoCRegistry.getRegistry().getImplementation(IConfigurations.class).writeConfiguration(logConfig);
            }
            catch (Exception e)
            {
                new BusinessException("Error while saving logging configurations for stage intercept", e)
                        .addToExceptionContext("Stage", stage).log(LogLevel.ERROR);
            }

            DIFLoggerInterceptorImpl.reavaluateInterceptorState();
        }
    }

    /**
     * Removes a stage from the classes to intercept list
     *
     * @param stageID the stage ID
     */
    public static void removeStageFromInterpectList(String stageID)
    {
        removeStageFromInterpectList(
                DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class).getStage(stageID));
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#addInterceptor(pt.digitalis.log.ILoggerInterceptor)
     */
    public void addInterceptor(ILoggerInterceptor interceptor)
    {
        this.logger.addInterceptor(interceptor);
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#debug(java.lang.Object)
     */
    public void debug(Object message)
    {
        this.log(LogLevel.DEBUG, message);
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#decreaseIndentation()
     */
    public void decreaseIndentation()
    {
        this.logger.decreaseIndentation();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#error(java.lang.Object)
     */
    public void error(Object message)
    {
        this.log(LogLevel.ERROR, message);
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#fatal(java.lang.Object)
     */
    public void fatal(Object message)
    {
        this.log(LogLevel.FATAL, message);
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#getIndentationString()
     */
    public String getIndentationString()
    {
        return logger.getIndentationString();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#getLevel()
     */
    public LogLevel getLevel()
    {
        return this.logger.getLevel();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#setLevel(pt.digitalis.log.LogLevel)
     */
    public void setLevel(LogLevel level)
    {
        this.logger.setLevel(level);
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#increaseIndentation()
     */
    public void increaseIndentation()
    {
        this.logger.increaseIndentation();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#info(java.lang.Object)
     */
    public void info(Object message)
    {
        this.log(LogLevel.INFO, message);
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#isDebugEnabled()
     */
    public boolean isDebugEnabled()
    {
        return logger.isDebugEnabled();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#isErrorEnabled()
     */
    public boolean isErrorEnabled()
    {
        return logger.isErrorEnabled();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#isFatalEnabled()
     */
    public boolean isFatalEnabled()
    {
        return logger.isFatalEnabled();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#isInfoEnabled()
     */
    public boolean isInfoEnabled()
    {
        return logger.isInfoEnabled();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#isTraceEnabled()
     */
    public boolean isTraceEnabled()
    {
        return logger.isTraceEnabled();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#isWarnEnabled()
     */
    public boolean isWarnEnabled()
    {
        return logger.isWarnEnabled();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#log(pt.digitalis.log.LogLevel, java.lang.Object)
     */
    public void log(LogLevel level, Object message)
    {
        FlightRecorder.reportLog(level, message);

        if (interceptMode)
        {
            boolean intercept = false;
            String className = null;
            StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
            int traceItem = 0;
            boolean ignoreClass = false;

            do
            {
                // determine className and if it is a class to ignore
                className = stackTrace[traceItem++].getClassName();
                ignoreClass = classesToIgnore.contains(className);

                if (!ignoreClass)
                {
                    // See if the class is one of the declared classes to intercept
                    for (String clazz : classesToMonitor)
                    {
                        intercept = intercept || StringUtils.containsIgnoreCase(className, clazz);

                        if (intercept)
                            break;
                    }

                    // See if the class belongs to one of the declared packages to intercept
                    for (String packageName : packagesToMonitor)
                    {
                        intercept = intercept || StringUtils.containsIgnoreCase(className, packageName);

                        if (intercept)
                            break;
                    }

                    // Only direct calls to logger from the class should be logged when interceptClassCalls is true.
                    // If the class is not to ignore and we do not capture class calls then the current evaluation is
                    // the only one to perform.
                    if (!inteceptClassCals && !ignoreClass)
                        break;
                }
            } while (!intercept && traceItem < stackTrace.length);

            if (intercept)
            {
                // Decide whether to intercept or not depending on the requested log level to intercept
                if (level == LogLevel.TRACE)
                    intercept = true;
                else if (level == LogLevel.DEBUG)
                    intercept = (inteceptLogLevel == LogLevel.DEBUG) || (inteceptLogLevel == LogLevel.INFO) ||
                                (inteceptLogLevel == LogLevel.WARN) || (inteceptLogLevel == LogLevel.ERROR) ||
                                (inteceptLogLevel == LogLevel.FATAL);
                else if (level == LogLevel.INFO)
                    intercept = (inteceptLogLevel == LogLevel.INFO) || (inteceptLogLevel == LogLevel.WARN) ||
                                (inteceptLogLevel == LogLevel.ERROR) || (inteceptLogLevel == LogLevel.FATAL);
                else if (level == LogLevel.WARN)
                    intercept = (inteceptLogLevel == LogLevel.WARN) || (inteceptLogLevel == LogLevel.ERROR) ||
                                (inteceptLogLevel == LogLevel.FATAL);
                else if (level == LogLevel.ERROR)
                    intercept = (inteceptLogLevel == LogLevel.ERROR) || (inteceptLogLevel == LogLevel.FATAL);
                else if (level == LogLevel.FATAL)
                    intercept = (inteceptLogLevel == LogLevel.FATAL);
            }

            if (intercept)
                System.out.println("[" + level.name() + "] " + message);
            else
                this.logger.log(level, message);
        }
        else
            this.logger.log(level, message);
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#setDebugLogLevel()
     */
    public void setDebugLogLevel()
    {
        this.logger.setDebugLogLevel();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#setErrorLogLevel()
     */
    public void setErrorLogLevel()
    {
        this.logger.setErrorLogLevel();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#setFatalLogLevel()
     */
    public void setFatalLogLevel()
    {
        this.logger.setFatalLogLevel();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#setInfoLogLevel()
     */
    public void setInfoLogLevel()
    {
        this.logger.setInfoLogLevel();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#setLevel(pt.digitalis.log.LogLevel, java.lang.Boolean)
     */
    public void setLevel(LogLevel level, Boolean changeAppenderLevel)
    {
        this.logger.setLevel(level, changeAppenderLevel);
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#setTraceLogLevel()
     */
    public void setTraceLogLevel()
    {
        this.logger.setTraceLogLevel();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#setWarnLogLevel()
     */
    public void setWarnLogLevel()
    {
        this.logger.setWarnLogLevel();
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#trace(java.lang.Object)
     */
    public void trace(Object message)
    {
        this.log(LogLevel.TRACE, message);
    }

    /**
     * @see pt.digitalis.log.ILogWrapper#warn(java.lang.Object)
     */
    public void warn(Object message)
    {
        this.log(LogLevel.WARN, message);
    }
}
