/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.logging;

import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFStartupConfiguration;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.config.IConfigurations;
import pt.digitalis.utils.config.annotations.ConfigAlias;
import pt.digitalis.utils.config.annotations.ConfigDefault;
import pt.digitalis.utils.config.annotations.ConfigID;
import pt.digitalis.utils.config.annotations.ConfigIgnore;
import pt.digitalis.utils.config.annotations.ConfigLOVValues;
import pt.digitalis.utils.config.annotations.ConfigSectionID;
import pt.digitalis.utils.config.annotations.ConfigVirtualPathForNode;

/**
 * Configuration point for the Logging features
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Sep 24, 2009
 */
@ConfigID("dif2")
@ConfigSectionID("Logging")
@ConfigVirtualPathForNode("dif2/Logs/General")
public class LoggingConfiguration
{

    /**
     * The singleton instance
     */
    static private LoggingConfiguration instance = null;

    /**
     * if T will capture calls for the calls the capture class makes
     */
    public Boolean captureLogForClassCalls;

    /**
     * If T activate auditing information in the model.
     */
    private Boolean auditDatabase;

    /**
     * The Audit database purge max entries.
     */
    private Long auditDatabasePurgeMaxEntries;

    /**
     * The Audit database purge max days.
     */
    private Long auditDatabasePurgeMaxDays;

    /**
     * the level of debug to capture
     */
    private String captureLogDebugLevel;

    /**
     * the list of classes to capture debug output to the system log
     */
    private String captureLogForClasses;

    /**
     * the list of packages to capture debug output to the system log
     */
    private String captureLogForPackages;

    /**
     * If T will send all emails to debugEmailAddress
     */
    private Boolean debugEmail;

    /**
     * The debug emaill address
     */
    private String debugEmailAddress;

    /**
     * If T activate debugging in the model.
     */
    private Boolean debugModel;

    /**
     * The Error Log Email Address
     */
    private String errorLogEmailAddress;

    /**
     * The error Log Email Body
     */
    private String errorLogEmailBody;

    /**
     * The error Log Email Subject
     */
    private String errorLogEmailSubject;

    /**
     * The Purge job interval.
     */
    private Long purgeJobInterval;

    /**
     * Inspector for the 'auditDatabase' attribute.<br/>
     * When this configuration has not been initialized will not trigger it's initialization, since that can trigger a
     * stack overflow as the initialization may call the DataSet API if it is database based and that API will inspact
     * the DebugModel property
     *
     * @return the auditDatabase value
     */
    @ConfigIgnore
    public static Boolean getAuditDatabaseIfInitialized()
    {
        if (instance == null)
            return false;
        else
            return getInstance().getAuditDatabase();
    }

    /**
     * Gets the debug model if initialized.<br/>
     * When this configuration has not been initialized will not trigger it's initialization, since that can trigger a
     * stack overflow as the initialization may call the DataSet API if it is database based and that API will inspact
     * the DebugModel property
     *
     * @return the debug model state
     */
    @ConfigIgnore
    static public boolean getDebugModelIfInitialized()
    {
        if (instance == null)
            return DIFStartupConfiguration.getLogLevel() == LogLevel.DEBUG;
        else
            return getInstance().getDebugModel();
    }

    /**
     * Returns the active configuration object instance.
     *
     * @return the configuration instance
     */
    @ConfigIgnore
    static public LoggingConfiguration getInstance()
    {
        if (instance == null)
            try
            {
                instance = DIFIoCRegistry.getRegistry().getImplementation(IConfigurations.class)
                        .readConfiguration(LoggingConfiguration.class);
            }
            catch (Exception e)
            {
                instance = null;
            }

        return instance;
    }

    /**
     * Inspector for the 'auditDatabase' attribute.
     *
     * @return the auditDatabase value
     */
    @ConfigDefault("false")
    public Boolean getAuditDatabase()
    {
        return auditDatabase;
    }

    /**
     * Modifier for the 'auditDatabase' attribute.
     *
     * @param auditDatabase the new auditDatabase value to set
     */
    public void setAuditDatabase(Boolean auditDatabase)
    {
        this.auditDatabase = auditDatabase;
    }

    /**
     * Gets audit database purge max days.
     *
     * @return the audit database purge max days
     */
    @ConfigDefault("1825")
    @ConfigAlias(name = "Audit Database (purge after max age reached)",
            description = "The maximum age in days of entries to keep. When it is reached a purge will begin deleting older entries a block at a time until the max limit is reached.")
    public Long getAuditDatabasePurgeMaxDays()
    {
        return auditDatabasePurgeMaxDays;
    }

    /**
     * Sets audit database purge max days.
     *
     * @param auditDatabasePurgeMaxDays the audit database purge max days
     */
    public void setAuditDatabasePurgeMaxDays(Long auditDatabasePurgeMaxDays)
    {
        this.auditDatabasePurgeMaxDays = auditDatabasePurgeMaxDays;
    }

    /**
     * Gets audit database purge max entries.
     *
     * @return the audit database purge max entries
     */
    @ConfigDefault("10000000")
    @ConfigAlias(name = "Audit Database (purge after max entries reached)",
            description = "The maximum number of entries to keep. When it is reached a purge will begin deleting older entries a block at a time until the max limit is reached.")
    public Long getAuditDatabasePurgeMaxEntries()
    {
        return auditDatabasePurgeMaxEntries;
    }

    /**
     * Sets audit database purge max entries.
     *
     * @param auditDatabasePurgeMaxEntries the audit database purge max entries
     */
    public void setAuditDatabasePurgeMaxEntries(Long auditDatabasePurgeMaxEntries)
    {
        this.auditDatabasePurgeMaxEntries = auditDatabasePurgeMaxEntries;
    }

    /**
     * Inspector for the 'captureLogDebugLevel' attribute.
     *
     * @return the captureLogDebugLevel value
     */
    @ConfigLOVValues("TRACE=Trace,DEBUG=Debug,WARN=Warn,INFO=Info,ERROR=Error,FATAL=Fatal")
    @ConfigDefault("DEBUG")
    public String getCaptureLogDebugLevel()
    {
        return captureLogDebugLevel;
    }

    /**
     * Modifier for the 'captureLogDebugLevel' attribute.
     *
     * @param captureLogDebugLevel the new captureLogDebugLevel value to set
     */
    public void setCaptureLogDebugLevel(String captureLogDebugLevel)
    {
        this.captureLogDebugLevel = captureLogDebugLevel;
    }

    /**
     * Inspector for the 'captureLogForClassCalls' attribute.
     *
     * @return the captureLogForClassCalls value
     */
    @ConfigDefault("false")
    public Boolean getCaptureLogForClassCalls()
    {
        return captureLogForClassCalls;
    }

    /**
     * Modifier for the 'captureLogForClassCalls' attribute.
     *
     * @param captureLogForClassCalls the new captureLogForClassCalls value to set
     */
    public void setCaptureLogForClassCalls(Boolean captureLogForClassCalls)
    {
        this.captureLogForClassCalls = captureLogForClassCalls;
    }

    /**
     * Inspector for the 'captureLogForClasses' attribute.
     *
     * @return the captureLogForClasses value
     */
    public String getCaptureLogForClasses()
    {
        return captureLogForClasses;
    }

    /**
     * Modifier for the 'captureLogForClasses' attribute.
     *
     * @param captureLogForClasses the new captureLogForClasses value to set
     */
    public void setCaptureLogForClasses(String captureLogForClasses)
    {
        this.captureLogForClasses = captureLogForClasses;
    }

    /**
     * Inspector for the 'captureLogForPackages' attribute.
     *
     * @return the captureLogForPackages value
     */
    public String getCaptureLogForPackages()
    {
        return captureLogForPackages;
    }

    /**
     * Modifier for the 'captureLogForPackages' attribute.
     *
     * @param captureLogForPackages the new captureLogForPackages value to set
     */
    public void setCaptureLogForPackages(String captureLogForPackages)
    {
        this.captureLogForPackages = captureLogForPackages;
    }

    /**
     * Inspector for the 'debugEmail' attribute.
     *
     * @return the debugEmail value
     */
    @ConfigDefault("false")
    public Boolean getDebugEmail()
    {
        return debugEmail;
    }

    /**
     * Modifier for the 'debugEmail' attribute.
     *
     * @param debugEmail the new debugEmail value to set
     */
    public void setDebugEmail(Boolean debugEmail)
    {
        this.debugEmail = debugEmail;
    }

    /**
     * Inspector for the 'debugEmailAddress' attribute.
     *
     * @return the debugEmailAddress value
     */
    @ConfigDefault("")
    public String getDebugEmailAddress()
    {
        return debugEmailAddress;
    }

    /**
     * Modifier for the 'debugEmailAddress' attribute.
     *
     * @param debugEmailAddress the new debugEmailAddress value to set
     */
    public void setDebugEmailAddress(String debugEmailAddress)
    {
        this.debugEmailAddress = debugEmailAddress;
    }

    /**
     * Inspector for the 'debugModel' attribute.
     *
     * @return the debugModel value
     */
    @ConfigDefault("false")
    public Boolean getDebugModel()
    {
        return debugModel;
    }

    /**
     * Modifier for the 'debugModel' attribute.
     *
     * @param debugModel the new debugModel value to set
     */
    public void setDebugModel(Boolean debugModel)
    {
        this.debugModel = debugModel;
    }

    /**
     * Inspector for the 'errorLogEmailAddress' attribute.
     *
     * @return the errorLogEmailAddress value
     */
    @ConfigDefault("")
    public String getErrorLogEmailAddress()
    {
        return errorLogEmailAddress;
    }

    /**
     * Modifier for the 'errorLogEmailAddress' attribute.
     *
     * @param errorLogEmailAddress the new errorLogEmailAddress value to set
     */
    public void setErrorLogEmailAddress(String errorLogEmailAddress)
    {
        this.errorLogEmailAddress = errorLogEmailAddress;
    }

    /**
     * Inspector for the 'errorLogEmailBody' attribute.
     *
     * @return the errorLogEmailBody value
     */
    @ConfigDefault(
            "An error occurred in application ${applicationName}. You can see the details authentication as administrator and see \"Error Log Viewer\" service.")
    public String getErrorLogEmailBody()
    {
        return errorLogEmailBody;
    }

    /**
     * Modifier for the 'errorLogEmailBody' attribute.
     *
     * @param errorLogEmailBody the new errorLogEmailBody value to set
     */
    public void setErrorLogEmailBody(String errorLogEmailBody)
    {
        this.errorLogEmailBody = errorLogEmailBody;
    }

    /**
     * Inspector for the 'errorLogEmailSubject' attribute.
     *
     * @return the errorLogEmailSubject value
     */
    @ConfigDefault("Error Log service")
    public String getErrorLogEmailSubject()
    {
        return errorLogEmailSubject;
    }

    /**
     * Modifier for the 'errorLogEmailSubject' attribute.
     *
     * @param errorLogEmailSubject the new errorLogEmailSubject value to set
     */
    public void setErrorLogEmailSubject(String errorLogEmailSubject)
    {
        this.errorLogEmailSubject = errorLogEmailSubject;
    }

    /**
     * Gets purge job interval.
     *
     * @return the purge job interval
     */
    @ConfigDefault("86400")
    public Long getPurgeJobInterval()
    {
        return purgeJobInterval;
    }

    /**
     * Sets purge job interval.
     *
     * @param purgeJobInterval the purge job interval
     */
    public void setPurgeJobInterval(Long purgeJobInterval)
    {
        this.purgeJobInterval = purgeJobInterval;
    }
}
