/**
 * 2016, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.logging.performance;

import org.apache.commons.collections4.queue.CircularFifoQueue;
import pt.digitalis.dif.controller.interfaces.IDIFRequest;

import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Queue;

/**
 * Default implementation of {@link IPerformanceLogger}
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 22/08/2016
 */
public class PerformanceLogger implements IPerformanceLogger
{

    /** The data snapshot origyn. */
    static public final Date dataSnapshotOrigyn = new Date();

    /** number milisecs in a day */
    private static final int DAY_IN_MILISECS = 24 * 60 * 60 * 1000;

    /** number milisecs in an hour */
    private static final int HOUR_IN_MILISECS = 60 * 60 * 1000;

    /** number milisecs in a minute */
    private static final int MINUTE_IN_MILISECS = 60 * 1000;

    /** number of days to keep statistics (default 60) */
    private static final int STATS_DAYS_LIMIT = 60;

    /** number of hours to keep statistics (default 240) */
    private static final int STATS_HOURS_LIMIT = 240;

    /** number of minutes to keep statistics (default 240) */
    private static final int STATS_MINUTES_LIMIT = 240;

    /** each stage/event request counter */
    private Map<String, StageRequestEntry> stageRequestTotals = new HashMap<String, StageRequestEntry>();

    /** Execution times data (by day). */
    private CircularFifoQueue<TimeStats> statsByDay = new CircularFifoQueue<TimeStats>(STATS_DAYS_LIMIT);

    /**
     *
     */
    private TimeStats statsByDayCurrentSample = null;

    /** Execution times data (by hour). */
    private CircularFifoQueue<TimeStats> statsByHour = new CircularFifoQueue<TimeStats>(STATS_HOURS_LIMIT);

    /**
     *
     */
    private TimeStats statsByHourCurrentSample = null;

    /** Execution times data (by minute). */
    private CircularFifoQueue<TimeStats> statsByMinute = new CircularFifoQueue<TimeStats>(STATS_MINUTES_LIMIT);

    /**
     *
     */
    private TimeStats statsByMinuteCurrentSample = null;

    /**
     * default constructor
     */
    public PerformanceLogger()
    {
        PerformanceLogger.populatePreviousEntriesFromTimeOrigyn(statsByDay, new TimeStats(), DAY_IN_MILISECS);
        PerformanceLogger.populatePreviousEntriesFromTimeOrigyn(statsByHour, new TimeStats(), HOUR_IN_MILISECS);
        PerformanceLogger.populatePreviousEntriesFromTimeOrigyn(statsByMinute, new TimeStats(), MINUTE_IN_MILISECS);

        new MonitorStatistics(this).start();
    }

    /**
     * Populate previous entries from time origyn.
     *
     * @param <T>
     * @param queue      the queue
     * @param emptyStats the empty stats
     * @param timeScale
     */
    static public <T extends AbstractTimeStats<?>> void populatePreviousEntriesFromTimeOrigyn(
            CircularFifoQueue<T> queue, T emptyStats, int timeScale)
    {
        Long currentDateTime = new Date().getTime();
        Long timeLapsePointer = PerformanceLogger.dataSnapshotOrigyn.getTime();

        while (timeLapsePointer < currentDateTime)
        {
            emptyStats.setDate(new Date(timeLapsePointer));
            queue.add(emptyStats);

            timeLapsePointer += timeScale;
        }
    }

    /**
     * Adds a sample to the timmed execution time statistics
     *
     * @param queue
     * @param timeStat
     * @param sample
     * @param timeFrame
     *
     * @return the updated or new time slot (when expired will create a new one)
     */
    synchronized private TimeStats addToStatsSpecific(Queue<TimeStats> queue, TimeStats timeStat, Long sample,
            long timeFrame)
    {
        // Freeze time for comparison
        long currentTime = System.currentTimeMillis();

        // Add the slot to the queue if not already there
        if (!queue.contains(timeStat))
            queue.add(timeStat);

        // If time frame of current statistical slot has expired, start a new one (previous is already in the queue)
        if (currentTime >= timeStat.getDate().getTime() + timeFrame)
        {
            timeStat = new TimeStats();
            queue.add(timeStat);
        }

        if (sample != null)
            timeStat.addSample(sample);

        return timeStat;
    }

    /**
     * Adds a sample to the timmed execution time statistics
     *
     * @param sample the execution time sample to add to time statistics
     */
    synchronized private void addToTimedStats(Long sample)
    {
        if (this.statsByMinuteCurrentSample == null)
            this.statsByMinuteCurrentSample = new TimeStats();

        if (this.statsByHourCurrentSample == null)
            this.statsByHourCurrentSample = new TimeStats();

        if (this.statsByDayCurrentSample == null)
            this.statsByDayCurrentSample = new TimeStats();

        this.statsByMinuteCurrentSample =
                addToStatsSpecific(statsByMinute, this.statsByMinuteCurrentSample, sample, MINUTE_IN_MILISECS);
        this.statsByHourCurrentSample =
                addToStatsSpecific(this.statsByHour, this.statsByHourCurrentSample, sample, HOUR_IN_MILISECS);
        this.statsByDayCurrentSample =
                addToStatsSpecific(this.statsByDay, this.statsByDayCurrentSample, sample, DAY_IN_MILISECS);
    }

    /**
     * Logs in audit mode the request.<br/>
     * No persistent layer exists in DIFCore so this class will be overridden if DIFDatabaseRepository is added to the
     * project. If not no action will be performed and no audit information will be kept.
     *
     * @param logData          the log data
     * @param timeSpent        the time spent
     * @param errorDescription if an error has occurred it's description will be present here
     */
    protected synchronized void auditRequest(LogRequestData logData, Long timeSpent, String errorDescription)
    {
        // No persistant layer in DIFCore. Will be overridden if DIFDatabaseRepository is added to the project
    }

    /**
     * @see pt.digitalis.dif.utils.logging.performance.IPerformanceLogger#getStageRequestEntries()
     */
    public Collection<StageRequestEntry> getStageRequestEntries()
    {
        return stageRequestTotals.values();
    }

    /**
     * @see pt.digitalis.dif.utils.logging.performance.IPerformanceLogger#getStatsByDay()
     */
    public Queue<TimeStats> getStatsByDay()
    {
        return statsByDay;
    }

    /**
     * @see pt.digitalis.dif.utils.logging.performance.IPerformanceLogger#getStatsByHour()
     */
    public Queue<TimeStats> getStatsByHour()
    {
        return statsByHour;
    }

    /**
     * @see pt.digitalis.dif.utils.logging.performance.IPerformanceLogger#getStatsByMinute()
     */
    public Queue<TimeStats> getStatsByMinute()
    {
        return statsByMinute;
    }

    /**
     * @see pt.digitalis.dif.utils.logging.performance.IPerformanceLogger#logRequest(pt.digitalis.dif.controller.interfaces.IDIFRequest,
     *         java.lang.Long, java.lang.String)
     */
    public void logRequest(IDIFRequest request, Long timeSpent, String errorDescription)
    {
        this.logRequest(new LogRequestData(request), timeSpent, errorDescription);
    }

    /**
     * @see pt.digitalis.dif.utils.logging.performance.IPerformanceLogger#logRequest(pt.digitalis.dif.utils.logging.performance.LogRequestData,
     *         java.lang.Long, java.lang.String)
     */
    public synchronized void logRequest(LogRequestData logData, Long timeSpent, String errorDescription)
    {
        PerformanceLoggerConfiguration config = PerformanceLoggerConfiguration.getInstance();

        if (config.getTrackRequests())
        {
            // Add to global time by period stats
            this.addToTimedStats(timeSpent);

            // Add to stage statistics
            String stageID = logData.getStageID();
            String eventID = logData.getEventID();

            String id = stageID;

            if (eventID == null)
                id += ":" + eventID;

            StageRequestEntry entry = this.stageRequestTotals.get(id);

            if (entry == null)
            {
                entry = new StageRequestEntry(stageID, eventID);
                this.stageRequestTotals.put(id, entry);
            }

            entry.addExecution(timeSpent);

            if (config.getAuditRequests())
                this.auditRequest(logData, timeSpent, errorDescription);
        }
    }

    /**
     * Updates timmed statistcs. Closed slots that have ended their period and starts new ones
     */
    synchronized public void updateTimedStats()
    {
        this.addToTimedStats(null);
    }

    /**
     * Monitor thread that will keep updating timed statistics in the background. Cannot rely on each time a sample is
     * added or served to update time statistics
     *
     * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
     * @created 26/07/2016
     */
    private class MonitorStatistics extends Thread
    {

        /**
         *
         */
        private PerformanceLogger performanceLogger;

        /**
         * Constructor
         *
         * @param performanceLogger the statistics object to update each second
         */
        private MonitorStatistics(PerformanceLogger performanceLogger)
        {
            this.performanceLogger = performanceLogger;
        }

        /**
         * @see java.lang.Thread#run()
         */
        @Override
        public void run()
        {
            while (true)
            {
                try
                {
                    Thread.sleep(1000);

                    if (PerformanceLoggerConfiguration.getInstance().getTrackRequests())
                        this.performanceLogger.updateTimedStats();
                }
                catch (Exception e)
                {
                    e.printStackTrace();
                }
            }
        }
    }
}
