package pt.digitalis.dif.utils.logging.performance;

/**
 * A piece of statistical information. The total number of requests and the average time it took to execute on a given
 * time slot
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 19/07/2016
 */
public class TimeStats extends AbstractTimeStats<Long>
{

    /** number of requests served until this time slot stat is created */
    private long numberOfRequests;

    /** RAM in use */
    private long ramUsed;

    /** the time spent on the sum of all executions */
    private long timeSpent;

    /**
     * Constructor
     */
    public TimeStats()
    {
        super();
        this.numberOfRequests = 0;
        this.timeSpent = 0;
    }

    /**
     * @see pt.digitalis.dif.utils.logging.performance.AbstractTimeStats#addSample(java.lang.Object)
     */
    @Override
    public void addSample(Long sample)
    {
        this.numberOfRequests++;
        this.timeSpent += sample;
        this.ramUsed =
                Math.round((Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) / (1024 * 1024));
    }

    /**
     * Inspector for the 'averageTime' attribute.
     *
     * @return the averageTime value
     */
    @Override
    public long getAverageTime()
    {
        if (Double.isNaN(numberOfRequests) || numberOfRequests == 0)
            return 0;
        else
            return Math.round(timeSpent / numberOfRequests);
    }

    /**
     * Inspector for the 'numberOfRequests' attribute.
     *
     * @return the numberOfRequests value
     */
    public long getNumberOfRequests()
    {
        return numberOfRequests;
    }

    /**
     * Inspector for the 'ramUsed' attribute.
     *
     * @return the ramUsed value
     */
    public long getRamUsed()
    {
        return ramUsed;
    }

    /**
     * Inspector for the 'timeSpent' attribute.
     *
     * @return the timeSpent value
     */
    public long getTimeSpent()
    {
        return timeSpent;
    }
}