/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.mail;

import pt.digitalis.dif.exception.UnsupportedAction;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.IObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.ObjectFormatter.Format;
import pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.IBeanAttributes;
import pt.digitalis.utils.common.NumericUtils;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.common.TimeUtils;
import pt.digitalis.utils.common.TimeUtils.Scale;
import pt.digitalis.utils.config.ConfigurationException;
import pt.digitalis.utils.pools.AbstractAction;
import pt.digitalis.utils.pools.ActionStatus;

import javax.mail.MessagingException;
import javax.mail.internet.AddressException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * A send mail action object
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Feb 18, 2009
 */
public class MailAction extends AbstractAction implements IBeanAttributes, IObjectFormatter
{

    /** The Serial version id for Serializable */
    static private final long serialVersionUID = 429211122232L;

    /**
     *
     */
    private static SimpleDateFormat dateFormatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

    /** The mail address BCC field */
    private String addressBCC;

    /** The mail address CC field */
    private String addressCC;

    /** The mail address FROM field */
    private String addressFrom;

    /** The mail address TO field */
    private String addressTo;

    /** The attachment list */
    private List<String> attachments;

    /** The mail body content */
    private String body;

    /**
     *
     */
    private Map<String, Object> businessKeys = new HashMap<String, Object>();

    /** The document repository entrys to attach */
    private List<DocumentRepositoryEntry> documentRepositoryEntries;

    /** The Mail sender Instance */
    private transient MailSender mailSenderInstance;

    /** The mail subject */
    private String subject;

    /** The mail MIME type */
    private MailType type;

    /**
     * @param date
     *
     * @return the formated date
     */
    private String dateToString(Date date)
    {
        return dateFormatter.format(date);
    }

    /**
     * @see pt.digitalis.utils.pools.AbstractAction#doExecute()
     */
    @Override
    protected AbstractAction doExecute() throws AddressException, MessagingException, ConfigurationException
    {
        boolean succcess = false;
        long currentAttempt = 0;
        String previousError = null;

        while (!succcess && currentAttempt < MailConfiguration.getInstance().getMaxAttemptsUntilFail())
        {
            currentAttempt++;
            try
            {
                succcess = this.getMailSenderInstance()
                        .sendEmail(getType(), getAddressFrom(), getAddressTo(), getAddressCC(), getAddressBCC(),
                                getSubject(), getBody(), this.getAttachments(), null,
                                this.getDocumentRepositoryEntries());

                if (currentAttempt > 1)
                    DIFLogger.getLogger().info("Send Mail succeeded on the attempt #" + currentAttempt +
                                               (StringUtils.isNotBlank(previousError) ? " (recovered from " +
                                                                                        previousError + ")" : "") +
                                               ".");
            }
            catch (AddressException e)
            {
                throw e;
            }
            catch (MessagingException e)
            {
                previousError = e.getMessage();

                if (currentAttempt >= MailConfiguration.getInstance().getMaxAttemptsUntilFail())
                {
                    // Give up...
                    DIFLogger.getLogger().error("Send Mail failed after " +
                                                MailConfiguration.getInstance().getMaxAttemptsUntilFail() +
                                                " attempts with " + e.getMessage() +
                                                ". Gave up and report error to sender!");

                    throw e;
                }
                else
                {
                    DIFLogger.getLogger()
                            .info("Send Mail failed with " + e.getMessage() + " (attempt #" + currentAttempt +
                                  "). Waiting " + TimeUtils.getTimePassed(
                                    MailConfiguration.getInstance().getMaxAttemptsWaitTimeBetweenAttempts(),
                                    Scale.MILI_SECONDS) + " to retry...");

                    // Wait 5 seconds and retry...
                    try
                    {
                        Thread.sleep(MailConfiguration.getInstance().getMaxAttemptsWaitTimeBetweenAttempts());
                    }
                    catch (InterruptedException e1)
                    {
                    }
                }
            }
        }

        /** {@link ILMailActionLogger} the mail action logger */
        for (IMailActionLogger mailActionLogger : DIFIoCRegistry.getRegistry()
                .getImplementations(IMailActionLogger.class))
        {
            mailActionLogger.logMailAction(this);
        }

        return this;
    }

    /**
     * Inspector for the 'adressBCC' attribute.
     *
     * @return the adressBCC value
     */
    public String getAddressBCC()
    {
        return addressBCC;
    }

    /**
     * Modifier for the 'adressBCC' attribute.
     *
     * @param adressBCC the new adressBCC value to set
     */
    public void setAddressBCC(String adressBCC)
    {
        this.addressBCC = adressBCC;
    }

    /**
     * Inspector for the 'adressCC' attribute.
     *
     * @return the adressCC value
     */
    public String getAddressCC()
    {
        return addressCC;
    }

    /**
     * Modifier for the 'adressCC' attribute.
     *
     * @param adressCC the new adressCC value to set
     */
    public void setAddressCC(String adressCC)
    {
        this.addressCC = adressCC;
    }

    /**
     * Inspector for the 'adressFrom' attribute.
     *
     * @return the adressFrom value
     */
    public String getAddressFrom()
    {
        return addressFrom;
    }

    /**
     * Modifier for the 'adressFrom' attribute.
     *
     * @param adressFrom the new adressFrom value to set
     */
    public void setAddressFrom(String adressFrom)
    {
        this.addressFrom = adressFrom;
    }

    /**
     * Inspector for the 'adressTo' attribute.
     *
     * @return the adressTo value
     */
    public String getAddressTo()
    {
        return addressTo;
    }

    /**
     * Modifier for the 'adressTo' attribute.
     *
     * @param adressTo the new adressTo value to set
     */
    public void setAddressTo(String adressTo)
    {
        this.addressTo = adressTo;
    }

    /**
     * Inspector for the 'attachments' attribute.
     *
     * @return the attachments value
     */
    public List<String> getAttachments()
    {
        return attachments;
    }

    /**
     * Modifier for the 'attachments' attribute.
     *
     * @param attachments the new attachments value to set
     */
    public void setAttachments(List<String> attachments)
    {
        this.attachments = attachments;
    }

    /**
     * @see pt.digitalis.utils.common.IBeanAttributes#getAttribute(java.lang.String)
     */
    @Override
    public Object getAttribute(String attributeName)
    {
        if ("addressBCC".equals(attributeName))
            return addressBCC;
        else if ("addressCC".equals(attributeName))
            return addressCC;
        else if ("addressFrom".equals(attributeName))
            return addressFrom;
        else if ("addressTo".equals(attributeName))
            return addressTo;
        else if ("body".equals(attributeName))
            return body;
        else if ("businessKeys".equals(attributeName))
            return businessKeys;
        else if ("enrollmentTime".equals(attributeName))
            return this.getEnrollmentTime();
        else if ("executionEndTime".equals(attributeName))
            return this.getExecutionEndTime();
        else if ("executionStartTime".equals(attributeName))
            return this.getExecutionStartTime();
        else if ("expirationTime".equals(attributeName))
            return this.getExpirationTime();
        else if ("id".equals(attributeName))
            return this.getId();
        else if ("reason".equals(attributeName))
            return this.getReason();
        else if ("status".equals(attributeName))
            return this.getStatus();
        else if ("subject".equals(attributeName))
            return this.getSubject();
        else if ("reasonException".equals(attributeName))
            return this.getReasonException();
        else if ("type".equals(attributeName))
            return type;
        else if ("attachments".equals(attributeName))
            return attachments;
        else
            return null;
    }

    /**
     * @see pt.digitalis.utils.common.IBeanPropertyInspector#getAttributeAsString(java.lang.String)
     */
    @Override
    public String getAttributeAsString(String attributeName)
    {
        if ("addressBCC".equals(attributeName))
            return addressBCC;
        else if ("addressCC".equals(attributeName))
            return addressCC;
        else if ("addressFrom".equals(attributeName))
            return addressFrom;
        else if ("addressTo".equals(attributeName))
            return addressTo;
        else if ("body".equals(attributeName))
            return body;
        else if ("businessKeys".equals(attributeName))
        {
            if (businessKeys == null)
                return null;
            else
            {
                List<String> list = new ArrayList<String>();
                for (Entry<String, Object> businessKeyEntry : businessKeys.entrySet())
                {
                    list.add(businessKeyEntry.getKey() + "=" +
                             StringUtils.nvl(StringUtils.toStringOrNull(businessKeyEntry.getValue()), "n/d"));
                }

                return CollectionUtils.listToCommaSeparatedString(list);
            }
        }
        else if ("enrollmentTime".equals(attributeName))
            return this.getEnrollmentTime() == null ? null : this.dateToString(this.getEnrollmentTime());
        else if ("executionEndTime".equals(attributeName))
            return this.getExecutionEndTime() == null ? null : this.dateToString(this.getExecutionEndTime());
        else if ("executionStartTime".equals(attributeName))
            return this.getExecutionStartTime() == null ? null : this.dateToString(this.getExecutionStartTime());
        else if ("expirationTime".equals(attributeName))
            return this.getExpirationTime() == null ? null : this.dateToString(this.getExpirationTime());
        else if ("id".equals(attributeName))
            return StringUtils.toStringOrNull(this.getId());
        else if ("reason".equals(attributeName))
            return this.getReason();
        else if ("status".equals(attributeName))
            return this.getStatus() == null ? null
                                            : StringUtils.camelCaseToString(this.getStatus().name().toLowerCase());
        else if ("subject".equals(attributeName))
            return subject;
        else if ("reasonException".equals(attributeName))
            return this.getReasonException() == null ? null : this.getReasonException().getMessage();
        else if ("type".equals(attributeName))
            return type == null ? null : type.name();
        else if ("attachments".equals(attributeName))
            return CollectionUtils.listToCommaSeparatedString(attachments);
        else
            return null;
    }

    /**
     * Inspector for the 'body' attribute.
     *
     * @return the body value
     */
    public String getBody()
    {
        return body;
    }

    /**
     * Modifier for the 'body' attribute.
     *
     * @param body the new body value to set
     */
    public void setBody(String body)
    {
        this.body = body;
    }

    /**
     * Inspector for the 'businessKeys' attribute.
     *
     * @return the businessKeys value
     */
    public Map<String, Object> getBusinessKeys()
    {
        return businessKeys;
    }

    /**
     * Modifier for the 'businessKeys' attribute.
     *
     * @param businessKeys the new businessKeys value to set
     */
    public void setBusinessKeys(Map<String, Object> businessKeys)
    {
        this.businessKeys = businessKeys;
    }

    /**
     * Inspector for the 'documentRepositoryEntries' attribute.
     *
     * @return the documentRepositoryEntries value
     */
    public List<DocumentRepositoryEntry> getDocumentRepositoryEntries()
    {
        return documentRepositoryEntries;
    }

    /**
     * Modifier for the 'documentRepositoryEntries' attribute.
     *
     * @param documentRepositoryEntries the new documentRepositoryEntries value to set
     */
    public void setDocumentRepositoryEntries(List<DocumentRepositoryEntry> documentRepositoryEntries)
    {
        this.documentRepositoryEntries = documentRepositoryEntries;
    }

    /**
     * Inspector for the 'mailSenderInstance' attribute.
     *
     * @return the mailSenderInstance value
     */
    public MailSender getMailSenderInstance()
    {
        return mailSenderInstance;
    }

    /**
     * Modifier for the 'mailSenderInstance' attribute.
     *
     * @param mailSenderInstance the new mailSenderInstance value to set
     */
    public void setMailSenderInstance(MailSender mailSenderInstance)
    {
        this.mailSenderInstance = mailSenderInstance;
    }

    /**
     * @see pt.digitalis.utils.pools.AbstractAction#getProperties()
     */
    @Override
    public Map<String, Object> getProperties()
    {
        Map<String, Object> props = new HashMap<String, Object>();
        props.put("From", getAddressFrom());
        props.put("To", getAddressTo());
        props.put("CC", getAddressCC());
        props.put("BCC", getAddressBCC());
        props.put("Subject", getSubject());
        props.put("Type", getType().toString());
        props.put("Body", getBody());

        return props;
    }

    /**
     * Inspector for the 'subject' attribute.
     *
     * @return the subject value
     */
    public String getSubject()
    {
        return subject;
    }

    /**
     * Modifier for the 'subject' attribute.
     *
     * @param subject the new subject value to set
     */
    public void setSubject(String subject)
    {
        this.subject = subject;
    }

    /**
     * Inspector for the 'type' attribute.
     *
     * @return the type value
     */
    public MailType getType()
    {
        return type;
    }

    /**
     * Modifier for the 'type' attribute.
     *
     * @param type the new type value to set
     */
    public void setType(MailType type)
    {
        this.type = type;
    }

    /**
     * @see pt.digitalis.utils.common.IBeanAttributes#setAttribute(java.lang.String, java.lang.Object)
     */
    @Override
    public void setAttribute(String attributeName, Object attributeValue)
    {
        if ("addressBCC".equals(attributeName))
            addressBCC = (String) attributeValue;
        else if ("addressCC".equals(attributeName))
            addressCC = (String) attributeValue;
        else if ("addressFrom".equals(attributeName))
            addressFrom = (String) attributeValue;
        else if ("addressTo".equals(attributeName))
            addressTo = (String) attributeValue;
        else if ("body".equals(attributeName))
            body = (String) attributeValue;
        else if ("businessKeys".equals(attributeName))
            businessKeys = (Map<String, Object>) attributeValue;
        else if ("enrollmentTime".equals(attributeName))
            this.setEnrollmentTime((Date) attributeValue);
        else if ("executionEndTime".equals(attributeName))
            this.setExecutionEndTime((Date) attributeValue);
        else if ("executionStartTime".equals(attributeName))
            this.setExecutionStartTime((Date) attributeValue);
        else if ("expirationTime".equals(attributeName))
            this.setExpirationTime((Date) attributeValue);
        else if ("id".equals(attributeName))
            this.setId((Long) attributeValue);
        else if ("reason".equals(attributeName))
            this.setReason((String) attributeValue);
        else if ("status".equals(attributeName))
            this.setStatus((ActionStatus) attributeValue);
        else if ("subject".equals(attributeName))
            this.setSubject((String) attributeValue);
        else if ("reasonException".equals(attributeName))
            this.setReasonException((Exception) attributeValue);
        else if ("type".equals(attributeName))
            type = (MailType) attributeValue;
        else if ("attachments".equals(attributeName))
            attachments = (List<String>) attributeValue;
    }

    /**
     * @see pt.digitalis.utils.common.IBeanAttributes#setAttributeFromString(java.lang.String, java.lang.String)
     */
    @Override
    public void setAttributeFromString(String attributeName, String attributeValue)
    {
        try
        {
            if ("addressBCC".equals(attributeName))
                addressBCC = attributeValue;
            else if ("addressCC".equals(attributeName))
                addressCC = attributeValue;
            else if ("addressFrom".equals(attributeName))
                addressFrom = attributeValue;
            else if ("addressTo".equals(attributeName))
                addressTo = attributeValue;
            else if ("body".equals(attributeName))
                body = attributeValue;
            else if ("businessKeys".equals(attributeName))
                throw new UnsupportedAction("Cannot set \"businessKeys\" from string");
            else if ("enrollmentTime".equals(attributeName))
                this.setEnrollmentTime(this.stringToDate(attributeValue));
            else if ("executionEndTime".equals(attributeName))
                this.setExecutionEndTime(this.stringToDate(attributeValue));
            else if ("executionStartTime".equals(attributeName))
                this.setExecutionStartTime(this.stringToDate(attributeValue));
            else if ("expirationTime".equals(attributeName))
                this.setExpirationTime(this.stringToDate(attributeValue));
            else if ("id".equals(attributeName))
                this.setId(NumericUtils.toLong(attributeValue));
            else if ("reason".equals(attributeName))
                this.setReason(attributeValue);
            else if ("status".equals(attributeName))
                this.setStatus(ActionStatus.valueOf(attributeValue));
            else if ("subject".equals(attributeName))
                subject = attributeValue;
            else if ("reasonException".equals(attributeName))
                throw new UnsupportedAction("Cannot set \"exception\" from string");
            else if ("type".equals(attributeName))
                type = MailType.valueOf(attributeValue);
            else if ("attachments".equals(attributeName))
                throw new UnsupportedAction("Cannot set \"attachments\" from string");
        }
        catch (ParseException e)
        {
            throw new RuntimeException("Erro seting \"" + attributeName + "\" from value \"" + attributeValue + "\"",
                    e);
        }
        catch (UnsupportedAction e)
        {
            throw new RuntimeException("Erro seting \"" + attributeName + "\" from value \"" + attributeValue + "\"",
                    e);
        }
    }

    /**
     * @see pt.digitalis.utils.common.IBeanAttributes#setNestedAttribute(java.lang.String, java.lang.Object)
     */
    @Override
    public void setNestedAttribute(String attributeName, Object attributeValue)
    {
        this.setAttribute(attributeName, attributeValue);
    }

    /**
     * @see pt.digitalis.utils.common.IBeanAttributes#setNestedAttributeFromString(java.lang.String,
     *         java.lang.String)
     */
    @Override
    public void setNestedAttributeFromString(String attributeName, String attributeValue)
    {
        this.setNestedAttributeFromString(attributeName, attributeValue);
    }

    /**
     * @param dateString
     *
     * @return the date
     *
     * @exception ParseException
     */
    private Date stringToDate(String dateString) throws ParseException
    {
        return dateFormatter.parse(dateString);
    }

    @Override
    public ObjectFormatter toObjectFormatter(Format format, List<Object> dumpedObjects)
    {
        ObjectFormatter formatter = new ObjectFormatter(format, dumpedObjects);
        formatter.addItem("ID", getId());
        formatter.addItem("Status", getStatus());
        formatter.addItem("Reason", getReason());
        formatter.addItem("Reason Exception", getReasonException());

        formatter.addItemIfNotNull("Mail From", addressFrom);
        formatter.addItemIfNotNull("Mail To", addressTo);
        formatter.addItemIfNotNull("Mail CC", addressCC);
        formatter.addItemIfNotNull("Mail BCC", addressBCC);
        formatter.addItemIfNotNull("Mail Body", body);
        formatter.addItemIfNotNull("Mail Subject", subject);
        formatter.addItemIfNotNull("Mail Type", type);
        formatter.addItem("Enrollment Time", getEnrollmentTime());
        formatter.addItemIfNotNull("Execution Start Time", getExecutionStartTime());
        formatter.addItemIfNotNull("Execution End Time", getExecutionEndTime());
        formatter.addItemIfNotNull("Expiration Time", getExpirationTime());
        formatter.addItemIfNotNull("Waiting Threads", getWaitingThreads());

        return formatter;
    }

    @Override
    public String toString()
    {
        return toObjectFormatter(Format.TEXT, null).getFormatedObject();
    }
}