/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.system;

import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.common.collections.IndexedHashMap;

import java.io.IOException;
import java.io.InputStream;
import java.lang.management.ManagementFactory;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.Enumeration;
import java.util.Map;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

/**
 * The Class SystemUtils.
 *
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @created Sep 11, 2012
 */
public class SystemUtils
{

    /** The inet address cache. */
    static private InetAddress inetAddressCache = null;

    /** the JBoss version tag. */
    static private String jbossTagCache;

    /**
     * Gets the inet address.
     *
     * @return the inet address
     *
     * @exception SocketException
     * @exception UnknownHostException
     */
    private static InetAddress getInetAddress() throws SocketException, UnknownHostException
    {
        if (inetAddressCache == null)
        {
            InetAddress ip = null;
            Enumeration<NetworkInterface> nInterfaces = NetworkInterface.getNetworkInterfaces();

            while (ip == null && nInterfaces.hasMoreElements())
            {

                NetworkInterface networkInterface = nInterfaces.nextElement();
                // Ignore VMWare interfaces (only possible by name!)
                if (!networkInterface.getName().startsWith("vmnet"))
                {
                    boolean isUp = false;

                    try
                    {
                        isUp = networkInterface.isUp();
                    }
                    catch (Exception e)
                    {
                        // When this fails we assume it is up and will try the reachable bellow
                        isUp = true;
                    }

                    if (isUp)
                    {
                        Enumeration<InetAddress> inetAddresses = networkInterface.getInetAddresses();

                        while (ip == null && inetAddresses.hasMoreElements())
                        {
                            InetAddress inetAddress = inetAddresses.nextElement();

                            try
                            {
                                if (inetAddress.isReachable(500) && !inetAddress.isLoopbackAddress() &&
                                    inetAddress instanceof Inet4Address)
                                {
                                    DIFLogger.getLogger().debug("Determined default network interface: " +
                                                                networkInterface.getName());

                                    if (isUp)
                                    {
                                        ip = inetAddress;
                                        break;
                                    }
                                }
                            }
                            catch (IOException e)
                            {
                                DIFLogger.getLogger().debug("Machine IP detection: Comunication test with " +
                                                            inetAddress.getHostAddress() + " of interface " +
                                                            networkInterface.getName() + " failed with: " +
                                                            e.getMessage());

                                if (DIFLogger.getLogger().isDebugEnabled())
                                    e.printStackTrace();
                            }
                        }
                    }
                }
            }

            if (ip == null)
            {
                // If this was reached must fail over to the default host
                ip = InetAddress.getLocalHost();
                DIFLogger.getLogger().debug("Determined default network interface: localhost");
            }

            inetAddressCache = ip;
        }

        return inetAddressCache;
    }

    /**
     * Gets the j boss version.
     *
     * @return the JBoss version string
     */
    public static String getJBossVersion()
    {
        if (jbossTagCache == null)
        {
            Manifest manifest = null;
            Enumeration<URL> resEnum;
            try
            {
                resEnum = Thread.currentThread().getContextClassLoader().getResources(JarFile.MANIFEST_NAME);
                while (resEnum.hasMoreElements())
                {
                    URL url = resEnum.nextElement();
                    InputStream is = url.openStream();
                    if (is != null)
                    {
                        Manifest manifestTemp = new Manifest(is);
                        Attributes mainAttribs = manifestTemp.getMainAttributes();
                        String title = mainAttribs.getValue(Attributes.Name.IMPLEMENTATION_TITLE);
                        if (title != null && title.toUpperCase().contains("JBOSS"))
                        {
                            manifest = manifestTemp;
                            break;
                        }
                    }
                }
            }
            catch (Exception e)
            {
                // Silently ignore wrong manifests on classpath?
            }

            if (manifest == null)
                jbossTagCache = "Unknown";
            else
            {
                jbossTagCache = manifest.getMainAttributes().getValue(Attributes.Name.IMPLEMENTATION_TITLE) + " v." +
                                manifest.getMainAttributes().getValue(Attributes.Name.IMPLEMENTATION_VERSION);
            }
        }

        return jbossTagCache;
    }

    /**
     * Gets the server IP.
     *
     * @return the server IP
     *
     * @exception UnknownHostException
     * @exception SocketException
     */
    public static String getServerIP() throws UnknownHostException, SocketException
    {
        // If this was reached must fail over to the default host
        InetAddress ip = getInetAddress();
        String ipAddress = ip.getHostAddress();

        DIFLogger.getLogger().debug("Determined server IP address: " + ipAddress);

        return ipAddress;
    }

    /**
     * Gets the server mac address.
     *
     * @return the server mac address
     *
     * @exception UnknownHostException
     * @exception SocketException
     */
    public static String getServerMacAddress() throws UnknownHostException, SocketException
    {
        InetAddress ip = getInetAddress();

        NetworkInterface network = NetworkInterface.getByInetAddress(ip);
        byte[] mac = network.getHardwareAddress();

        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < mac.length; i++)
        {
            sb.append(String.format("%02X%s", mac[i], (i < mac.length - 1) ? "-" : ""));
        }
        String macAddress = sb.toString();

        DIFLogger.getLogger().debug("Determined server MacAddress: " + macAddress);

        return macAddress;
    }

    /**
     * Gets the system properties.
     *
     * @return the system properties
     */
    public static Map<String, Object> getSystemProperties()
    {
        Map<String, Object> props = new IndexedHashMap<String, Object>();
        try
        {
            props.put("COMPUTERNAME", InetAddress.getLocalHost().getHostName());
        }
        catch (UnknownHostException e)
        {
            props.put("COMPUTERNAME", System.getenv().get("COMPUTERNAME"));
        }

        String OS = System.getenv().get("OS");
        if (OS == null || "".equals(OS))
        {
            OS = System.getProperty("os.name") + " " + System.getProperty("os.version") + "," +
                 System.getProperty("os.arch");
        }

        props.put("OS", OS);
        props.put("USERDOMAIN", System.getenv().get("USERDOMAIN"));
        props.put("USERNAME", System.getenv().get("USERNAME"));

        if (System.getenv().get("PROCESSOR_ARCHITECTURE") != null)
        {
            props.put("PROCESSOR_ARCHITECTURE", System.getenv().get("PROCESSOR_ARCHITECTURE"));
        }
        else
        {
            props.put("PROCESSOR_ARCHITECTURE", ManagementFactory.getOperatingSystemMXBean().getArch());
        }

        props.put("PROCESSOR_IDENTIFIER", System.getenv().get("PROCESSOR_IDENTIFIER"));

        props.put("javaVersion",
                System.getProperty("java.vendor") + " (v." + System.getProperty("java.version") + ")" + " - VM: " +
                System.getProperty("java.vm.name") + " " + System.getProperty("java.vm.version") + "," +
                System.getProperty("java.vm.vendor"));
        props.put("jBossVersion", getJBossVersion());

        props.put("totalMemory", Runtime.getRuntime().totalMemory() / 1024 / 1024 + " Mb");
        props.put("maxMemory", Runtime.getRuntime().maxMemory() / 1024 / 1024 + " Mb");
        props.put("freeMemory", Runtime.getRuntime().freeMemory() / 1024 / 1024 + " Mb");
        props.put("processorNumber", Runtime.getRuntime().availableProcessors());

        return props;
    }

    /**
     * @param args
     *
     * @exception UnknownHostException
     * @exception SocketException
     */
    public static void main(String[] args) throws SocketException, UnknownHostException
    {

        ;
        System.out.println(getServerIP());
    }
}
