/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.tests.authorization;

import junit.framework.TestCase;
import pt.digitalis.dif.controller.security.managers.IAuthorizationManager;
import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.objects.ACLEntry;
import pt.digitalis.dif.controller.security.objects.DIFGroupImpl;
import pt.digitalis.dif.controller.security.objects.DIFUserImpl;
import pt.digitalis.dif.controller.security.objects.IDIFGroup;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.dem.DEMRegistryImpl;
import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.exception.security.AuthorizationManagerException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.ioc.DIFDefaultModulesConfiguration;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.startup.DIFInitializer;
import pt.digitalis.utils.config.ConfigurationException;
import pt.digitalis.utils.config.ConfigurationsProvider;
import pt.digitalis.utils.config.IConfigurations;

import java.util.List;
import java.util.Properties;

/**
 * Defines unit tests for the authorization manager.
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @created 30/11/2017
 */
public abstract class AbstractAuthorizationManagerImplTest extends TestCase
{

    /** Another stage for access. */
    final static private String ANOTHER_STAGE_ID = "anotherStageID";

    /** An application for access. */
    final static private String APPLICATION_ID = "applicationID";

    /** A bad group id as a constant. */
    final static private String BAD_GROUP_ID = "badGroupID";

    /** A bad user id as a constant. */
    final static private String BAD_USER_ID = "badUserID";

    /** The ID of a "child group". */
    static final private String CHILD_GROUP_ID = "childGroupId";

    /** Another stage for default access tests. */
    final static private String DEFAULT_STAGE_ID = "defaultStageID";

    /** The ID of a "grandparent group". */
    static final private String GRANDPARENT_GROUP_ID = "grandParentGroupId";

    /** The ID of a "great-grandparent group". */
    static final private String GREAT_GRANDPARENT_GROUP_ID = "greatGrandParentGroupId";

    /** The group id as a constant. */
    final static private String GROUP_ID = "groupID";

    /** The group id for default tests as a constant. */
    final static private String GROUP_ID_FOR_DEFAULT_TEST = "groupIdForDefaultTest";

    /** Id of an inaccessible stage. */
    final static private String INACCESSIBLE_STAGE_ID = "inaccessibleStageID";

    /** Another stage for default access tests. */
    final static private String MOCKUP_STAGE_FOR_DEFAULT_ACCESS_TEST_STAGE_ID = "MockupStageForDefaultAccessTest";

    /** A stage for access. */
    final static private String MOCKUP_STAGE_ID = "mockupstage";

    /** Non existant application */
    final static private String NON_EXISTANT_APPLICATION = "nonExistantApplication";

    /** Non existant service */
    final static private String NON_EXISTANT_SERVICE = "nonExistantService";

    /** Id of an nonexistent stage. */
    final static private String NONEXISTENT_STAGE_ID = "nonexistantStageID";

    /** The ID of a "parent group". */
    static final private String PARENT_GROUP_ID = "parentGroupId";

    /** The profile group id as a constant. */
    final static private String PROFILE_ID = "profileGroupID";

    /** A provider for access. */
    final static private String PROVIDER_ID = "providerID";

    /** A stage for public access. */
    final static private String PUBLIC_STAGE_ID = "publicStageID";

    /** A service for access. */
    final static private String SERVICE_ID = "serviceID";

    /** A stage for access. */
    final static private String STAGE_ID = "stageID";

    /** The user id as a constant. */
    final static private String USER_ID = "userID";

    /** The user id for default tests as a constant. */
    final static private String USER_ID_FOR_DEFAULT_TEST = "userIdForDefaultTest";

    /** A stage instance. */
    static private IStage stageInstance = null;

    /** The authentication manager. */
    private IAuthorizationManager authManager = null;

    /** The DEM manager. */
    private IDEMManager demManager = null;

    /** A group for test purposes. */
    private IDIFGroup group_id = null;

    /** The authentication manager. */
    private IIdentityManager identManager = null;

    /** A group for test purposes. To be used has a profile. */
    private IDIFGroup profile_id = null;

    /** A user for test purposes. */
    private IDIFUser user_id = null;

    /**
     * Gets the implementation type.
     *
     * @return the implementation type
     */
    protected abstract String getImplementationType();

    /**
     * Removes the group hierarchy on the identity manager for the inherited grant tests.
     *
     * @exception IdentityManagerException if the identity manager can't be accessed
     */
    private void removeGroupHierarchy() throws IdentityManagerException
    {
        // Great-grandparent group
        if (identManager.groupExists(GREAT_GRANDPARENT_GROUP_ID))
        {
            identManager.removeGroup(GREAT_GRANDPARENT_GROUP_ID);
        }

        // Grandparent group
        if (identManager.groupExists(GRANDPARENT_GROUP_ID))
        {
            identManager.removeGroup(GRANDPARENT_GROUP_ID);
        }

        // Parent group
        if (identManager.groupExists(PARENT_GROUP_ID))
        {
            identManager.removeGroup(PARENT_GROUP_ID);
        }

        // Child group
        if (identManager.groupExists(CHILD_GROUP_ID))
        {
            identManager.removeGroup(CHILD_GROUP_ID);
        }
    }

    /**
     * Sets a group hierarchy on the identity manager for the inherited grant tests.
     *
     * @exception IdentityManagerException if the identity manager can't be accessed
     */
    private void setGroupHierarchy() throws IdentityManagerException
    {

        // Great-grandparent group
        if (!identManager.groupExists(GREAT_GRANDPARENT_GROUP_ID))
        {
            IDIFGroup greatGrandParentGroup = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);
            greatGrandParentGroup.setID(GREAT_GRANDPARENT_GROUP_ID);
            greatGrandParentGroup.setName(GREAT_GRANDPARENT_GROUP_ID);
            identManager.addGroup(greatGrandParentGroup);
        }

        // Grandparent group
        if (!identManager.groupExists(GRANDPARENT_GROUP_ID))
        {
            IDIFGroup grandParentGroup = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);
            grandParentGroup.setID(GRANDPARENT_GROUP_ID);
            grandParentGroup.setName(GRANDPARENT_GROUP_ID);
            grandParentGroup.setParentGroupID(GREAT_GRANDPARENT_GROUP_ID);
            identManager.addGroup(grandParentGroup);
        }

        // Parent group
        if (!identManager.groupExists(PARENT_GROUP_ID))
        {
            IDIFGroup parentGroup = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);
            parentGroup.setID(PARENT_GROUP_ID);
            parentGroup.setName(PARENT_GROUP_ID);
            parentGroup.setParentGroupID(GRANDPARENT_GROUP_ID);
            identManager.addGroup(parentGroup);
        }

        // Child group
        if (!identManager.groupExists(CHILD_GROUP_ID))
        {
            IDIFGroup childGroup = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);
            childGroup.setID(CHILD_GROUP_ID);
            childGroup.setName(CHILD_GROUP_ID);
            childGroup.setParentGroupID(PARENT_GROUP_ID);
            identManager.addGroup(childGroup);
        }
    }

    /**
     * Sets the test environment.
     *
     * @exception ConfigurationException
     */
    @Override
    protected void setUp() throws ConfigurationException
    {

        Properties attributeConfigs = new Properties();
        attributeConfigs.put(IAuthorizationManager.class.getCanonicalName(), getImplementationType());

        IConfigurations configAPI = ConfigurationsProvider.getConfigurations();
        configAPI.writeConfiguration(DIFDefaultModulesConfiguration.DEFAULT_MODULE_CONFIG_ID,
                DIFDefaultModulesConfiguration.DEFAULT_MODULE_CONFIG_SECTION_ID, attributeConfigs);

        DIFInitializer.initialize(true, false);
        assertTrue(DIFInitializer.isInitialized());

        identManager = DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class);
        authManager = DIFIoCRegistry.getRegistry().getImplementation(IAuthorizationManager.class);
        demManager = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class);

        user_id = DIFIoCRegistry.getRegistry().getImplementation(IDIFUser.class);
        user_id.setID(USER_ID);
        user_id.setName(USER_ID);
        user_id.setNick(USER_ID);

        group_id = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);
        group_id.setID(GROUP_ID);

        profile_id = DIFIoCRegistry.getRegistry().getImplementation(IDIFGroup.class);
        profile_id.setID(PROFILE_ID);

        try
        {

            if (!identManager.userExists(USER_ID))
            {
                identManager.addUser(user_id);
                assertTrue(identManager.userExists(USER_ID));
            }

            if (!identManager.groupExists(GROUP_ID))
            {
                identManager.addGroup(group_id);
                assertTrue(identManager.groupExists(GROUP_ID));
            }

            if (!identManager.groupExists(PROFILE_ID))
            {
                identManager.addGroup(profile_id);
                assertTrue(identManager.groupExists(PROFILE_ID));
            }
        }
        catch (IdentityManagerException identityManagerException)
        {
            identityManagerException.printStackTrace();
            assertTrue("Test case could not be intialized and thus the unit tests can't proceed!", false);
        }

        stageInstance = DEMRegistryImpl.getRegistry().getStage("mockupstage");
    }

    /**
     * Delete test settings.
     *
     * @exception ConfigurationException
     */
    @Override
    public void tearDown() throws ConfigurationException
    {
        try
        {

            if (identManager.userExists(USER_ID))
            {
                identManager.removeUser(USER_ID);
                assertFalse(identManager.userExists(USER_ID));
            }

            if (identManager.groupExists(GROUP_ID))
            {
                identManager.removeGroup(GROUP_ID);
                assertFalse(identManager.groupExists(GROUP_ID));
            }

            if (identManager.groupExists(PROFILE_ID))
            {
                identManager.removeGroup(PROFILE_ID);
                assertFalse(identManager.groupExists(PROFILE_ID));
            }
        }
        catch (IdentityManagerException identityManagerException)
        {
            identityManagerException.printStackTrace();
            assertTrue("Test case could not be properly shutdown and thus the unit tests can't proceed!", false);
        }
        IConfigurations configAPI = ConfigurationsProvider.getConfigurations();
        configAPI.removeConfiguration(DIFDefaultModulesConfiguration.DEFAULT_MODULE_CONFIG_ID,
                DIFDefaultModulesConfiguration.DEFAULT_MODULE_CONFIG_SECTION_ID);
    }

    /**
     * Tests the ACLEntry object.
     *
     * @exception AuthorizationManagerException if the authorization manager can't be accessed
     * @exception IdentityManagerException
     */
    public void testACLEntry() throws AuthorizationManagerException, IdentityManagerException
    {
        ACLEntry entry1, entry2;

        entry1 = new ACLEntry();
        assertNotNull(entry1);

        entry1.setUserID(USER_ID);
        assertTrue(entry1.isUserACL());
        assertFalse(entry1.isGroupACL());
        assertFalse(entry1.isPublicAccess());

        entry1.setEntityType(Entity.STAGE);
        entry1.setEntityID(STAGE_ID);

        assertEquals(entry1.getEntityType(), Entity.STAGE);
        assertEquals(entry1.getEntityID(), STAGE_ID.toLowerCase());

        entry2 = new ACLEntry();
        assertNotNull(entry2);
        assertFalse(entry1.equals(entry2));

        entry2.setGroupID(GROUP_ID);
        assertFalse(entry2.isUserACL());
        assertTrue(entry2.isGroupACL());
        assertFalse(entry2.isPublicAccess());

        entry2.setEntityType(Entity.STAGE);
        entry2.setEntityID(STAGE_ID);

        assertFalse(entry1.equals(entry2));
        entry2.setUserID(USER_ID);
        assertTrue(entry1.equals(entry2));

        entry2.setPublicAccess();
        assertFalse(entry2.isUserACL());
        assertFalse(entry2.isGroupACL());
        assertTrue(entry2.isPublicAccess());
        assertFalse("".equals(entry2.toString()));

        // Create an empty (invalid) ACL
        entry1 = new ACLEntry();
        // Add empty ACL will not add nothing and return false
        assertFalse(authManager.addACLEntry(entry1));
        // Revoke empty ACL will not add nothing and return false
        assertFalse(authManager.revokeACLEntry(entry1));
        // Set the ACL for grant to user
        entry1.setUserID(USER_ID);
        entry1.setEntityType(Entity.STAGE);
        entry1.setEntityID(STAGE_ID);
        // No previous access
        assertFalse(authManager.hasAccessUser(user_id, Entity.STAGE, STAGE_ID));

        int previousAccessCount = authManager.findACLEntriesByStage(STAGE_ID).size();

        // Grant it...
        assertTrue(authManager.addACLEntry(entry1));
        // Prove that it gained access
        assertTrue(authManager.findACLEntriesByStage(STAGE_ID).get(0).getUserID().equals(USER_ID));
        // Still no access though, has the authorization manager determines that the stage does not exist
        assertFalse(authManager.hasAccessUser(user_id, Entity.STAGE, STAGE_ID));
        // Remove the grant...
        assertTrue(authManager.revokeACLEntry(entry1));
        // No more access
        assertEquals(authManager.findACLEntriesByStage(STAGE_ID).size(), previousAccessCount);

        // Set the ACL for grant to group
        entry1.setGroupID(GROUP_ID);
        entry1.setEntityType(Entity.STAGE);
        entry1.setEntityID(STAGE_ID);
        // No previous access
        assertFalse(authManager.hasAccessGroup(group_id, Entity.STAGE, STAGE_ID));
        // Grant it...
        assertTrue(authManager.addACLEntry(entry1));
        // Prove that it gained access
        assertTrue(authManager.findACLEntriesByStage(STAGE_ID).get(0).getGroupID().equals(GROUP_ID));
        // Still no access though, has the authorization manager determines that the stage does not exist
        assertFalse(authManager.hasAccessGroup(group_id, Entity.STAGE, STAGE_ID));
        // Remove the grant...
        authManager.revokeACLEntry(entry1);
        // No more access
        assertEquals(authManager.findACLEntriesByStage(STAGE_ID).size(), 0);

        // Force addition of invalid group (ie, non-existent group on the id manager)
        try
        {
            identManager.removeGroup(GROUP_ID);
            assertFalse(identManager.groupExists(GROUP_ID));
            assertTrue(authManager.addACLEntry(entry1));
            identManager.addGroup(group_id);
            assertTrue(identManager.groupExists(GROUP_ID));
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

    /**
     * Tests DEM access entity inheritance.
     *
     * @exception AuthorizationManagerException
     * @exception IdentityManagerException
     * @exception ConfigurationException
     */
    public void testDEMInheritanceAccess()
            throws AuthorizationManagerException, IdentityManagerException, ConfigurationException
    {
        DIFInitializer.initialize(true, false);
        assertTrue(DIFInitializer.isInitialized());

        IDEMManager dem = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class);
        IAuthorizationManager auth = DIFIoCRegistry.getRegistry().getImplementation(IAuthorizationManager.class);
        IIdentityManager ident = DIFIoCRegistry.getRegistry().getImplementation(IIdentityManager.class);

        System.out.println(dem.getService("ServiceGroup1"));
        System.out.println(ident.getGroup("secGroup1"));
        System.out.println(auth.findACLEntriesByService("ServiceGroup1"));

        IDIFUser user1 = identManager.getUser(USER_ID);
        assertNotNull(user1);

        IDIFGroup secGroup1 = identManager.getGroup("secGroup1");
        assertNotNull(secGroup1);

        IDIFGroup secGroup2 = identManager.getGroup("secGroup2");
        assertNotNull(secGroup2);

        assertNotNull(demManager);
        assertNotNull(authManager);

        assertNotNull(demManager.getStage("stagepubservicepub"));
        assertTrue(authManager.hasAccessPublic(Entity.STAGE, "stagepubservicepub"));
        assertTrue(authManager.hasAccessUser(user1, Entity.STAGE, "stagepubservicepub"));
        assertTrue(authManager.hasAccessGroup(secGroup1, Entity.STAGE, "stagepubservicepub"));
        assertTrue(authManager.hasAccessGroup(secGroup2, Entity.STAGE, "stagepubservicepub"));

        assertNotNull(demManager.getStage("stagegroup1servicepub"));
        assertFalse(authManager.hasAccessPublic(Entity.STAGE, "stagegroup1servicepub"));
        assertFalse(authManager.hasAccessUser(user1, Entity.STAGE, "stagegroup1servicepub"));
        assertTrue(authManager.hasAccessGroup(secGroup1, Entity.STAGE, "stagegroup1servicepub"));
        assertFalse(authManager.hasAccessGroup(secGroup2, Entity.STAGE, "stagegroup1servicepub"));

        assertNotNull(demManager.getStage("stagepubservicegroup1"));
        assertFalse(authManager.hasAccessPublic(Entity.STAGE, "stagepubservicegroup1"));
        assertFalse(authManager.hasAccessUser(user1, Entity.STAGE, "stagepubservicegroup1"));
        assertTrue(authManager.hasAccessGroup(secGroup1, Entity.STAGE, "stagepubservicegroup1"));
        assertFalse(authManager.hasAccessGroup(secGroup2, Entity.STAGE, "stagepubservicegroup1"));

        assertNotNull(demManager.getStage("stagegroup2servicegroup1"));
        assertFalse(authManager.hasAccessPublic(Entity.STAGE, "stagegroup2servicegroup1"));
        assertFalse(authManager.hasAccessUser(user1, Entity.STAGE, "stagegroup2servicegroup1"));
        assertFalse(authManager.hasAccessGroup(secGroup1, Entity.STAGE, "stagegroup2servicegroup1"));
        assertTrue(authManager.hasAccessGroup(secGroup2, Entity.STAGE, "stagegroup2servicegroup1"));

        assertNotNull(demManager.getStage("stagenoneservicegroup1"));
        assertFalse(authManager.hasAccessPublic(Entity.STAGE, "stagenoneservicegroup1"));
        assertFalse(authManager.hasAccessUser(user1, Entity.STAGE, "stagenoneservicegroup1"));
        assertFalse(authManager.hasAccessGroup(secGroup1, Entity.STAGE, "stagenoneservicegroup1"));
        assertFalse(authManager.hasAccessGroup(secGroup2, Entity.STAGE, "stagenoneservicegroup1"));

        assertNotNull(demManager.getStage("stagegroup1servicenone"));
        assertFalse(authManager.hasAccessPublic(Entity.STAGE, "stagegroup1servicenone"));
        assertFalse(authManager.hasAccessUser(user1, Entity.STAGE, "stagegroup1servicenone"));
        assertTrue(authManager.hasAccessGroup(secGroup1, Entity.STAGE, "stagegroup1servicenone"));
        assertFalse(authManager.hasAccessGroup(secGroup2, Entity.STAGE, "stagegroup1servicenone"));

        identManager.addUserToGroup(user1.getID(), secGroup2.getID());

        assertTrue(authManager.findACLEntriesByUserInherited(user1.getID()).get(0).getEntityID()
                .equals("stagegroup2servicegroup1"));
        identManager.removeUserFromGroup(user1.getID(), secGroup2.getID());
        secGroup2.setParentGroupID(secGroup1.getID());
        identManager.updateGroup(secGroup2);
        identManager.addUserToGroup(user1.getID(), secGroup1.getID());
        assertTrue(!authManager.findACLEntriesByUserInherited(user1.getID()).isEmpty());
    }

    /**
     * Tests the default access operations.
     *
     * @exception AuthorizationManagerException if the authorization manager can't be accessed
     * @exception IdentityManagerException      if the identity manager can't be accessed
     */
    public void testDefaultAccess() throws AuthorizationManagerException, IdentityManagerException
    {

        // Default access for groups
        assertTrue(authManager.grantDefaultAccessToGroup(GROUP_ID, Entity.STAGE, STAGE_ID));
        assertTrue(authManager.findACLEntriesByStage(STAGE_ID).get(0).getGroupID().equals(GROUP_ID));
        // No access reported since the stage is fictitious
        assertFalse(authManager.hasAccessGroup(group_id, Entity.STAGE, STAGE_ID));
        assertTrue(authManager.revokeAccessFromGroup(GROUP_ID, Entity.STAGE, STAGE_ID));
        assertTrue(authManager.grantAccessToGroup(GROUP_ID, Entity.STAGE, STAGE_ID));
        assertTrue(authManager.findACLEntriesByStage(STAGE_ID).get(0).getGroupID().equals(GROUP_ID));
        assertTrue(authManager.grantAccessToGroup(GROUP_ID, Entity.STAGE, STAGE_ID));
        assertTrue(authManager.findACLEntriesByStage(STAGE_ID).get(0).getGroupID().equals(GROUP_ID));

        // Default access for users
        assertTrue(authManager.grantDefaultAccessToUser(USER_ID, Entity.STAGE, STAGE_ID));
        assertTrue(authManager.findACLEntriesByStage(STAGE_ID).get(0).getUserID().equals(USER_ID));
        // No access reported since the stage is ficticious
        assertFalse(authManager.hasAccessUser(user_id, Entity.STAGE, STAGE_ID));
        assertTrue(authManager.revokeAccessFromUser(USER_ID, Entity.STAGE, STAGE_ID));

        // Default public access
        assertTrue(authManager.grantDefaultPublicAccess(Entity.STAGE, MOCKUP_STAGE_ID));
        assertTrue(authManager.hasAccessPublic(Entity.STAGE, MOCKUP_STAGE_ID));

        // *** Ensure that a default entry can't be removed ***
        // * Users *

        // Create new default user
        IDIFUser defaultUser = new DIFUserImpl();
        defaultUser.setDefault(true);
        defaultUser.setID(USER_ID_FOR_DEFAULT_TEST);
        defaultUser.setEnabled(true);

        // Add user to the identity manager
        identManager.addUser(defaultUser);

        // Grant user access
        authManager.grantDefaultAccessToUser(USER_ID_FOR_DEFAULT_TEST, Entity.STAGE, DEFAULT_STAGE_ID);
        assertTrue(authManager.findACLEntriesByUser(USER_ID_FOR_DEFAULT_TEST).get(0).getEntityID()
                .equals(DEFAULT_STAGE_ID.toLowerCase()));

        // Revoke user access
        authManager.revokeAccessFromUser(USER_ID_FOR_DEFAULT_TEST, Entity.STAGE, DEFAULT_STAGE_ID);
        // If the returned user ACL contains the entry, it wasn't removed
        assertTrue(authManager.findACLEntriesByUser(USER_ID_FOR_DEFAULT_TEST).get(0).getEntityID()
                .equals(DEFAULT_STAGE_ID.toLowerCase()));

        // * Groups *

        // Create new default group
        IDIFGroup defaultGroup = new DIFGroupImpl();
        defaultGroup.setDefault(true);
        defaultGroup.setID(GROUP_ID_FOR_DEFAULT_TEST);

        // Add group to the identity manager
        identManager.addGroup(defaultGroup);

        // Grant default group access
        authManager.grantDefaultAccessToGroup(GROUP_ID_FOR_DEFAULT_TEST, Entity.STAGE,
                MOCKUP_STAGE_FOR_DEFAULT_ACCESS_TEST_STAGE_ID);

        // Validate access for group
        assertEquals(1, authManager.findACLEntriesByGroup(GROUP_ID_FOR_DEFAULT_TEST).size());
        assertTrue(authManager.findACLEntriesByGroup(GROUP_ID_FOR_DEFAULT_TEST).get(0).getEntityID()
                .equals(MOCKUP_STAGE_FOR_DEFAULT_ACCESS_TEST_STAGE_ID.toLowerCase()));

        assertTrue(
                authManager.hasAccessGroup(defaultGroup, Entity.STAGE, MOCKUP_STAGE_FOR_DEFAULT_ACCESS_TEST_STAGE_ID));

        // Grant the same access as non default
        authManager.grantAccessToGroup(GROUP_ID_FOR_DEFAULT_TEST, Entity.STAGE,
                MOCKUP_STAGE_FOR_DEFAULT_ACCESS_TEST_STAGE_ID);

        // Confirm that still has access
        assertTrue(
                authManager.hasAccessGroup(defaultGroup, Entity.STAGE, MOCKUP_STAGE_FOR_DEFAULT_ACCESS_TEST_STAGE_ID));

        // Confirm that has only one ACL
        assertEquals(authManager.findACLEntriesByGroup(GROUP_ID_FOR_DEFAULT_TEST).size(), 1);

        // Confirm that the default is the access that remained
        assertTrue(authManager.findACLEntriesByGroup(GROUP_ID_FOR_DEFAULT_TEST).get(0).isDefault());

        // Revoke group access
        authManager.revokeAccessFromGroup(GROUP_ID_FOR_DEFAULT_TEST, Entity.STAGE,
                MOCKUP_STAGE_FOR_DEFAULT_ACCESS_TEST_STAGE_ID);

        // Confirm that the group no longer has access
        assertFalse(
                authManager.hasAccessGroup(defaultGroup, Entity.STAGE, MOCKUP_STAGE_FOR_DEFAULT_ACCESS_TEST_STAGE_ID));
        // If the returned user ACL contains the entry, it wasn't removed
        assertTrue(authManager.findACLEntriesByGroup(GROUP_ID_FOR_DEFAULT_TEST).get(0).getEntityID()
                .equals(MOCKUP_STAGE_FOR_DEFAULT_ACCESS_TEST_STAGE_ID.toLowerCase()));
        assertTrue(authManager.findACLEntriesByGroup(GROUP_ID_FOR_DEFAULT_TEST).get(0).isDefault());
        assertFalse(authManager.findACLEntriesByGroup(GROUP_ID_FOR_DEFAULT_TEST).get(0).isEnabled());

        // * Public access *

        // Grant default public access
        authManager.grantDefaultPublicAccess(Entity.STAGE, DEFAULT_STAGE_ID);
        // If the returned user ACL contains the entry, it was set
        List<ACLEntry> publicEntries = authManager.findPublicACLEntries();

        // Assert control flag (assume entry was removed on initialization)
        boolean publicEntryPresent = true;

        // Check entries...
        for (ACLEntry publicEntry : publicEntries)
        {
            // If the entry was found and belongs to the stage change control flags status and exit
            if (publicEntry.getEntityID() != null && publicEntry.getEntityID().equals(DEFAULT_STAGE_ID))
            {
                publicEntryPresent = true;
                break;
            }
        }

        assertTrue(publicEntryPresent);

        // Revoke public access
        authManager.revokeAccessFromPublic(Entity.STAGE, DEFAULT_STAGE_ID);
        // If the returned user ACL contains the entry, it wasn't removed
        publicEntries = authManager.findPublicACLEntries();

        // Assert control flag (assume entry was removed on initialization)
        boolean publicEntryRemoved = true;
        boolean publicEntryDisabled = false;

        // Check entries...
        for (ACLEntry publicEntry : publicEntries)
        {
            // If the entry was found and belongs to the stage change control flags status and exit
            if (publicEntry.getEntityID() != null && publicEntry.getEntityID().equals(DEFAULT_STAGE_ID.toLowerCase()) &&
                !publicEntry.isEnabled())
            {
                publicEntryRemoved = false;
                publicEntryDisabled = true;
                break;
            }
        }

        assertFalse(publicEntryRemoved);
        assertTrue(publicEntryDisabled);

        authManager.revokeAccessFromPublic(Entity.STAGE, MOCKUP_STAGE_ID);
        assertFalse(authManager.hasAccessPublic(Entity.STAGE, MOCKUP_STAGE_ID));
        authManager.grantAccessToPublic(Entity.STAGE, MOCKUP_STAGE_ID);
        assertTrue(authManager.hasAccessPublic(Entity.STAGE, MOCKUP_STAGE_ID));
    }

    /** Tests the "findACLEntriesBy..." method family. */
    public void testFindACLEntriesByX()
    {
        assertTrue(authManager.findACLEntriesByStage(MOCKUP_STAGE_ID).size() > 0);
        assertEquals(authManager.findACLEntriesByService(SERVICE_ID).size(), 0);
        assertEquals(authManager.findACLEntriesByApplication(APPLICATION_ID).size(), 0);
        assertEquals(authManager.findACLEntriesByProvider(PROVIDER_ID).size(), 0);

        ACLEntry userEntry = new ACLEntry();
        userEntry.setUserID(USER_ID);

        try
        {
            authManager.addACLEntry(userEntry);
        }
        catch (AuthorizationManagerException e)
        {
            e.printStackTrace();
        }

        assertTrue(authManager.findACLEntriesByUser(USER_ID).size() > 0);

        ACLEntry groupEntry = new ACLEntry();
        groupEntry.setGroupID(GROUP_ID);

        try
        {
            authManager.addACLEntry(groupEntry);
        }
        catch (AuthorizationManagerException e)
        {
            e.printStackTrace();
        }

        assertTrue(authManager.findACLEntriesByGroup(GROUP_ID).size() > 0);

        ACLEntry publicEntry = new ACLEntry();
        publicEntry.setPublicAccess();

        try
        {
            authManager.addACLEntry(publicEntry);
        }
        catch (AuthorizationManagerException e)
        {
            e.printStackTrace();
        }

        assertTrue(authManager.findPublicACLEntries().size() > 0);
    }

    /**
     * Test user access
     *
     * @exception AuthorizationManagerException
     * @exception IdentityManagerException
     */
    public void testGroupAccess() throws AuthorizationManagerException, IdentityManagerException
    {
        // Grant stage access to group (group will be created on the ACL)
        authManager.grantAccessToGroup(GROUP_ID, Entity.STAGE, STAGE_ID);
        // Grant another stage access to group (group will NOT be created on the ACL)
        authManager.grantAccessToGroup(GROUP_ID, Entity.STAGE, ANOTHER_STAGE_ID);
        // Group has access to both stages
        assertTrue(authManager.findACLEntriesByStage(STAGE_ID).get(0).getGroupID().equals(GROUP_ID));
        assertTrue(authManager.findACLEntriesByStage(ANOTHER_STAGE_ID).get(0).getGroupID().equals(GROUP_ID));
        // Report no access since the stages are fictional
        assertFalse(authManager.hasAccessGroup(group_id, Entity.STAGE, STAGE_ID));
        assertFalse(authManager.hasAccessGroup(group_id, Entity.STAGE, ANOTHER_STAGE_ID));
        // Revoke stage access to group
        authManager.revokeAccessFromGroup(GROUP_ID, Entity.STAGE, STAGE_ID);
        // No access for group!
        assertTrue(authManager.findACLEntriesByStage(STAGE_ID).isEmpty());
        assertTrue(authManager.revokeAllAccessFromGroup(GROUP_ID));

        // Add this group as a subgroup of the main group
        authManager.grantAccessToGroup(PROFILE_ID, Entity.STAGE, MOCKUP_STAGE_ID);
        // Both groups should has access, one direct, the other inherited
        assertTrue(authManager.hasAccessGroup(profile_id, Entity.STAGE, MOCKUP_STAGE_ID));
        assertTrue(authManager.hasAccessGroup(group_id, Entity.STAGE, MOCKUP_STAGE_ID));
        // Restore empty grant
        authManager.revokeAccessFromGroup(PROFILE_ID, Entity.STAGE, MOCKUP_STAGE_ID);

        // Revoke stage access to non-existent group
        authManager.revokeAccessFromGroup(BAD_GROUP_ID, Entity.STAGE, MOCKUP_STAGE_ID);
        // Still has access, since it is public!
        assertTrue(authManager.hasAccessGroup(identManager.getGroup(BAD_GROUP_ID), Entity.STAGE, MOCKUP_STAGE_ID));
        authManager.revokeAccessFromPublic(Entity.STAGE, MOCKUP_STAGE_ID);
        // No longer public, no access for bad user!
        assertFalse(authManager.hasAccessGroup(identManager.getGroup(BAD_GROUP_ID), Entity.STAGE, MOCKUP_STAGE_ID));
        authManager.grantAccessToPublic(Entity.STAGE, MOCKUP_STAGE_ID);
        // Has access again, since it is public!
        assertTrue(authManager.hasAccessGroup(identManager.getGroup(BAD_GROUP_ID), Entity.STAGE, MOCKUP_STAGE_ID));

        // Check access to a nonexistent stage
        assertFalse(authManager.hasAccessGroup(group_id, Entity.STAGE, NONEXISTENT_STAGE_ID));

        // Check access to an inaccessible stage (access was not granted to this group or it's ancestors)
        assertFalse(authManager.hasAccessGroup(group_id, Entity.STAGE, INACCESSIBLE_STAGE_ID));
    }

    /**
     * Tests the inherited group access.
     *
     * @exception AuthorizationManagerException if the authorization manager can't be accessed~
     * @exception IdentityManagerException      if the identity manager can't be accessed
     */
    public void testInheritedGroupAccess() throws AuthorizationManagerException, IdentityManagerException
    {

        // Set group hierarchy
        setGroupHierarchy();

        // Grant stage access to top-level group
        assertTrue(authManager.grantAccessToGroup(GRANDPARENT_GROUP_ID, Entity.STAGE, MOCKUP_STAGE_ID));
        assertTrue(authManager.hasAccessGroup(identManager.getGroup(CHILD_GROUP_ID), Entity.STAGE, MOCKUP_STAGE_ID));
        assertTrue(authManager.revokeAccessFromGroup(GRANDPARENT_GROUP_ID, Entity.STAGE, MOCKUP_STAGE_ID));

        // Remove group hierarchy
        removeGroupHierarchy();
    }

    /**
     * Tests the mass removal methods
     *
     * @exception AuthorizationManagerException if the authorization manager can't be accessed
     */
    public void testMassRemoval() throws AuthorizationManagerException
    {
        ACLEntry entry1, entry2;

        entry1 = new ACLEntry();
        entry1.setUserID(USER_ID);
        entry1.setEntityType(Entity.STAGE);
        entry1.setEntityID(STAGE_ID);

        entry2 = new ACLEntry();
        entry2.setUserID(USER_ID);
        entry2.setEntityType(Entity.STAGE);
        entry2.setEntityID(ANOTHER_STAGE_ID);

        authManager.addACLEntry(entry1);
        authManager.addACLEntry(entry2);

        assertTrue(authManager.findACLEntriesByStage(STAGE_ID).get(0).getUserID().equals(USER_ID));
        assertTrue(authManager.findACLEntriesByStage(ANOTHER_STAGE_ID).get(0).getUserID().equals(USER_ID));
        assertTrue(authManager.revokeAllAccessFromUser(USER_ID));
        assertTrue(authManager.findACLEntriesByUser(USER_ID).isEmpty());

        entry1 = null;
        entry2 = null;

        entry1 = new ACLEntry();
        entry1.setGroupID(GROUP_ID);
        entry1.setEntityType(Entity.STAGE);
        entry1.setEntityID(STAGE_ID);

        entry2 = new ACLEntry();
        entry2.setGroupID(GROUP_ID);
        entry2.setEntityType(Entity.STAGE);
        entry2.setEntityID(ANOTHER_STAGE_ID);

        authManager.addACLEntry(entry1);
        authManager.addACLEntry(entry2);

        assertTrue(authManager.findACLEntriesByStage(STAGE_ID).get(0).getGroupID().equals(GROUP_ID));
        assertTrue(authManager.findACLEntriesByStage(ANOTHER_STAGE_ID).get(0).getGroupID().equals(GROUP_ID));
        assertTrue(authManager.revokeAllAccessFromGroup(GROUP_ID));
        assertTrue(authManager.findACLEntriesByGroup(GROUP_ID).isEmpty());

        entry1 = null;
        entry2 = null;
    }

    /**
     * @exception AuthorizationManagerException
     */
    public void testNonExistantEntity() throws AuthorizationManagerException
    {
        assertFalse(authManager.hasAccessUser(user_id, Entity.APPLICATION, NON_EXISTANT_APPLICATION));
        assertFalse(authManager.hasAccessUser(user_id, Entity.SERVICE, NON_EXISTANT_SERVICE));
        assertFalse(authManager.hasAccessUser(user_id, Entity.STAGE, NONEXISTENT_STAGE_ID));
        assertFalse(authManager.hasAccessUser(user_id, null, ""));
        assertFalse(authManager.hasAccessPublic(null, ""));

        assertFalse(authManager.hasAccessGroup(group_id, Entity.APPLICATION, NON_EXISTANT_APPLICATION));
        assertFalse(authManager.hasAccessGroup(group_id, Entity.SERVICE, NON_EXISTANT_SERVICE));
        assertFalse(authManager.hasAccessGroup(group_id, Entity.STAGE, NONEXISTENT_STAGE_ID));
        assertFalse(authManager.hasAccessGroup(group_id, null, ""));
        assertFalse(authManager.hasAccessPublic(null, ""));
    }

    /**
     * Tests the public access operations.
     *
     * @exception AuthorizationManagerException if the authorization manager can't be accessed
     */
    public void testPublicAccess() throws AuthorizationManagerException
    {

        assertFalse(authManager.revokeAccessFromPublic(Entity.STAGE, NONEXISTENT_STAGE_ID));

        ACLEntry publicEntry = new ACLEntry();
        publicEntry.setPublicAccess();
        publicEntry.setDefault(false);
        publicEntry.setEntityID(PUBLIC_STAGE_ID);
        publicEntry.setEntityType(Entity.STAGE);
        publicEntry.setEnabled(true);

        assertTrue(authManager.addACLEntry(publicEntry));

        // Force ACL entry update
        publicEntry.setEnabled(false);

        // Force entry update
        assertTrue(authManager.addACLEntry(publicEntry));
        assertTrue(authManager.revokeACLEntry(publicEntry));

        // Restore previous settings
        publicEntry.setEnabled(true);
        assertTrue(authManager.addACLEntry(publicEntry));

        // Force 'isPublicAccess' branch
        authManager.grantAccessToPublic(Entity.STAGE, PUBLIC_STAGE_ID);

        // Force removal of an existent entry
        assertTrue(authManager.revokeACLEntry(publicEntry));
    }

    /**
     * Tests the methods that have an IStage as a parameters type.
     *
     * @exception AuthorizationManagerException if the authorization manager can't be accessed
     * @exception IdentityManagerException
     */
    public void testStageMethods() throws AuthorizationManagerException, IdentityManagerException
    {
        assertNotNull(stageInstance);

        assertTrue(authManager.hasAccessPublic(stageInstance));
        // Stage has public access so despite no direct grant is given the following user and group will have acesss
        // granted
        assertTrue(authManager.hasAccessGroup(group_id, stageInstance));
        assertTrue(authManager.hasAccessUser(user_id, stageInstance));
    }

    /**
     * Test user access
     *
     * @exception AuthorizationManagerException if the authorization manager can't be accessed
     * @exception IdentityManagerException      if the identity manager can't be accessed
     */
    public void testUserAccess() throws AuthorizationManagerException, IdentityManagerException
    {
        // Grant stage access to user (user will be created on the ACL)
        assertTrue(authManager.grantAccessToUser(USER_ID, Entity.STAGE, STAGE_ID));
        // Grant another stage access to user (user will NOT be created on the ACL)
        assertTrue(authManager.grantAccessToUser(USER_ID, Entity.STAGE, ANOTHER_STAGE_ID));
        // Repeated grant... will have no consequence.
        assertTrue(authManager.grantAccessToUser(USER_ID, Entity.STAGE, ANOTHER_STAGE_ID));

        // User has access to both stages
        assertTrue(authManager.findACLEntriesByStage(STAGE_ID).get(0).getUserID().equals(USER_ID));
        assertTrue(authManager.findACLEntriesByStage(ANOTHER_STAGE_ID).get(0).getUserID().equals(USER_ID));
        // No access reported since the stage is ficticious
        assertFalse(authManager.hasAccessUser(user_id, Entity.STAGE, STAGE_ID));
        assertFalse(authManager.hasAccessUser(user_id, Entity.STAGE, ANOTHER_STAGE_ID));
        // Revoke stage access to user
        assertTrue(authManager.revokeAccessFromUser(USER_ID, Entity.STAGE, STAGE_ID));
        // User lost access
        assertTrue(authManager.findACLEntriesByStage(STAGE_ID).isEmpty());
        // Revoke remaining stage access from user
        assertTrue(authManager.revokeAllAccessFromUser(USER_ID));
        assertTrue(authManager.findACLEntriesByUser(USER_ID).isEmpty());

        // Create the group PROFILE_ID
        DIFGroupImpl group = new DIFGroupImpl();
        group.setID(PROFILE_ID);
        identManager.addGroup(group);

        // Remove default public access for testing
        assertTrue(authManager.revokeAccessFromPublic(Entity.STAGE, MOCKUP_STAGE_ID));

        // Add access to a profile group
        assertTrue(authManager.grantAccessToGroup(PROFILE_ID, Entity.STAGE, MOCKUP_STAGE_ID));
        // Group has access
        assertTrue(authManager.hasAccessGroup(profile_id, Entity.STAGE, MOCKUP_STAGE_ID));
        // Set the profile group as the profile of the user
        user_id.setProfileID(PROFILE_ID);
        identManager.updateUser(user_id, USER_ID);
        // User should have access again... inherited from the profile
        assertTrue(authManager.hasAccessUser(user_id, Entity.STAGE, MOCKUP_STAGE_ID));
        // Remove profile
        user_id.setProfileID(null);
        identManager.removeUserFromGroup(USER_ID, PROFILE_ID);
        identManager.updateUser(user_id, USER_ID);
        user_id.refresh();
        // User should not have access
        assertFalse(authManager.hasAccessUser(user_id, Entity.STAGE, MOCKUP_STAGE_ID));
        // Try to revoke an inexistent access
        assertFalse(authManager.revokeAccessFromUser(USER_ID, Entity.STAGE, STAGE_ID));
        assertTrue(authManager.revokeAccessFromGroup(PROFILE_ID, Entity.STAGE, MOCKUP_STAGE_ID));

        // Add user to group
        identManager.addUserToGroup(USER_ID, GROUP_ID);
        user_id.refresh();
        // Grant access to group
        authManager.grantAccessToGroup(GROUP_ID, Entity.STAGE, MOCKUP_STAGE_ID);
        // User should have access again... inherited from the group
        assertTrue(authManager.hasAccessUser(user_id, Entity.STAGE, MOCKUP_STAGE_ID));
        // Remove group association
        authManager.revokeAccessFromGroup(GROUP_ID, Entity.STAGE, MOCKUP_STAGE_ID);
        // User should not have access
        assertFalse(authManager.hasAccessUser(user_id, Entity.STAGE, MOCKUP_STAGE_ID));
        authManager.revokeAccessFromGroup(GROUP_ID, Entity.STAGE, MOCKUP_STAGE_ID);
        // Remove user from group
        identManager.removeUserFromGroup(GROUP_ID, USER_ID);

        // Grant stage access to non-existent user
        authManager.grantAccessToUser(BAD_USER_ID, Entity.STAGE, MOCKUP_STAGE_ID);
        // No access for bad user!
        assertFalse(authManager.hasAccessUser(identManager.getUser(BAD_USER_ID), Entity.STAGE, MOCKUP_STAGE_ID));
        // Revoke stage access to non-existent user
        authManager.revokeAccessFromUser(BAD_USER_ID, Entity.STAGE, MOCKUP_STAGE_ID);
        // No access for bad user!
        assertFalse(authManager.hasAccessUser(identManager.getUser(BAD_USER_ID), Entity.STAGE, MOCKUP_STAGE_ID));
        assertFalse(authManager.toString().equals(""));

        // Check access to a nonexistent stage
        assertFalse(authManager.hasAccessUser(user_id, Entity.STAGE, NONEXISTENT_STAGE_ID));

        // Check access to an inaccessible stage (access was not granted to this user, to it's profile or it's groups)
        assertFalse(authManager.hasAccessUser(user_id, Entity.STAGE, INACCESSIBLE_STAGE_ID));

        /* User has access through one of his groups */

        // Check that nor the group nor the user have access to the stage
        assertFalse(authManager.hasAccessUser(user_id, Entity.STAGE, MOCKUP_STAGE_ID));
        assertFalse(authManager
                .hasAccessGroup(identManager.getGroup(GREAT_GRANDPARENT_GROUP_ID), Entity.STAGE, MOCKUP_STAGE_ID));

        // Test case set up
        try
        {
            // Set group hierarchy
            setGroupHierarchy();

            // Associate the user to the group child group (not to the GGP group, to force loop unrolling on group
            // access checking branch)
            if (!identManager.isUserInGroup(USER_ID, CHILD_GROUP_ID))
            {
                identManager.addUserToGroup(USER_ID, CHILD_GROUP_ID);
                user_id.refresh();
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }

        // Grant public access to "great-grandparent group"
        assertTrue(authManager.grantAccessToGroup(GREAT_GRANDPARENT_GROUP_ID, Entity.STAGE, MOCKUP_STAGE_ID));
        // Check that both group and user have access
        assertTrue(authManager
                .hasAccessGroup(identManager.getGroup(GREAT_GRANDPARENT_GROUP_ID), Entity.STAGE, MOCKUP_STAGE_ID));
        assertTrue(authManager.hasAccessUser(user_id, Entity.STAGE, MOCKUP_STAGE_ID));

        // Revoke public access to "great-grandparent group"
        assertTrue(authManager.revokeAccessFromGroup(GREAT_GRANDPARENT_GROUP_ID, Entity.STAGE, MOCKUP_STAGE_ID));
        // Check that both group and user have access
        assertFalse(authManager
                .hasAccessGroup(identManager.getGroup(GREAT_GRANDPARENT_GROUP_ID), Entity.STAGE, MOCKUP_STAGE_ID));
        assertFalse(authManager.hasAccessUser(user_id, Entity.STAGE, MOCKUP_STAGE_ID));

        // Test case tear down
        try
        {
            // Remove user from group if needed
            if (!identManager.isUserInGroup(USER_ID, CHILD_GROUP_ID))
            {
                identManager.addUserToGroup(USER_ID, CHILD_GROUP_ID);
            }
            // Remove group hierarchy
            removeGroupHierarchy();
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }

        // Re-grant default public access for testing
        assertTrue(authManager.grantAccessToPublic(Entity.STAGE, MOCKUP_STAGE_ID));
    }
}
