/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dataminer.definition;

import java.beans.PropertyVetoException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.hibernate.cfg.Environment;

import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.dataminer.session.DDMUserProfile;
import pt.digitalis.dif.dataminer.session.IDDMUserProfile;
import pt.digitalis.dif.model.jdbc.DatabaseSessionFactory;
import pt.digitalis.utils.common.StringUtils;

/**
 * Defines a Statistical area of analysis. Logic group of dash-boards and indicators
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 31 de Jan de 2013
 */
public class Area {

    /** list of inner dashboards of this area */
    private Map<String, DashBoard> dashBoards = new LinkedHashMap<String, DashBoard>();

    /** the database password to connecto to */
    private String databasePassword;

    /** the database url to connecto to */
    private String databaseURL;

    /** the database username to connecto to */
    private String databaseUsername;

    /**
     * these will be added to all area indicators, unless they have a definition of their own (either profile or group)
     */
    private List<String> defaultGroupsToRestrict;

    /**
     * these will be added to all area indicators, unless they have a definition of their own (either profile or group)
     */
    private List<String> defaultProfilesToRestrict;

    /** the ID */
    private String id;

    /** list of inner indicators of this area */
    private Map<String, IIndicator> indicators = new LinkedHashMap<String, IIndicator>();

    /** the managerID */
    private String managerID;

    /** the title */
    private String title;

    /**
     * @param id
     * @param title
     */
    public Area(String id, String title)
    {
        super();
        this.id = id;
        this.title = title;
    }

    /**
     * @param dash
     *            the dashBoard to add
     */
    public void addDashBoard(DashBoard dash)
    {
        dash.setManagerID(this.getManagerID());
        dash.setAreaID(this.getId());

        getDashBoards().put(dash.getId(), dash);
    }

    /**
     * Adds an indicator to the Area
     *
     * @param indicator
     *            the indicator to add
     */
    public void addIndicator(IIndicator indicator)
    {
        indicator.setManagerID(this.getManagerID());
        indicator.setAreaID(this.getId());

        // If not access restrictions exist on the indicator set Area default ones
        if (!indicator.hasRestrictionsDefined()
                && (defaultGroupsToRestrict != null || defaultProfilesToRestrict != null))
        {
            indicator.setRestrictToGroups(defaultProfilesToRestrict);
            indicator.setRestrictToProfiles(getDefaultProfilesToRestrict());
        }

        getIndicators().put(indicator.getId(), indicator);
    }

    /**
     * Clones a area based on a profile
     *
     * @param profile
     * @param session
     * @return cloned area
     */
    public Area cloneAreaForProfile(IDDMUserProfile profile, IDIFSession session)
    {
        Area cloneArea = new Area(this.id, this.title);
        cloneArea.setManagerID(this.managerID);
        cloneArea.setDatabaseURL(this.databaseURL);
        cloneArea.setDatabaseUsername(this.databaseUsername);
        cloneArea.setDatabasePassword(this.databasePassword);
        cloneArea.setDefaultGroupsToRestrict(this.defaultGroupsToRestrict);
        cloneArea.setDefaultProfilesToRestrict(this.defaultProfilesToRestrict);

        List<String> indicatorsWithDirectAccess = DashboardManager.getInstance(managerID)
                .getIndicatorsWithDirectAccessForArea(session, this.id);

        for (Entry<String, IIndicator> entry: this.indicators.entrySet())
        {
            IIndicator cloneIndicator = entry.getValue();
            Boolean addIndicator = false;

            if (indicatorsWithDirectAccess.contains(cloneIndicator.getId()))
            {
                addIndicator = true;
            }
            else if (profile.getID().equals(DDMUserProfile.class.getSimpleName()))
            {
                if (cloneIndicator.getRestrictToGroups() == null)
                    addIndicator = true;
                else
                {
                    if (profile.getInstanceIDs() != null)
                    {
                        for (String instanceID: profile.getInstanceIDs())
                        {
                            if (cloneIndicator.getRestrictToGroups().contains(instanceID))
                            {
                                addIndicator = true;
                                break;
                            }
                        }
                    }
                }
            }
            else if (cloneIndicator.getRestrictToProfiles() == null || (cloneIndicator.getRestrictToProfiles() != null
                    && cloneIndicator.getRestrictToProfiles().contains(profile.getClass().getSimpleName())))
            {
                addIndicator = true;
            }

            if (addIndicator)
                cloneArea.addIndicator(cloneIndicator);
        }

        for (Entry<String, DashBoard> entry: this.dashBoards.entrySet())
        {
            DashBoard cloneDashBoard = entry.getValue();
            String cloneDashboardIndicators = null;

            // Validar indicadores associados ao dashboard
            if (StringUtils.isNotEmpty(cloneDashBoard.getIndicatorList()))
            {
                String[] dashboardIndicators = cloneDashBoard.getIndicatorList().split(";");

                for (String dashboardIndicator: dashboardIndicators)
                {
                    String[] indicators = dashboardIndicator.split(",");
                    String temp = null;

                    for (String indicator: indicators)
                    {
                        if (cloneArea.getIndicators().containsKey(indicator))
                        {
                            if (temp == null)
                                temp = indicator.toString();
                            else
                                temp = temp + "," + indicator.toString();
                        }
                    }

                    if (temp != null)
                    {
                        if (cloneDashboardIndicators == null)
                            cloneDashboardIndicators = temp;
                        else
                            cloneDashboardIndicators = cloneDashboardIndicators + ";" + temp;
                    }
                }
            }

            if (cloneDashboardIndicators != null)
            {
                DashBoard newDashBoard = new DashBoard(cloneDashBoard.getAreaID(), cloneDashBoard.getId(),
                        cloneDashBoard.getTitle(), cloneDashBoard.getTemplate());
                newDashBoard.setIndicatorList(cloneDashboardIndicators);
                cloneArea.addDashBoard(newDashBoard);
            }
        }

        return cloneArea;
    }

    /**
     * Builds a new connection to the configured database. If no database connection is present in this area then the
     * given session factory will be used to create a new connection
     *
     * @param defaultSessionFactory
     * @return the JDBC connection
     * @throws PropertyVetoException
     * @throws SQLException
     */
    @SuppressWarnings("deprecation")
    public Connection getConnection(SessionFactory defaultSessionFactory) throws SQLException, PropertyVetoException
    {
        if (StringUtils.isBlank(this.getDatabaseURL()) || StringUtils.isBlank(this.getDatabaseUsername())
                || StringUtils.isBlank(this.getDatabasePassword()))
        {
            boolean wasActive = defaultSessionFactory.getCurrentSession().getTransaction().isActive();
            if (!wasActive)
                defaultSessionFactory.getCurrentSession().beginTransaction();

            return defaultSessionFactory.getCurrentSession().connection();
        }
        else
        {
            DatabaseSessionFactory databaseSession = DatabaseSessionFactory.getInstance(this.getDatabaseURL(),
                    this.getDatabaseUsername(), this.getDatabasePassword());

            return databaseSession.getConnection();
        }
    }

    /**
     * Inspector for the 'dashBoards' attribute.
     *
     * @return the dashBoards value
     */
    public Map<String, DashBoard> getDashBoards()
    {
        return dashBoards;
    }

    /**
     * Inspector for the 'databasePassword' attribute.
     *
     * @return the databasePassword value
     */
    public String getDatabasePassword()
    {
        return databasePassword;
    }

    /**
     * Inspector for the 'databaseURL' attribute.
     *
     * @return the databaseURL value
     */
    public String getDatabaseURL()
    {
        return databaseURL;
    }

    /**
     * Inspector for the 'databaseUsername' attribute.
     *
     * @return the databaseUsername value
     */
    public String getDatabaseUsername()
    {
        return databaseUsername;
    }

    /**
     * Inspector for the 'defaultGroupsToRestrict' attribute.
     *
     * @return the defaultGroupsToRestrict value
     */
    public List<String> getDefaultGroupsToRestrict()
    {
        return defaultGroupsToRestrict;
    }

    /**
     * Inspector for the 'defaultProfilesToRestrict' attribute.
     *
     * @return the defaultProfilesToRestrict value
     */
    public List<String> getDefaultProfilesToRestrict()
    {
        return defaultProfilesToRestrict;
    }

    /**
     * Inspector for the 'id' attribute.
     *
     * @return the id value
     */
    public String getId()
    {
        return id;
    }

    /**
     * Inspector for the 'indicators' attribute.
     *
     * @return the indicators value
     */
    public Map<String, IIndicator> getIndicators()
    {
        return indicators;
    }

    /**
     * Inspector for the 'indicators' attribute.<br/>
     * Will filter out indicators from the current profile
     *
     * @param profile
     *            the active session profile
     * @return the indicators value
     */
    public Map<String, IIndicator> getIndicators(IDDMUserProfile profile)
    {
        return indicators;
    }

    /**
     * Inspector for the 'managerID' attribute.
     *
     * @return the managerID value
     */
    public String getManagerID()
    {
        return managerID;
    }

    /**
     * Inspector for the 'title' attribute.
     *
     * @return the title value
     */
    public String getTitle()
    {
        return title;
    }

    /**
     * Modifier for the 'dashBoards' attribute.
     *
     * @param dashBoards
     *            the new dashBoards value to set
     */
    public void setDashBoards(Map<String, DashBoard> dashBoards)
    {
        for (DashBoard dashboard: dashBoards.values())
            dashboard.setManagerID(this.managerID);

        this.dashBoards = dashBoards;
    }

    /**
     * Modifier for the 'databaseSessionFactory' attribute.
     *
     * @param hibernateConfiguration
     *            the new hibernate configuration to get the configurations from
     */
    public void setDatabaseConfiguration(Configuration hibernateConfiguration)
    {
        this.databaseURL = hibernateConfiguration.getProperty(Environment.URL);
        this.databaseUsername = hibernateConfiguration.getProperty(Environment.USER);
        this.databasePassword = hibernateConfiguration.getProperty(Environment.PASS);
    }

    /**
     * Modifier for the 'databasePassword' attribute.
     *
     * @param databasePassword
     *            the new databasePassword value to set
     */
    public void setDatabasePassword(String databasePassword)
    {
        this.databasePassword = databasePassword;
    }

    /**
     * Modifier for the 'databaseURL' attribute.
     *
     * @param databaseURL
     *            the new databaseURL value to set
     */
    public void setDatabaseURL(String databaseURL)
    {
        this.databaseURL = databaseURL;
    }

    /**
     * Modifier for the 'databaseUsername' attribute.
     *
     * @param databaseUsername
     *            the new databaseUsername value to set
     */
    public void setDatabaseUsername(String databaseUsername)
    {
        this.databaseUsername = databaseUsername;
    }

    /**
     * Modifier for the 'defaultGroupsToRestrict' attribute.
     *
     * @param defaultGroupsToRestrict
     *            the new defaultGroupsToRestrict value to set
     */
    public void setDefaultGroupsToRestrict(List<String> defaultGroupsToRestrict)
    {
        this.defaultGroupsToRestrict = defaultGroupsToRestrict;
    }

    /**
     * Modifier for the 'defaultProfilesToRestrict' attribute.
     *
     * @param defaultProfilesToRestrict
     *            the new defaultProfilesToRestrict value to set
     */
    public void setDefaultProfilesToRestrict(List<String> defaultProfilesToRestrict)
    {
        this.defaultProfilesToRestrict = defaultProfilesToRestrict;
    }

    /**
     * Modifier for the 'id' attribute.
     *
     * @param id
     *            the new id value to set
     */
    public void setId(String id)
    {
        this.id = id;
    }

    /**
     * Modifier for the 'indicators' attribute.
     *
     * @param indicators
     *            the new indicators value to set
     */
    public void setIndicators(Map<String, IIndicator> indicators)
    {
        for (IIndicator indicator: indicators.values())
        {
            indicator.setManagerID(this.managerID);
            indicator.setAreaID(this.getId());
        }

        this.indicators = indicators;
    }

    /**
     * Modifier for the 'managerID' attribute.
     *
     * @param managerID
     *            the new managerID value to set
     */
    public void setManagerID(String managerID)
    {
        this.managerID = managerID;
    }

    /**
     * Modifier for the 'title' attribute.
     *
     * @param title
     *            the new title value to set
     */
    public void setTitle(String title)
    {
        this.title = title;
    }
}
