/**
 * 2015, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dataminer.session;

import java.util.ArrayList;
import java.util.List;

import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.utils.common.StringUtils;

/**
 * Session manager that will manage and provide details about what the current user has access and has selected
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 26/08/2015
 */
public class DDMSession implements IDDMSession {

    /** the ID of the DDMSession in session */
    static public final String DDMSESSION_OBJECT_ID = "DDMObjectInSession";

    /**
     * @param session
     *            the current session
     * @return the DDMSession initialized o
     * @throws IdentityManagerException
     */
    public static IDDMSession getInstance(IDIFSession session) throws IdentityManagerException
    {
        IDDMSession ddmSession = (IDDMSession) session.getAttribute(DDMSESSION_OBJECT_ID);

        if (ddmSession == null)
        {
            ddmSession = DIFIoCRegistry.getRegistry().getImplementation(IDDMSession.class).initialize(session);
            session.addAttribute(DDMSESSION_OBJECT_ID, ddmSession);
        }
        // If user is not the same has the one in session (logout/login actions may cause this) must reset the
        // DDMSessions internals
        else if (ddmSession.getUserIDInSession() == null && session.isLogged()
                || StringUtils.isNotBlank(ddmSession.getUserIDInSession()) && !session.isLogged()
                || (ddmSession.getUserIDInSession() != null
                        && !ddmSession.getUserIDInSession().equals(session.getUser().getID())))
        {
            ddmSession = DIFIoCRegistry.getRegistry().getImplementation(IDDMSession.class).initialize(session);
            session.addAttribute(DDMSESSION_OBJECT_ID, ddmSession);
        }

        return ddmSession;
    }

    /**  */
    private IDDMUserProfile activeProfile;

    /**  */
    private String activeProfileInstanceValue;

    /** the list of user profiles */
    private List<IDDMUserProfile> profileList;

    /**  */
    protected IDIFSession session;

    /** user in session */
    private String userIDInSession = null;

    /**
     * Private constructor so that only getInstance or IoC can instanciate this class
     */
    protected DDMSession()
    {
        super();
    }

    /**
     * @see pt.digitalis.dif.dataminer.session.IDDMSession#getActiveProfile()
     */
    public IDDMUserProfile getActiveProfile()
    {
        return this.activeProfile;
    }

    /**
     * Inspector for the 'activeProfileInstanceValue' attribute.
     *
     * @return the activeProfileInstanceValue value
     */
    public String getActiveProfileInstanceValue()
    {
        return activeProfileInstanceValue;
    }

    /**
     * @see pt.digitalis.dif.dataminer.session.IDDMSession#getAvailableProfiles()
     */
    public List<IDDMUserProfile> getAvailableProfiles()
    {
        return profileList;
    }

    /**
     * @see pt.digitalis.dif.dataminer.session.IDDMSession#getUserIDInSession()
     */
    public String getUserIDInSession()
    {
        return userIDInSession;
    }

    /**
     * @see pt.digitalis.dif.dataminer.session.IDDMSession#initialize(pt.digitalis.dif.controller.interfaces.IDIFSession)
     */
    public IDDMSession initialize(IDIFSession session) throws IdentityManagerException
    {
        this.profileList = new ArrayList<IDDMUserProfile>();
        this.session = session;
        this.userIDInSession = (session.isLogged() ? session.getUser().getID() : null);

        DDMUserProfile userProfile = new DDMUserProfile(session);

        this.activeProfile = userProfile;
        this.profileList.add(activeProfile);

        return this;
    }

    /**
     * @see pt.digitalis.dif.dataminer.session.IDDMSession#setActiveProfile(pt.digitalis.dif.dataminer.session.IDDMUserProfile,
     *      java.lang.String)
     */
    public void setActiveProfile(IDDMUserProfile profile, String profileInstanceValue)
    {
        if (!this.getAvailableProfiles().contains(profile))
            this.getAvailableProfiles().add(profile);

        this.activeProfile = profile;
        this.activeProfileInstanceValue = profileInstanceValue;

        this.session.addAttribute(DDMSESSION_OBJECT_ID, this);
    }
}
