/**
 * 2013, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dataminer.definition;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.dataminer.DDMUtils;
import pt.digitalis.dif.dem.managers.IMessageManager;
import pt.digitalis.dif.exception.BusinessException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.utils.common.StringUtils;

/**
 * A manager class that manages the existant dashboards
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 21 de Jan de 2013
 */
public class DashboardManager {

    /**  */
    public static final String DEFAULT_TEMPLATE_ID = "default";

    /** Attribute ID in session that will keep all directly accessible indicators for the current user */
    static private final String DIRECT_INDICATORS_ACCESS_IN_SESSION = "directIndicatorsAccessInSession";

    /**
     * Attribute ID in session that will keep all editable indicator managers (allow access to all indicators of the
     * manager) for the current user
     */
    static private final String INDICATOR_MANAGERS_CHANGE_ACCESS_IN_SESSION = "indicatorManagersChangeAccessInSession";

    /** Attribute ID in session that will keep all editable indicators for the current user */
    static private final String INDICATORS_CHANGE_ACCESS_IN_SESSION = "indicatorsChangeAccessInSession";

    /** the singleton instance */
    static private Map<String, DashboardManager> instances = new HashMap<String, DashboardManager>();

    /**
     * Tests if a given manager already exists
     *
     * @param instanceID
     *            the instance ID to test
     * @return T if it exists
     */
    static public boolean exists(String instanceID)
    {
        return instances.containsKey(instanceID);
    }

    /**
     * Builds or retrives a previously build DashBoard manager instance. To see if one exists call exists method prior
     * to this one.
     *
     * @param instanceID
     *            the instanceID of the manager to get
     * @return the manager instance
     */
    synchronized static public DashboardManager getInstance(String instanceID)
    {
        if (instances.get(instanceID) == null)
            instances.put(instanceID, new DashboardManager(instanceID));

        return instances.get(instanceID);
    }

    /**
     * Inspector for the 'instances' attribute.
     *
     * @return the instances value
     */
    public static Map<String, DashboardManager> getInstances()
    {
        return instances;
    }

    /** list of inner indicators of this area */
    private Map<String, Area> areas = new LinkedHashMap<String, Area>();

    /** the persistence layer */
    private IDashboardPersistence configPersistence = DIFIoCRegistry.getRegistry()
            .getImplementation(IDashboardPersistence.class);

    /** the instance ID */
    private String id;

    /** common templates to offer in the SQL editor of this dashboard */
    private Map<String, DashboardSQLTemplateData> sqlTemplates = new LinkedHashMap<String, DashboardSQLTemplateData>();

    /**
     * Private constructor
     *
     * @param id
     *            the instance id
     */
    private DashboardManager(String id)
    {
        this.id = id;

        DashboardSQLTemplateData defaultSQLTemplate = new DashboardSQLTemplateData(DEFAULT_TEMPLATE_ID,
                "select 1 \"id\", 1 \"key\", 'Test description' \"desc\", 10 \"value\", 20 \"value2\" from dual");

        for (String language: DIFIoCRegistry.getRegistry().getImplementation(IMessageManager.class)
                .getSupportedLanguages())
        {
            Map<String, String> messages = DDMUtils.getDDMApplicationMessages(language);
            defaultSQLTemplate.addLanguage(language, messages.get("indicator.template.default.title"),
                    messages.get("indicator.template.default.description"));

            this.sqlTemplates.put(DEFAULT_TEMPLATE_ID, defaultSQLTemplate);
        }
    }

    /**
     * @param area
     *            the area to add
     */
    public void addArea(Area area)
    {
        area.setManagerID(id);

        this.getAreas().put(area.getId(), area);
    }

    /**
     * Creates a copy of an existing indicator
     *
     * @param sourceIndicator
     *            the existing indicator to copy
     * @param destinationArea
     *            the area that the new indicator will belog to
     * @param newIndicatorID
     *            the new indicator ID
     * @return the new indicator
     * @throws IllegalAccessException
     * @throws InstantiationException
     */
    public IIndicator copyIndicator(IIndicator sourceIndicator, Area destinationArea, String newIndicatorID)
            throws InstantiationException, IllegalAccessException
    {
        IIndicator newIndicator = sourceIndicator.getClass().newInstance();

        newIndicator.setId(newIndicatorID);

        newIndicator.setAutoRefreshInterval(sourceIndicator.getAutoRefreshInterval());
        newIndicator.setDescription(sourceIndicator.getDescription());
        newIndicator.setDescriptionTitle(sourceIndicator.getDescriptionTitle());
        newIndicator.setEditable(sourceIndicator.isEditable());
        newIndicator.setGroupTitle(sourceIndicator.getGroupTitle());
        newIndicator.setHasTotalField(sourceIndicator.isHasTotalField());
        newIndicator.setHideMarkers(sourceIndicator.isHideMarkers());
        newIndicator.setLegend(sourceIndicator.isLegend());
        newIndicator.setLimitTopRecords(sourceIndicator.getLimitTopRecords());
        newIndicator.setMaxValue(sourceIndicator.getMaxValue());
        newIndicator.setOrigyn(sourceIndicator.getOrigyn());

        if (sourceIndicator.getRestrictToGroups() != null)
            newIndicator.setRestrictToGroups(new ArrayList<String>(sourceIndicator.getRestrictToGroups()));

        if (sourceIndicator.getRestrictToProfiles() != null)
            newIndicator.setRestrictToProfiles(new ArrayList<String>(sourceIndicator.getRestrictToProfiles()));

        newIndicator.setTimeKeys(sourceIndicator.isTimeKeys());
        newIndicator.setTitle(sourceIndicator.getTitle());
        newIndicator.setUnitSuffix(sourceIndicator.getUnitSuffix());
        newIndicator.setUseMinutes(sourceIndicator.isUseMinutes());
        newIndicator.setxAxisTitle(sourceIndicator.getxAxisTitle());
        newIndicator.setyAxisTitle(sourceIndicator.getyAxisTitle());

        if (newIndicator instanceof IIndicatorSQL)
        {
            IIndicatorSQL newIndicatorSQL = (IIndicatorSQL) newIndicator;
            IIndicatorSQL sourceIndicatorSQL = (IIndicatorSQL) sourceIndicator;

            newIndicatorSQL.setDataSourceDescription(sourceIndicatorSQL.getDataSourceDescription());
            newIndicatorSQL.setMaterializedViews(sourceIndicatorSQL.getMaterializedViews());
            newIndicatorSQL.setQuerySQL(sourceIndicatorSQL.getQuerySQL());
        }

        // Copy series
        List<Series> newSeries = new ArrayList<Series>();
        for (Series serie: sourceIndicator.getSeries())
        {
            newSeries.add(new Series(serie.getChartType(), serie.getUnitName(), serie.getDescription(),
                    serie.getValueField()));
        }
        newIndicator.setSeries(newSeries);

        // Copy filters
        List<Filter> newFilters = new ArrayList<Filter>();
        for (Filter filter: sourceIndicator.getFilters())
        {
            Filter newFilter = new Filter(filter.getId(), filter.getTitle(), filter.getDescription(),
                    filter.getFilterSQLTemplate(), filter.getFilterDataset(), filter.getDataSetDescField(),
                    filter.getDefaultValue());
            newFilter.setBindToProfileID(filter.getBindToProfileID());

            if (filter.getLovItems() != null)
                newFilter.setLovItems(new HashMap<String, String>(filter.getLovItems()));

            newFilter.setLovQuerySQL(filter.getLovQuerySQL());
            newFilter.setType(filter.getType());

            newFilters.add(newFilter);
        }
        newIndicator.setFilters(newFilters);

        // Add indicator copy to the area
        destinationArea.addIndicator(newIndicator);

        return newIndicator;
    }

    /**
     * Inspector for the 'areas' attribute.
     *
     * @return the areas value
     */
    public Map<String, Area> getAreas()
    {
        return this.areas;
    }

    /**
     * Returns the list of areas that the current session has direct access to.<br/>
     * Direct Access: Can view this indicators data even if has not access to it by the user's current profiles
     *
     * @param session
     *            the current session
     * @return the list of direct access areas
     */
    public List<String> getAreasWithDirectAccess(IDIFSession session)
    {
        @SuppressWarnings("unchecked")
        List<String> directAccesses = (List<String>) session.getAttribute(DIRECT_INDICATORS_ACCESS_IN_SESSION);
        List<String> directAccessAreas = new ArrayList<String>();

        if (directAccesses != null)
        {
            for (String directAccessID: directAccesses)
            {
                if (StringUtils.isNotBlank(directAccessID))
                {
                    String areaID = directAccessID.split(":")[0];

                    if (!directAccessAreas.contains(areaID))
                        directAccessAreas.add(areaID);
                }
            }
        }

        return directAccessAreas;
    }

    /**
     * Inspector for the 'id' attribute.
     *
     * @return the id value
     */
    public String getId()
    {
        return id;
    }

    /**
     * Returns the list of indicators of the given area that the current session has direct access to.<br/>
     * Direct Access: Can view this indicators data even if has not access to it by the user's current profiles
     *
     * @param session
     *            the current session
     * @param areaID
     * @return the list of direct access indicators of the given area that
     */
    public List<String> getIndicatorsWithDirectAccessForArea(IDIFSession session, String areaID)
    {
        @SuppressWarnings("unchecked")
        List<String> directAccesses = (List<String>) session.getAttribute(DIRECT_INDICATORS_ACCESS_IN_SESSION);
        List<String> directAccessIndicadors = new ArrayList<String>();

        if (directAccesses != null)
        {
            for (String directAccessID: directAccesses)
            {
                if (StringUtils.isNotBlank(directAccessID))
                {
                    String directAreaID = directAccessID.split(":")[0];
                    String directIndicatorID = directAccessID.split(":")[1];

                    if (areaID.equals(directAreaID) && !directAccessIndicadors.contains(directIndicatorID))
                        directAccessIndicadors.add(directIndicatorID);
                }
            }
        }

        return directAccessIndicadors;
    }

    /**
     * Inspector for the 'sqlTemplates' attribute.
     *
     * @return the sqlTemplates value
     */
    public Map<String, DashboardSQLTemplateData> getSqlTemplates()
    {
        return sqlTemplates;
    }

    /**
     * @return the total areas of the current manager
     */
    public int getTotalAreas()
    {
        return this.getAreas().size();
    }

    /**
     * @return the total areas of the current manager
     */
    public int getTotalIndicators()
    {
        int total = 0;

        for (Area area: this.getAreas().values())
            total += area.getIndicators().size();

        return total;
    }

    /**
     * Gives the current session direct access to the given indicator.<br/>
     * Direct Access: Can view this indicators data even if has not access to it by the user's current profiles
     *
     * @param session
     *            the current DIF session
     * @param areaID
     * @param indicatorID
     */
    public void grantDirectAccessToIndicator(IDIFSession session, String areaID, String indicatorID)
    {
        @SuppressWarnings("unchecked")
        List<String> directAccesses = (List<String>) session.getAttribute(DIRECT_INDICATORS_ACCESS_IN_SESSION);
        String compositeID = areaID + ":" + indicatorID;

        if (directAccesses == null)
            directAccesses = new ArrayList<String>();

        if (!directAccesses.contains(compositeID))
            directAccesses.add(compositeID);

        session.addAttribute(DIRECT_INDICATORS_ACCESS_IN_SESSION, directAccesses);
    }

    /**
     * Gives the current session edit access to the given manager's indicators.<br/>
     * Will also give direct access to the indicators (so that if the user cannot already access it, will not prevent
     * editing)<br/>
     * Edit Access: Can change the indicator definition and thus change all it's properties and behavior.<br/>
     *
     * @param session
     *            the current DIF session
     */
    public void grantEditAccessToAllIndicators(IDIFSession session)
    {
        @SuppressWarnings("unchecked")
        List<DashboardManager> editAccesses = (List<DashboardManager>) session
                .getAttribute(INDICATOR_MANAGERS_CHANGE_ACCESS_IN_SESSION);

        if (editAccesses == null)
            editAccesses = new ArrayList<DashboardManager>();

        if (!editAccesses.contains(this))
            editAccesses.add(this);

        session.addAttribute(INDICATOR_MANAGERS_CHANGE_ACCESS_IN_SESSION, editAccesses);

        for (Area area: this.getAreas().values())
            for (IIndicator indicator: area.getIndicators().values())
                this.grantDirectAccessToIndicator(session, indicator.getAreaID(), indicator.getId());
    }

    /**
     * Gives the current session edit access to the given indicator.<br/>
     * Will also give direct access to the indicator (so that if the user cannot already access it, will not prevent
     * editing)<br/>
     * Edit Access: Can change the indicator definition and thus change all it's properties and behavior.<br/>
     *
     * @param session
     *            the current DIF session
     * @param indicator
     */
    public void grantEditAccessToIndicator(IDIFSession session, IIndicator indicator)
    {
        @SuppressWarnings("unchecked")
        List<IIndicator> editAccesses = (List<IIndicator>) session.getAttribute(INDICATORS_CHANGE_ACCESS_IN_SESSION);

        if (editAccesses == null)
            editAccesses = new ArrayList<IIndicator>();

        if (!editAccesses.contains(indicator))
            editAccesses.add(indicator);

        session.addAttribute(INDICATORS_CHANGE_ACCESS_IN_SESSION, editAccesses);

        this.grantDirectAccessToIndicator(session, indicator.getAreaID(), indicator.getId());
    }

    /**
     * Determines if the current session has been given direct access to the given indicator.<br/>
     * Direct Access: Can view this indicators data even if has not access to it by the user's current profiles
     *
     * @param session
     *            the current DIF session
     * @param areaID
     * @param indicatorID
     * @return T if the current session has direct access
     */
    public boolean hasDirectAccessToIndicator(IDIFSession session, String areaID, String indicatorID)
    {
        @SuppressWarnings("unchecked")
        List<String> directAccesses = (List<String>) session.getAttribute(DIRECT_INDICATORS_ACCESS_IN_SESSION);
        String compositeID = areaID + ":" + indicatorID;

        return (directAccesses != null && directAccesses.contains(compositeID));
    }

    /**
     * Determines if the current session has been given edit access to the given indicator.<br/>
     * Edit Access: Can change the indicator definition and thus change all it's properties and behavior
     *
     * @param session
     *            the current DIF session
     * @param indicator
     * @return T if the current session has edit access
     */
    public boolean hasEditAccessToIndicator(IDIFSession session, IIndicator indicator)
    {
        @SuppressWarnings("unchecked")
        List<IIndicator> editAccessesIndicators = (List<IIndicator>) session
                .getAttribute(INDICATORS_CHANGE_ACCESS_IN_SESSION);

        DashboardManager indicatorManager = DashboardManager.getInstance(indicator.getManagerID());
        @SuppressWarnings("unchecked")
        List<DashboardManager> editAccessesManagers = (List<DashboardManager>) session
                .getAttribute(INDICATOR_MANAGERS_CHANGE_ACCESS_IN_SESSION);

        return (editAccessesIndicators != null && editAccessesIndicators.contains(indicator))
                || (editAccessesManagers != null && editAccessesManagers.contains(indicatorManager));
    }

    /**
     * @return T if this dashboard has a persistence layer available to load and save custom indicators
     */
    public boolean isPersistenceLayerAvailable()
    {
        return this.configPersistence.isAvailable();
    }

    /**
     * Load indicator from persistence layer.
     *
     * @param indicator
     *            the indicator
     * @throws BusinessException
     *             the business exception
     */
    public void loadIndicatorFromPersistenceLayer(IIndicator indicator) throws BusinessException
    {
        this.configPersistence.loadIndicatorConfiguration(indicator);
    }

    /**
     * Removes a given indicator's configuration (will revert to default or disappear is it is a custom one)
     *
     * @param indicator
     *            the indicator to remove
     * @throws BusinessException
     */
    public void removeIndicatorFromPersistenceLayer(IIndicator indicator) throws BusinessException
    {
        this.configPersistence.removeIndicatorConfiguration(indicator);
    }

    /**
     * Removes a given indicator's configuration (will revert to default or disappear is it is a custom one)
     *
     * @param indicatorID
     *            the indicator's ID to remove
     * @throws BusinessException
     */
    public void removeIndicatorFromPersistenceLayer(String indicatorID) throws BusinessException
    {
        this.configPersistence.removeIndicatorConfiguration(indicatorID);
    }

    /**
     * Saves an indicator configuration to the persistence layer. Will register new indicators or update existing ones.
     *
     * @param indicator
     *            the indicator to save
     * @throws BusinessException
     */
    public void saveIndicatorConfigurationToPersistenceLayer(IIndicator indicator) throws BusinessException
    {
        this.configPersistence.saveIndicatorConfiguration(indicator);
    }

    /**
     * Modifier for the 'areas' attribute.
     *
     * @param areas
     *            the new areas value to set
     */
    public void setAreas(Map<String, Area> areas)
    {
        for (Area area: areas.values())
        {
            area.setManagerID(id);

            for (DashBoard dashboard: area.getDashBoards().values())
                dashboard.setManagerID(id);
            for (IIndicator indicator: area.getIndicators().values())
                indicator.setManagerID(id);
        }

        this.areas = areas;
    }

    /**
     * Modifier for the 'sqlTemplates' attribute.
     *
     * @param sqlTemplates
     *            the new sqlTemplates value to set
     */
    public void setSqlTemplates(Map<String, DashboardSQLTemplateData> sqlTemplates)
    {
        this.sqlTemplates = sqlTemplates;
    }
}
