/**
 * 2015, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dataminer.session;

import java.util.List;
import java.util.Map;

import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.dataminer.definition.Area;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 26/08/2015
 */
public interface IDDMUserProfile {

    /**
     * Clears the current profile indicator cache. Useful after indicator changes are made by administrators. Will defer
     * the indicator load to when they are requested again later on
     */
    public void clearIndicatorCache();

    /**
     * Gets all areas that the current profile has access to.<br/>
     * All inner dashboards and indicators will also be filtered by profile access (only indicators that this profile
     * has access will be available)
     *
     * @param session
     *            the session (needed to check for direct access given to specific areas/indicators)
     * @param dashBoardManagerInstanceID
     *            the instance ID (there can be several dashboard managers so we need to state wich one we want)
     * @return the list of filtered areas
     */
    public Map<String, Area> getAreas(IDIFSession session, String dashBoardManagerInstanceID);

    /**
     * Builds a description for the current profile and the given instance ID if relevant
     *
     * @param instanceID
     *            the instance ID for the profile if relevant
     * @param languageID
     *            the language ID
     * @return the description of this profile for user information
     */
    public String getDescription(String instanceID, String languageID);

    /**
     * @return the profile ID
     */
    public String getID();

    /**
     * Will calculate a business description for this profile's specific instance (i.e. for DDMUserProfile the instances
     * are groupIDs. This method when given a groupID will return the group name from the identity manager)
     *
     * @param profileInstanceID
     * @return the business description for this profiles intanceID
     */
    public String getInstanceIDBusinessDescription(String profileInstanceID);

    /**
     * @return the specific instance IDs (list of groups or list of business IDs - depends on each profile type)
     */
    public List<String> getInstanceIDs();

    /**
     * Builds a name for the current profile and the given instance ID if relevant
     *
     * @param instanceID
     *            the instance ID for the profile if relevant
     * @param languageID
     *            the language ID
     * @return the name of this profile for user information
     */
    public String getName(String instanceID, String languageID);

    /**
     * Builds a description for the current instanceID only if it exists (not adding the profile name as a prefix)
     *
     * @param instanceID
     *            the instance ID for the profile if relevant
     * @param languageID
     *            the language ID
     * @return the name of this profile for user information
     */
    public String getProfileInstanceDescriptionOnly(String instanceID, String languageID);

    /**
     * Builds a name for the current profile only (not adding instanceID information if any)
     *
     * @param instanceID
     *            the instance ID for the profile if relevant
     * @param languageID
     *            the language ID
     * @return the name of this profile for user information
     */
    public String getProfileNameOnly(String instanceID, String languageID);

    /**
     * @return T if the user is in administrator mode
     */
    public boolean isAdminMode();

    /**
     * @return T if the user is allowed to select an instanceID when chooses this profile
     */
    public boolean isAllowUserToSelectInstance();

    /**
     * @param adminMode
     *            the new administrator mode value
     */
    public void setAdminMode(boolean adminMode);

    /**
     * Modifier for the 'instanceIDs' attribute.<br/>
     * the specific instance IDs (list of groups or list of business IDs - depends on each profile type)
     *
     * @param instanceIDs
     *            the new instanceIDs value to set
     * @return the updated object
     */
    public IDDMUserProfile setInstanceIDs(List<String> instanceIDs);
}
