/**
 * 2015, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dataminer.session;

import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.dataminer.definition.Area;
import pt.digitalis.dif.dataminer.definition.DashboardManager;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.dem.managers.IMessageManager;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.templates.TemplateUtils;
import pt.digitalis.utils.common.CollectionUtils;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * Base implementation of a {@link IDDMUserProfile}
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 26/08/2015
 */
public abstract class AbstractDDMUserProfile implements IDDMUserProfile
{

    /**
     *
     */
    protected IDEMManager demManager = DIFIoCRegistry.getRegistry().getImplementation(IDEMManager.class);

    /**
     *
     */
    protected IMessageManager messageManager = DIFIoCRegistry.getRegistry().getImplementation(IMessageManager.class);

    /** the filtered areas cache instance */
    HashMap<String, LinkedHashMap<String, Area>> areasForManager = new HashMap<String, LinkedHashMap<String, Area>>();

    /** T if the user is in admin mode */
    private boolean adminMode = false;

    /** the list of instance IDs */
    private List<String> instanceIDs = null;

    /**
     * Default constructor without initialization context.<br/>
     * NOT FOR USE outside the DDM API classes as it may be removed in next releases
     */
    public AbstractDDMUserProfile()
    {
        super();
    }

    /**
     * @see pt.digitalis.dif.dataminer.session.IDDMUserProfile#clearIndicatorCache()
     */
    public void clearIndicatorCache()
    {
        areasForManager.clear();
    }

    /**
     * @see pt.digitalis.dif.dataminer.session.IDDMUserProfile#getAreas(pt.digitalis.dif.controller.interfaces.IDIFSession,
     *         java.lang.String)
     */
    public Map<String, Area> getAreas(IDIFSession session, String dashBoardManagerInstanceID)
    {

        if (this.isAdminMode())
            return DashboardManager.getInstance(dashBoardManagerInstanceID).getAreas();
        else
        {
            if (areasForManager.get(dashBoardManagerInstanceID) == null)
            {
                LinkedHashMap<String, Area> areas = new LinkedHashMap<String, Area>();

                List<String> areasWithDirectAccess =
                        DashboardManager.getInstance(dashBoardManagerInstanceID).getAreasWithDirectAccess(session);

                for (Entry<String, Area> entry : DashboardManager.getInstance(dashBoardManagerInstanceID).getAreas()
                        .entrySet())
                {
                    Area area = entry.getValue();

                    Boolean addArea = false;

                    if (areasWithDirectAccess.contains(area.getId()))
                    {
                        addArea = true;
                    }
                    else if (this.getID().equals(DDMUserProfile.class.getSimpleName()))
                    {
                        if (area.getDefaultGroupsToRestrict() == null)
                            addArea = true;
                        else
                        {
                            if (this.getInstanceIDs() != null)
                            {
                                for (String instanceID : this.getInstanceIDs())
                                {
                                    if (area.getDefaultGroupsToRestrict().contains(instanceID))
                                    {
                                        addArea = true;
                                        break;
                                    }
                                }
                            }
                        }
                    }
                    else
                    {
                        if (area.getDefaultProfilesToRestrict() == null)
                            addArea = true;
                        else if (area.getDefaultProfilesToRestrict().contains(this.getID()))
                            addArea = true;
                    }

                    if (addArea)
                    {
                        Area clone = area.cloneAreaForProfile(this, session);

                        if (clone != null && !clone.getIndicators().isEmpty())
                            areas.put(entry.getKey(), clone);
                    }
                }

                areasForManager.put(dashBoardManagerInstanceID, areas);
            }

            return areasForManager.get(dashBoardManagerInstanceID);
        }
    }

    /**
     * @see pt.digitalis.dif.dataminer.session.IDDMUserProfile#getID()
     */
    public String getID()
    {
        return this.getClass().getSimpleName();
    }

    /**
     * @see pt.digitalis.dif.dataminer.session.IDDMUserProfile#getInstanceIDs()
     */
    public List<String> getInstanceIDs()
    {
        return this.instanceIDs;
    }

    /**
     * Builds a message for the current profile, searching it in the given applications messages. If user
     * isAllowedToSelectInstance it T will parse the message and replace the given instanceID in the proper place.<br/>
     * <br/>
     * <ul>
     * <li>Messages sintax (no instance selection): [ProfileID].[messagesuffix]=Backoffice user</li>
     * <li>Messages sintax (with instance selection): [ProfileID].[messagesuffix]=Diretor do Curso {$instanceID}</li>
     * <li>Sufixes: "name" and "description"</li>
     * </ul>
     *
     * @param applicationID
     * @param messageSuffix
     * @param instanceID
     * @param languageID
     *
     * @return the parsed message
     */
    protected String getMessageFromApplicationMessages(String applicationID, String messageSuffix, String instanceID,
            String languageID)
    {
        Map<String, String> appMessages = DIFIoCRegistry.getRegistry().getImplementation(IMessageManager.class)
                .getMessages(this.demManager.getApplication(applicationID), languageID);

        String messageTemplate = appMessages.get(this.getID() + "." + messageSuffix);

        if (isAllowUserToSelectInstance())
            return TemplateUtils.parseTemplateLine(messageTemplate,
                    CollectionUtils.stringToKeyValueMap("instanceID=" + instanceID));
        else
            return messageTemplate;
    }

    /**
     * @see pt.digitalis.dif.dataminer.session.IDDMUserProfile#isAdminMode()
     */
    public boolean isAdminMode()
    {
        return this.adminMode;
    }

    /**
     * @see pt.digitalis.dif.dataminer.session.IDDMUserProfile#setAdminMode(boolean)
     */
    public void setAdminMode(boolean adminMode)
    {
        this.adminMode = adminMode;
    }

    /**
     * @see pt.digitalis.dif.dataminer.session.IDDMUserProfile#setInstanceIDs(java.util.List)
     */
    public IDDMUserProfile setInstanceIDs(List<String> instanceIDs)
    {
        this.instanceIDs = instanceIDs;

        return this;
    }
}
