/**
 * 2016, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.utils.logging.performance;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.StatelessSession;

import pt.digitalis.dif.dem.managers.impl.model.DIFRepositoryFactory;
import pt.digitalis.dif.dem.managers.impl.model.data.AccessLog;
import pt.digitalis.dif.exception.BusinessException;
import pt.digitalis.dif.model.hibernate.HibernateUtil;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.common.Chronometer;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 22/08/2016
 */
public class PerformanceLoggerDBImplementation extends PerformanceLogger {

    /**
     * A thread that will flush all given records form the log to the database. Will run as a thread so that we don't
     * make the DIF dispatcher wait for this action.<br/>
     * This will write the records in a stateless session in batch insert mode for performance sake.
     *
     * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
     * @created 26/08/2016
     */
    private class FlushLogbufferToDB extends Thread {

        /**  */
        private List<AccessLog> logsToWrite = new ArrayList<AccessLog>();

        /**
         * @param logsToWrite
         */
        public FlushLogbufferToDB(List<AccessLog> logsToWrite)
        {
            this.logsToWrite = logsToWrite;
        }

        /**
         * @see java.lang.Thread#run()
         */
        @Override
        public void run()
        {
            Chronometer crono = new Chronometer();
            crono.start();

            StatelessSession session = HibernateUtil.getSessionFactory(DIFRepositoryFactory.SESSION_FACTORY_NAME)
                    .openStatelessSession();
            session.beginTransaction();

            Connection c = session.connection();
            PreparedStatement ps;
            try
            {
                ps = c.prepareStatement("INSERT INTO dif.access_log(application_id,service_id,stage_id,"
                        + "event_id,request_type,user_id,access_date,execution_time,client_ip,error_report,request_dump) "
                        + "VALUES (?,?,?,?,?,?,?,?,?,?,?)");

                long inserted = 0L;

                for (AccessLog log: logsToWrite)
                {
                    inserted++;

                    ps.setString(1, log.getApplicationId());
                    ps.setString(2, log.getServiceId());
                    ps.setString(3, log.getStageId());
                    ps.setString(4, log.getEventId());
                    ps.setString(5, log.getRequestType().toString());
                    ps.setString(6, log.getUserId());
                    ps.setTimestamp(7, log.getAccessDate());
                    ps.setLong(8, log.getExecutionTime());
                    ps.setString(9, log.getClientIp());
                    if (StringUtils.isNotBlank(log.getErrorReport()))
                    {
                        String errorReport = log.getErrorReport();
                        if (errorReport.length() > 3999)
                        {
                            errorReport = errorReport.substring(0, 3999);
                        }
                        ps.setString(10, errorReport);
                    }
                    else
                    {
                        ps.setString(10, null);
                    }
                    ps.setString(11, log.getRequestDump());

                    ps.addBatch();
                    ps.clearParameters();
                }

                ps.executeBatch();
                session.getTransaction().commit();

                crono.end();

                DIFLogger.getLogger().info("Performance Access logs: " + inserted + " logs flushed in "
                        + crono.getTimePassedAsFormattedString());
            }
            catch (SQLException e)
            {
                new BusinessException(e).addToExceptionContext("LogsToWrite", logsToWrite).log(LogLevel.ERROR);
            }
        }
    }

    /** Every X records a batch insert will be made */
    private static final long MAX_RECORDS_BEFORE_FLUSH = 200;

    /** Every X seconds a batch insert will be made */
    private static final long MAX_SECONDS_BEFORE_FLUSH = 30;

    /**  */
    private Calendar lastBatchInsert = Calendar.getInstance();
    /** current log buffer to persist */
    private List<AccessLog> logBuffer = new ArrayList<AccessLog>();

    /**
     * @see pt.digitalis.dif.utils.logging.performance.PerformanceLogger#auditRequest(pt.digitalis.dif.utils.logging.performance.LogRequestData,
     *      java.lang.Long, java.lang.String)
     */
    @Override
    protected synchronized void auditRequest(LogRequestData logData, Long timeSpent, String errorDescription)
    {
        // Add the request log to the temporary buffer
        AccessLog access = this.createLogRecord(logData, timeSpent, errorDescription);
        logBuffer.add(access);

        long secondsSinceLastBatchInsert = (Calendar.getInstance().getTimeInMillis()
                - lastBatchInsert.getTimeInMillis()) / 1000;

        // See if we should flush the buffer to the database
        if (logBuffer.size() > MAX_RECORDS_BEFORE_FLUSH || secondsSinceLastBatchInsert > MAX_SECONDS_BEFORE_FLUSH)
        {
            FlushLogbufferToDB writer = new FlushLogbufferToDB(new ArrayList<AccessLog>(logBuffer));
            logBuffer.clear();
            lastBatchInsert = Calendar.getInstance();

            writer.start();
        }
    }

    /**
     * Create a request log for the given parameters.
     *
     * @param logData
     *            the log data
     * @param timeSpent
     *            the time spent
     * @param errorDescription
     *            the error description
     * @return the log record
     */
    protected AccessLog createLogRecord(LogRequestData logData, Long timeSpent, String errorDescription)
    {

        String eventID = logData.getEventID();
        String userID = logData.getUserID();

        // Create the request LOG
        AccessLog access = new AccessLog();
        access.setUserId(userID);
        access.setClientIp(logData.getClientRemoteAddress());
        access.setAccessDate(new Timestamp(new Date().getTime()));
        access.setApplicationId(logData.getApplicationID());
        access.setServiceId(logData.getServiceID());
        access.setStageId(logData.getStageID());
        access.setEventId(eventID);
        access.setRequestType(logData.getRequestType());
        access.setExecutionTime(timeSpent);
        access.setErrorReport(errorDescription);

        if (StringUtils.isNotBlank(errorDescription)
                || PerformanceLoggerConfiguration.getInstance().getAuditRequestsDebugMode())
            access.setRequestDump(logData.getRequestDump());

        return access;
    }
}
