/**
 * 2016, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.ddm;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.hibernate.Session;

import net.sf.json.JSONObject;
import pt.digitalis.dif.dataminer.definition.ChartType;
import pt.digitalis.dif.dataminer.definition.DashboardManager;
import pt.digitalis.dif.dataminer.definition.FilterType;
import pt.digitalis.dif.dataminer.definition.IDashboardPersistence;
import pt.digitalis.dif.dataminer.definition.IIndicator;
import pt.digitalis.dif.dataminer.definition.IIndicator.INDICATOR_ORIGINS;
import pt.digitalis.dif.dataminer.definition.IIndicatorSQL;
import pt.digitalis.dif.dataminer.definition.Series;
import pt.digitalis.dif.dem.managers.impl.model.DIFRepositoryFactory;
import pt.digitalis.dif.dem.managers.impl.model.data.Area;
import pt.digitalis.dif.dem.managers.impl.model.data.Filter;
import pt.digitalis.dif.dem.managers.impl.model.data.Indicator;
import pt.digitalis.dif.dem.managers.impl.model.data.Manager;
import pt.digitalis.dif.dem.managers.impl.model.data.Serie;
import pt.digitalis.dif.dem.managers.impl.model.impl.DDMServiceImpl;
import pt.digitalis.dif.exception.BusinessException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.model.dataset.DataSetException;
import pt.digitalis.dif.model.dataset.IDataSet;
import pt.digitalis.dif.model.dataset.JoinType;
import pt.digitalis.dif.model.dataset.Query;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.NumericUtils;
import pt.digitalis.utils.common.StringUtils;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 03/10/2016
 */
public class DashboardPersistenceBDImpl implements IDashboardPersistence {

    /**
     * Defines the type of DB indicator definition object.
     * <ul>
     * <li>CUSTOMIZATION: An application defined indicator customized by the user</li>
     * <li>DEFINITION: An indicator created by the an user</li>
     * </ul>
     *
     * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
     * @created 27/12/2016
     */
    public enum DEFINITION_TYPES {
        /**  */
        CUSTOMIZATION,
        /**  */
        DEFINITION;

        /**
         * Determines the indicator origyn based on the definition type in DB notation.
         *
         * @param type
         *            the indicator type
         * @return the DB definition type for the indicator origyn
         */
        static IIndicator.INDICATOR_ORIGINS indicatorOrigynForPersistanceDefinition(Character type)
        {
            if (CUSTOMIZATION.getDBRepresentation().equals(type))
                return IIndicator.INDICATOR_ORIGINS.APPLICATION;
            else
                return IIndicator.INDICATOR_ORIGINS.USER;
        }

        /**
         * Determines the type of definition type in DB notation for a given indicator origyn.
         *
         * @param origyn
         *            the indicator origyn
         * @return the DB definition type for the indicator origyn
         */
        static Character persistanceDefinitionTypeForOrigyn(IIndicator.INDICATOR_ORIGINS origyn)
        {
            if (origyn == INDICATOR_ORIGINS.APPLICATION)
                return CUSTOMIZATION.getDBRepresentation();
            else
                return DEFINITION.getDBRepresentation();
        }

        /**
         * @return the character representation of the type for DB persistence
         */
        public Character getDBRepresentation()
        {
            if (this == CUSTOMIZATION)
                return new Character('C');
            else
                return new Character('D');
        }
    }

    /**
     * Converts a JSON from string to Map
     *
     * @param jsonRepresentation
     *            the string to convert to map
     * @return the map
     */
    protected Map<String, String> fromJSONToMap(String jsonRepresentation)
    {
        JSONObject jsonObject = JSONObject.fromObject(jsonRepresentation);
        Map<String, String> dataMap = new HashMap<String, String>();
        @SuppressWarnings("unchecked")
        Iterator<String> keys = jsonObject.keys();

        while (keys.hasNext())
        {
            String key = keys.next();
            String value = jsonObject.getString(key);

            dataMap.put(key, value);
        }

        return dataMap;
    }

    /**
     * Loads an {@link Area} object from the persistent database. If one does not exists creates a new one
     *
     * @param indicator
     *            the indicator to get the DB area for
     * @return the area
     * @throws DataSetException
     */
    protected Area getOrCreateAreaDB(IIndicator indicator) throws DataSetException
    {
        DDMServiceImpl ddmDB = new DDMServiceImpl();
        IDataSet<Area> areaDataset = ddmDB.getAreaDataSet();

        Area areaDB = areaDataset.query().equals(Area.Fields.UNIQUEID, indicator.getAreaID()).singleValue();

        if (areaDB == null)
        {
            DashboardManager manager = DashboardManager.getInstance(indicator.getManagerID());
            pt.digitalis.dif.dataminer.definition.Area area = manager.getAreas().get(indicator.getAreaID());

            areaDB = new Area();
            areaDB.setUniqueId(indicator.getAreaID());
            areaDB.setTitle(area.getTitle());
            areaDB.setDatabasePass(area.getDatabasePassword());
            areaDB.setDatabaseUrl(area.getDatabaseURL());
            areaDB.setDatabaseUser(area.getDatabaseUsername());
            areaDB.setManager(this.getOrCreateManagerDB(manager));

            areaDB = areaDataset.insert(areaDB);
        }

        return areaDB;
    }

    /**
     * Loads an {@link Manager} object from the persistent database. If one does not exists creates a new one
     *
     * @param manager
     *            the manager to get the DB object for
     * @return the manager
     * @throws DataSetException
     */
    private Manager getOrCreateManagerDB(DashboardManager manager) throws DataSetException
    {
        DDMServiceImpl ddmDB = new DDMServiceImpl();
        IDataSet<Manager> managerDataset = ddmDB.getManagerDataSet();

        Manager managerDB = managerDataset.query().equals(Manager.Fields.UNIQUEID, manager.getId()).singleValue();

        if (managerDB == null)
        {
            managerDB = new Manager();
            managerDB.setUniqueId(manager.getId());

            managerDB = managerDataset.insert(managerDB);
        }

        return managerDB;
    }

    /**
     * @see pt.digitalis.dif.dataminer.definition.IDashboardPersistence#isAvailable()
     */
    public boolean isAvailable()
    {
        return true;
    }

    /**
     * Load an area from the BD
     *
     * @param areaDB
     *            the area to load
     * @param area
     *            the destination area instance to fill with the DB configuration
     * @param dashboardManager
     *            the dashboard manager that contains the indicator
     */
    protected void loadAreaFromDB(Area areaDB, pt.digitalis.dif.dataminer.definition.Area area,
            DashboardManager dashboardManager)
    {
        // set/override all original fields with custom one
        if (StringUtils.isNotBlank(areaDB.getRestrictGroups()))
            area.setDefaultGroupsToRestrict(Arrays.asList(areaDB.getRestrictGroups().split(",")));
        else
            area.setDefaultGroupsToRestrict(null);

        if (StringUtils.isNotBlank(areaDB.getRestrictProfiles()))
            area.setDefaultProfilesToRestrict(Arrays.asList(areaDB.getRestrictProfiles().split(",")));
        else
            area.setDefaultProfilesToRestrict(null);
    }

    /**
     * @see pt.digitalis.dif.dataminer.definition.IDashboardPersistence#loadConfigurations(pt.digitalis.dif.dataminer.definition.DashboardManager)
     */
    public DashboardManager loadConfigurations(DashboardManager dashboardManager) throws BusinessException
    {
        try
        {
            int customizedAreas = 0;
            int customizedIndicators = 0;
            int userIndicators = 0;

            DIFLogger.getLogger()
                    .info("Loading Dashboard Manager \"" + dashboardManager.getId() + "\" ("
                            + dashboardManager.getTotalIndicators() + " indicator(s) in "
                            + dashboardManager.getTotalAreas() + " area(s))");

            DDMServiceImpl ddmDB = new DDMServiceImpl();

            // Load area customizations...
            Query<Area> queryAreas = ddmDB.getAreaDataSet().query().equals(Area.FK().manager().UNIQUEID(),
                    dashboardManager.getId());

            for (Area areaDB: queryAreas.asList())
            {
                DIFLogger.getLogger().debug("  - Customized area [" + areaDB.getId() + "] " + areaDB.getUniqueId()
                        + " - " + areaDB.getTitle());

                customizedAreas++;

                // Customize existing indicator
                pt.digitalis.dif.dataminer.definition.Area area = dashboardManager.getAreas().get(areaDB.getUniqueId());
                this.loadAreaFromDB(areaDB, area, dashboardManager);
            }

            // Load indicator customizations...
            Query<Indicator> queryIndicators = ddmDB.getIndicatorDataSet().query()
                    .equals(Indicator.FK().area().manager().UNIQUEID(), dashboardManager.getId())
                    .addJoin(Indicator.FK().series(), JoinType.LEFT_OUTER_JOIN)
                    .addJoin(Indicator.FK().filters(), JoinType.LEFT_OUTER_JOIN);
            queryIndicators.setDistinctEntities(true);

            for (Indicator indicatorDB: queryIndicators.asList())
            {
                boolean customization = DEFINITION_TYPES.CUSTOMIZATION.getDBRepresentation()
                        .equals(indicatorDB.getDefinitionType());

                DIFLogger.getLogger().debug("  - " + (customization ? "Customized" : "User defined") + " indicator ["
                        + indicatorDB.getId() + "] " + indicatorDB.getUniqueId() + " - " + indicatorDB.getTitle());

                if (DEFINITION_TYPES.CUSTOMIZATION.getDBRepresentation().equals(indicatorDB.getDefinitionType()))
                    customizedIndicators++;
                else
                    userIndicators++;

                // Get indicator area
                pt.digitalis.dif.dataminer.definition.Area area = dashboardManager.getAreas()
                        .get(indicatorDB.getArea().getUniqueId());

                if (customization)
                {
                    // Customize existing indicator
                    IIndicator indicator = area.getIndicators().get(indicatorDB.getUniqueId());
                    this.loadIndicatorFromDB(indicatorDB, indicator, dashboardManager);
                }
                else
                {
                    // load new user defined indicator
                    IIndicator indicator = DIFIoCRegistry.getRegistry().getImplementation(IIndicatorSQL.class);
                    indicator.setAreaID(area.getId());
                    indicator.setId(indicatorDB.getUniqueId());

                    this.loadIndicatorFromDB(indicatorDB, indicator, dashboardManager);

                    area.addIndicator(indicator);
                }

            }

            if (customizedAreas > 0)
                DIFLogger.getLogger().info("      " + customizedAreas + " customized areas");
            if (customizedIndicators > 0)
                DIFLogger.getLogger().info("      " + customizedIndicators + " customized indicators");
            if (userIndicators > 0)
                DIFLogger.getLogger().info("      " + userIndicators + " user defined indicators");
            if (customizedIndicators == 0 && userIndicators == 0)
                DIFLogger.getLogger().info("    No customization found. Default configuration active.");
        }
        catch (DataSetException e)
        {
            BusinessException exception = new BusinessException(
                    "Error loading dashboard customizations from the database", e);
            exception.addToExceptionContext("Manager", dashboardManager);

            throw exception;
        }

        return dashboardManager;
    }

    /**
     * Load an indicator from the BD
     *
     * @param indicatorDB
     *            the indicator to load
     * @param indicator
     *            the destination indicator instance to fill with the DB configuration
     * @param dashboardManager
     *            the dashboard manager that contains the indicator
     */
    protected void loadIndicatorFromDB(Indicator indicatorDB, IIndicator indicator, DashboardManager dashboardManager)
    {
        // set/override all original fields with custom one
        indicator.setAutoRefreshInterval(indicatorDB.getAutoRefreshInt());
        indicator.setxAxisTitle(indicatorDB.getAxisXTitle());
        indicator.setyAxisTitle(indicatorDB.getAxisYTitle());
        indicator.setOrigyn(DEFINITION_TYPES.indicatorOrigynForPersistanceDefinition(indicatorDB.getDefinitionType()));
        indicator.setDescription(indicatorDB.getDescription());
        indicator.setDescriptionTitle(indicatorDB.getDescriptionTitle());
        indicator.setGroupTitle(indicatorDB.getGroupTitle());
        indicator.setHideMarkers(indicatorDB.isHideMarkers());
        indicator.setLegend(indicatorDB.isLegend());
        indicator.setLimitTopRecords(
                indicatorDB.getLimitTopRecords() == null ? null : indicatorDB.getLimitTopRecords().intValue());
        indicator.setMaxValue(indicatorDB.getMaxValue());

        if (StringUtils.isNotBlank(indicatorDB.getRestrictGroups()))
            indicator.setRestrictToGroups(Arrays.asList(indicatorDB.getRestrictGroups().split(",")));

        if (StringUtils.isNotBlank(indicatorDB.getRestrictProfiles()))
            indicator.setRestrictToProfiles(Arrays.asList(indicatorDB.getRestrictProfiles().split(",")));

        indicator.setTimeKeys(indicatorDB.isTimeKeys());
        indicator.setTitle(indicatorDB.getTitle());
        indicator.setHasTotalField(indicatorDB.isTotalField());
        indicator.setUnitSuffix(indicatorDB.getUnitSuffix());
        indicator.setUseMinutes(indicatorDB.isUseMinutes());

        if (indicator instanceof IIndicatorSQL)
        {
            IIndicatorSQL sqlIndicator = (IIndicatorSQL) indicator;
            sqlIndicator.setQuerySQL(indicatorDB.getQuerySql());
            sqlIndicator.setMaterializedViews(indicatorDB.getViewsToRefresh());
        }

        // Process series
        // -----------------------------------------------------

        // identify series by value field
        Map<String, Series> seriesByValueField = new HashMap<String, Series>();
        for (Series serie: indicator.getSeries())
            seriesByValueField.put(serie.getValueField(), serie);

        Map<String, Serie> seriesBDByValueField = new HashMap<String, Serie>();
        for (Serie serie: indicatorDB.getSeries())
            seriesBDByValueField.put(serie.getValueField(), serie);

        // remove discarded series
        // Check DB series definitions
        for (Series serie: seriesByValueField.values())
        {
            if (!seriesBDByValueField.containsKey(serie.getValueField()))
            {
                // if it is no longer in the DB, remove from the indicator series list
                indicator.getSeries().remove(serie);
            }
        }

        // insert/update series
        for (Serie serieDB: indicatorDB.getSeries())
        {
            Series serie = seriesByValueField.get(serieDB.getValueField());

            if (serie != null)
            {
                // update
                serie.setDescription(serieDB.getDescription());
                serie.setChartType(ChartType.fromCharacterRepresentation(serieDB.getType()));
                serie.setUnitName(serieDB.getUnitName());
            }
            else
            {
                // insert
                serie = new Series(ChartType.fromCharacterRepresentation(serieDB.getType()), serieDB.getUnitName(),
                        serieDB.getDescription(), serieDB.getValueField());
                indicator.getSeries().add(serie);
            }
        }

        // Process filters
        // -----------------------------------------------------
        // identify filters by uniqueID
        Map<String, pt.digitalis.dif.dataminer.definition.Filter> filtersByUniqueID = new HashMap<String, pt.digitalis.dif.dataminer.definition.Filter>();
        for (pt.digitalis.dif.dataminer.definition.Filter filter: indicator.getFilters())
            filtersByUniqueID.put(filter.getId(), filter);

        Map<String, Filter> filtersDBByUniqueID = new HashMap<String, Filter>();
        for (Filter filter: indicatorDB.getFilters())
            filtersDBByUniqueID.put(filter.getUniqueId(), filter);

        // discard no longer used filters
        for (pt.digitalis.dif.dataminer.definition.Filter filter: filtersByUniqueID.values())
        {
            // No longer associated with this indicator
            if (filtersDBByUniqueID.get(filter.getId()) == null)
            {
                indicator.getFilters().remove(filtersDBByUniqueID.get(filter.getId()));
            }
        }

        // update older filters / insert new ones
        for (Filter filterDB: indicatorDB.getFilters())
        {
            pt.digitalis.dif.dataminer.definition.Filter filter = filtersByUniqueID.get(filterDB.getUniqueId());
            boolean newFilter = false;

            if (filter == null)
            {
                newFilter = true;

                filter = new pt.digitalis.dif.dataminer.definition.Filter(filterDB.getUniqueId(),
                        FilterType.fromCharacterRepresentation(filterDB.getType()), filterDB.getTitle(),
                        filterDB.getDescription(), filterDB.getSqlTemplate());
            }

            filter.setType(FilterType.fromCharacterRepresentation(filterDB.getType()));
            filter.setBindToProfileID(filterDB.getBindToProfile());
            filter.setDefaultValue(filterDB.getDefaultValue());
            filter.setDescription(filterDB.getDescription());

            if (StringUtils.isNotBlank(filterDB.getItemsList()))
                filter.setLovItems(this.fromJSONToMap(filterDB.getItemsList()));

            filter.setLovQuerySQL(filterDB.getItemsQuerySql());
            filter.setFilterSQLTemplate(filterDB.getSqlTemplate());
            filter.setTitle(filterDB.getTitle());

            if (newFilter)
                indicator.getFilters().add(filter);
        }
    }

    /**
     * @see pt.digitalis.dif.dataminer.definition.IDashboardPersistence#removeIndicatorConfiguration(pt.digitalis.dif.dataminer.definition.IIndicator)
     */
    public void removeIndicatorConfiguration(IIndicator indicator) throws BusinessException
    {
        this.removeIndicatorConfiguration(indicator.getId());
    }

    /**
     * @see pt.digitalis.dif.dataminer.definition.IDashboardPersistence#removeIndicatorConfiguration(java.lang.String)
     */
    public void removeIndicatorConfiguration(String indicatorID) throws BusinessException
    {
        try
        {
            DDMServiceImpl ddmDB = new DDMServiceImpl();
            IDataSet<Indicator> ds = ddmDB.getIndicatorDataSet();

            Indicator indicatorDB = ds.query().equals(Indicator.Fields.UNIQUEID, indicatorID)
                    .addJoin(Indicator.FK().series(), JoinType.LEFT_OUTER_JOIN)
                    .addJoin(Indicator.FK().filters(), JoinType.LEFT_OUTER_JOIN).singleValue();

            if (indicatorDB != null)
            {
                // Delete indicator series
                for (Serie serie: indicatorDB.getSeries())
                    ddmDB.getSerieDataSet().delete(serie.getId().toString());

                // Delete indicator filters associations (along with this indicator's orphaned filters)
                for (Filter filter: indicatorDB.getFilters())
                    // Remove filter from indicator
                    ddmDB.getFilterDataSet().delete(filter.getId().toString());

                // Delete indicator
                ds.delete(indicatorDB.getId().toString());
            }
        }
        catch (DataSetException e)
        {
            BusinessException exception = new BusinessException(
                    "Error deleting indicator customization from the database", e);
            exception.addToExceptionContext("Indicator ID", indicatorID);

            throw exception;
        }
    }

    /**
     * @see pt.digitalis.dif.dataminer.definition.IDashboardPersistence#saveAreaConfiguration(pt.digitalis.dif.dataminer.definition.Area)
     */
    public void saveAreaConfiguration(pt.digitalis.dif.dataminer.definition.Area area) throws BusinessException
    {
        try
        {
            DDMServiceImpl ddmDB = new DDMServiceImpl();
            IDataSet<Area> ds = ddmDB.getAreaDataSet();
            boolean newRecord = false;
            Session session = DIFRepositoryFactory.getSession();
            boolean transactionWasActive = session.getTransaction() != null && session.getTransaction().isActive();

            if (!transactionWasActive)
                session.beginTransaction();

            // Load existing indicator in DB to update (if any)
            Area areaDB = ds.query().equals(Area.Fields.UNIQUEID, area.getId()).singleValue();

            // If it does not exist, create a new record
            if (areaDB == null)
            {
                newRecord = true;

                areaDB = new Area();
                areaDB.setUniqueId(area.getId());
                areaDB.setDatabasePass(area.getDatabasePassword());
                areaDB.setDatabaseUrl(area.getDatabaseURL());
                areaDB.setDatabaseUser(area.getDatabaseUsername());
                areaDB.setManager(this.getOrCreateManagerDB(DashboardManager.getInstance(area.getManagerID())));
            }

            // set/override all (previous) fields
            areaDB.setTitle(area.getTitle());

            if (area.getDefaultGroupsToRestrict() != null)
                areaDB.setRestrictGroups(CollectionUtils.listToCommaSeparatedString(area.getDefaultGroupsToRestrict()));
            else
                areaDB.setRestrictGroups(null);

            if (area.getDefaultProfilesToRestrict() != null)
                areaDB.setRestrictProfiles(
                        CollectionUtils.listToCommaSeparatedString(area.getDefaultProfilesToRestrict()));
            else
                areaDB.setRestrictProfiles(null);

            // Persist indicator to the database
            if (newRecord)
                areaDB = ds.insert(areaDB);
            else
                ds.update(areaDB);

            if (!transactionWasActive)
                session.getTransaction().commit();
        }
        catch (DataSetException e)
        {
            BusinessException exception = new BusinessException("Error saving area customizations to the database", e);
            exception.addToExceptionContext("Area", area);

            throw exception;
        }
    }

    /**
     * @see pt.digitalis.dif.dataminer.definition.IDashboardPersistence#saveIndicatorConfiguration(pt.digitalis.dif.dataminer.definition.IIndicator)
     */
    public void saveIndicatorConfiguration(IIndicator indicator) throws BusinessException
    {
        try
        {
            DDMServiceImpl ddmDB = new DDMServiceImpl();
            IDataSet<Indicator> ds = ddmDB.getIndicatorDataSet();
            boolean newRecord = false;
            Session session = DIFRepositoryFactory.getSession();
            boolean transactionWasActive = session.getTransaction() != null && session.getTransaction().isActive();

            if (!transactionWasActive)
                session.beginTransaction();

            // Load existing indicator in DB to update (if any)
            Indicator indicatorDB = ds.query().equals(Indicator.Fields.UNIQUEID, indicator.getId().toString())
                    .addJoin(Indicator.FK().series(), JoinType.LEFT_OUTER_JOIN)
                    .addJoin(Indicator.FK().filters(), JoinType.LEFT_OUTER_JOIN).singleValue();

            // If it does not exist, create a new record
            if (indicatorDB == null)
            {
                newRecord = true;

                indicatorDB = new Indicator();
                indicatorDB.setUniqueId(indicator.getId());
                indicatorDB.setArea(this.getOrCreateAreaDB(indicator));
            }

            // set/override all (previous) fields
            indicatorDB.setAutoRefreshInt(indicator.getAutoRefreshInterval());
            indicatorDB.setAxisXTitle(indicator.getxAxisTitle());
            indicatorDB.setAxisYTitle(indicator.getyAxisTitle());
            indicatorDB.setDefinitionType(DEFINITION_TYPES.persistanceDefinitionTypeForOrigyn(indicator.getOrigyn()));
            indicatorDB.setDescription(indicator.getDescription());
            indicatorDB.setDescriptionTitle(indicator.getDescriptionTitle());
            indicatorDB.setGroupTitle(indicator.getGroupTitle());
            indicatorDB.setHideMarkers(indicator.isHideMarkers());
            indicatorDB.setLegend(indicator.isLegend());
            indicatorDB.setLimitTopRecords(NumericUtils.toLong(indicator.getLimitTopRecords()));
            indicatorDB.setMaxValue(indicator.getMaxValue());

            if (indicator.getRestrictToGroups() != null)
                indicatorDB
                        .setRestrictGroups(CollectionUtils.listToCommaSeparatedString(indicator.getRestrictToGroups()));
            else
                indicatorDB.setRestrictGroups(null);

            if (indicator.getRestrictToProfiles() != null)
                indicatorDB.setRestrictProfiles(
                        CollectionUtils.listToCommaSeparatedString(indicator.getRestrictToProfiles()));
            else
                indicatorDB.setRestrictProfiles(null);

            indicatorDB.setTimeKeys(indicator.isTimeKeys());
            indicatorDB.setTitle(indicator.getTitle());
            indicatorDB.setTotalField(indicator.isHasTotalField());
            indicatorDB.setUnitSuffix(indicator.getUnitSuffix());
            indicatorDB.setUseMinutes(indicator.isUseMinutes());

            if (indicator instanceof IIndicatorSQL)
            {
                IIndicatorSQL sqlIndicator = (IIndicatorSQL) indicator;
                indicatorDB.setQuerySql(sqlIndicator.getQuerySQL());
                indicatorDB.setViewsToRefresh(sqlIndicator.getMaterializedViews());
            }

            // Persist indicator to the database
            if (newRecord)
                indicatorDB = ds.insert(indicatorDB);
            else
                ds.update(indicatorDB);

            // Process series
            // -----------------------------------------------------

            // identify series by value field
            Map<String, Series> seriesByValueField = new HashMap<String, Series>();
            for (Series serie: indicator.getSeries())
                seriesByValueField.put(serie.getValueField(), serie);

            Map<String, Serie> seriesBDByValueField = new HashMap<String, Serie>();
            for (Serie serie: indicatorDB.getSeries())
                seriesBDByValueField.put(serie.getValueField(), serie);

            // discard no longer used series
            if (!newRecord)
            {
                // previous record exists, remove discarded series
                // Check DB series definitions
                for (Serie serie: indicatorDB.getSeries())
                    if (!seriesByValueField.containsKey(serie.getValueField()))
                    {
                        // if it is no longer is the indicator series list, remove from BD
                        ddmDB.getSerieDataSet().delete(serie.getId().toString());
                        seriesBDByValueField.remove(serie);
                    }
            }

            // insert/update series
            for (Series serie: indicator.getSeries())
            {
                Serie serieDB = seriesBDByValueField.get(serie.getValueField());

                if (serieDB != null)
                {
                    // update
                    serieDB.setDescription(serie.getDescription());
                    serieDB.setType(serie.getChartType().getCharacterRepresentation());
                    serieDB.setUnitName(serie.getUnitName());

                    ddmDB.getSerieDataSet().update(serieDB);
                }
                else
                {
                    // insert
                    serieDB = new Serie();
                    serieDB.setDescription(serie.getDescription());
                    serieDB.setIndicator(indicatorDB);
                    serieDB.setType(serie.getChartType().getCharacterRepresentation());
                    serieDB.setUnitName(serie.getUnitName());
                    serieDB.setValueField(serie.getValueField());

                    serieDB = ddmDB.getSerieDataSet().insert(serieDB);
                }
            }

            // Process filters
            // -----------------------------------------------------
            // identify filters by uniqueID
            Map<String, pt.digitalis.dif.dataminer.definition.Filter> filtersByUniqueID = new HashMap<String, pt.digitalis.dif.dataminer.definition.Filter>();
            for (pt.digitalis.dif.dataminer.definition.Filter filter: indicator.getFilters())
                filtersByUniqueID.put(filter.getId(), filter);

            Map<String, Filter> filtersDBByUniqueID = new HashMap<String, Filter>();
            for (Filter filter: indicatorDB.getFilters())
                filtersDBByUniqueID.put(filter.getUniqueId(), filter);

            // discard no longer used filters
            for (Filter filterDB: filtersDBByUniqueID.values())
            {
                // No longer associated with this indicator
                if (filtersByUniqueID.get(filterDB.getUniqueId()) == null)
                {
                    ddmDB.getFilterDataSet().delete(filterDB.getId().toString());
                    filtersDBByUniqueID.remove(filterDB);
                }
            }

            // update older filters / insert new ones
            for (pt.digitalis.dif.dataminer.definition.Filter filter: indicator.getFilters())
            {
                Filter filterDB = filtersDBByUniqueID.get(filter.getId());
                boolean newFilter = false;

                if (filterDB == null)
                {
                    newFilter = true;

                    filterDB = new Filter();
                    filterDB.setUniqueId(filter.getId());
                    filterDB.setIndicator(indicatorDB);
                }

                filterDB.setType(filter.getType().getCharacterRepresentation());
                filterDB.setBindToProfile(filter.getBindToProfileID());
                filterDB.setDefaultValue(filter.getDefaultValue());
                filterDB.setDescription(filter.getDescription());

                if (filter.getLovItems() != null && !filter.getLovItems().isEmpty())
                    filterDB.setItemsList(JSONObject.fromObject(filter.getLovItems()).toString());

                filterDB.setItemsQuerySql(filter.getLovQuerySQL());
                filterDB.setSqlTemplate(filter.getFilterSQLTemplate());
                filterDB.setTitle(filter.getTitle());

                if (newFilter)
                    ddmDB.getFilterDataSet().insert(filterDB);
                else
                    ddmDB.getFilterDataSet().update(filterDB);
            }

            if (!transactionWasActive)
                session.getTransaction().commit();
        }
        catch (DataSetException e)
        {
            BusinessException exception = new BusinessException("Error saving indicator customizations to the database",
                    e);
            exception.addToExceptionContext("Indicator", indicator);

            throw exception;
        }
    }
}
