/**
 * 2019, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 http://www.digitalis.pt
 */
package pt.digitalis.dif.servermanager;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import pt.digitalis.dif.exception.BusinessException;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.HttpRequestResult;
import pt.digitalis.utils.common.HttpUtils;
import pt.digitalis.utils.crypto.exeption.CryptoException;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created May 10, 2019
 */
public class NotifyServersOfNewMessagesToProcess extends Thread {

    /** The servers list. */
    private List<ServerApplicationNodeInstance> servers = new ArrayList<ServerApplicationNodeInstance>();

    /**
     * Instantiates a new notify servers of new messages to process.
     *
     * @param servers
     *            the servers
     */
    public NotifyServersOfNewMessagesToProcess(List<ServerApplicationNodeInstance> servers)
    {
        super();
        this.servers = servers;
    }

    /**
     * Notify server.
     *
     * @param server
     *            the server
     * @param timeOut
     * @return true, if successful
     * @throws CryptoException
     */
    private boolean notifyServer(ServerApplicationNodeInstance server, int timeOut) throws CryptoException
    {
        // Encrypt the parameter for safety
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put("serverURL", ServerManagerUtils.encryptValue(server.getServerBaseURL()));

        DIFLogger.getLogger().debug("Notifying server \"" + server.getServerBaseURL() + "\" of new messages...");

        // Make the HTTP Request to the server
        try
        {
            HttpRequestResult response = HttpUtils.makeHttpRequest(
                    server.getServerBaseURL() + "/ajax/servermanagerproxy/messagesavailable", "GET", parameters,
                    timeOut);

            // HTTP.200 all OK, otherwise WARN to log
            DIFLogger.getLogger().log((response.getStatus() == 200 ? LogLevel.INFO : LogLevel.WARN),
                    "Messages available trigger sent to " + server.getServerBaseURL() + " (response ["
                            + response.getStatus() + "] - " + response.getResponse() + ")");

            return true;
        }
        catch (IOException e)
        {
            new BusinessException("Cannot notify server", e).addToExceptionContext("server", server).log(LogLevel.WARN);

            return false;
        }
    }

    /**
     * @see java.lang.Thread#run()
     */
    @Override
    public void run()
    {
        // Servers to retry notification
        List<ServerApplicationNodeInstance> failedServers = new ArrayList<ServerApplicationNodeInstance>();

        for (ServerApplicationNodeInstance server: servers)
        {
            // Notify server: We want to notify all servers fast. Slow ones to respond will be retried in a second pass
            try
            {
                boolean result = this.notifyServer(server, 15 * 1000); // 15 seconds timeout

                if (!result)
                    failedServers.add(server);
            }
            catch (CryptoException e)
            {
                e.printStackTrace();
            }
        }

        // Second pass: Will wait patiently for each one. Can be restarting for instance...
        for (ServerApplicationNodeInstance server: failedServers)
        {
            try
            {
                boolean result = this.notifyServer(server, 10 * 60 * 1000); // 10 minutes timeout

                if (!result)
                    failedServers.add(server);
            }
            catch (CryptoException e)
            {
                e.printStackTrace();
            }
        }
    }
}
