/**
 * 2019, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 http://www.digitalis.pt
 */
package pt.digitalis.dif.configurations;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import pt.digitalis.dif.controller.interfaces.IModelManager;
import pt.digitalis.dif.dem.managers.impl.model.DIFRepositoryFactory;
import pt.digitalis.dif.dem.managers.impl.model.IConfigurationsService;
import pt.digitalis.dif.dem.managers.impl.model.IServersService;
import pt.digitalis.dif.dem.managers.impl.model.data.ConfigNode;
import pt.digitalis.dif.dem.managers.impl.model.data.ConfigServerSet;
import pt.digitalis.dif.dem.managers.impl.model.data.ConfigSet;
import pt.digitalis.dif.dem.managers.impl.model.data.Configuration;
import pt.digitalis.dif.dem.managers.impl.model.data.Server;
import pt.digitalis.dif.exception.BusinessException;
import pt.digitalis.dif.exception.InternalFrameworkException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.model.dataset.DataSetException;
import pt.digitalis.dif.model.dataset.IDataSet;
import pt.digitalis.dif.model.dataset.JoinType;
import pt.digitalis.dif.model.dataset.Query;
import pt.digitalis.dif.model.dataset.SortMode;
import pt.digitalis.dif.model.hibernate.HibernateUtil;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.dif.startup.DIFInitializer;
import pt.digitalis.dif.startup.DIFStartupConfiguration;
import pt.digitalis.dif.utils.logging.AuditContext;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.dif.utils.logging.LoggingConfiguration;
import pt.digitalis.dif.utils.logging.performance.PerformanceLoggerConfiguration;
import pt.digitalis.log.LogLevel;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.AbstractConfigurationsImpl;
import pt.digitalis.utils.config.ConfigurationException;
import pt.digitalis.utils.config.ConfigurationsPreferencesImpl;
import pt.digitalis.utils.config.IConfigurations;

/**
 * The Class ConfigurationDatabaseImpl.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Mar 28, 2019
 */
public class ConfigurationsDatabaseImpl extends AbstractConfigurationsImpl {

    /** The server UID. */
    private static String serverUID = null;

    /** The configurations service cache. */
    private IConfigurationsService configurationsService = null;

    /**
     * Instantiates a new configurations database impl.
     */
    public ConfigurationsDatabaseImpl()
    {
        this(true);
    }

    /**
     * Instantiates a new configurations database impl.
     *
     * @param ignoreTestConfigurations
     *            the ignore test configurations
     */
    public ConfigurationsDatabaseImpl(boolean ignoreTestConfigurations)
    {
        super(ignoreTestConfigurations);
    }

    /**
     * Convert record to properties.
     *
     * @param configProps
     *            the config props
     * @return the properties
     */
    private Properties convertRecordToProperties(List<Configuration> configProps)
    {
        Properties props = new Properties();

        for (Configuration config: configProps)
        {
            String value = config.getValue();
            if (value == null)
                value = config.getValueXL();

            if (value != null)
                props.setProperty(config.getKey(), value);
        }

        return props;
    }

    /**
     * Convert record to properties.
     *
     * @param configurations
     *            the configurations
     * @return the properties
     */
    private Properties convertRecordToProperties(Set<Configuration> configurations)
    {
        return this.convertRecordToProperties(new ArrayList<Configuration>(configurations));
    }

    /**
     * Gets the configurations DB.
     *
     * @return the configurations DB
     */
    protected IConfigurationsService getDB()
    {
        this.initializeDBAuditContext();

        if (configurationsService == null)
            configurationsService = DIFIoCRegistry.getRegistry().getImplementation(IConfigurationsService.class);

        return configurationsService;
    }

    /**
     * Gets the or create node.
     *
     * @param nodePath
     *            the node path
     * @return the or create node
     * @throws DataSetException
     *             the data set exception
     */
    private ConfigNode getOrCreateNode(String nodePath) throws DataSetException
    {
        this.initializeDBAuditContext();
        boolean wasActive = DIFRepositoryFactory.openTransaction();

        ConfigNode node = this.getDB().getConfigNodeDataSet().query()
                .equals(ConfigNode.FK().configSet().configServerSets().SERVERUID(), this.getServerUID())
                .sortBy(ConfigNode.Fields.ID, SortMode.ASCENDING).equals(ConfigNode.Fields.NODEPATH, nodePath)
                .singleValue();

        if (node != null)
        {
            if (!wasActive)
                DIFRepositoryFactory.getSession().getTransaction().commit();

            return node;
        }
        else
        {
            // Does not exist yet => Create it

            // Determine the correct set of the server to write to
            List<ConfigServerSet> serverSets = this.getDB().getConfigServerSetDataSet().query()
                    .equals(ConfigServerSet.Fields.SERVERUID, this.getServerUID())
                    .addJoin(ConfigServerSet.FK().configSet(), JoinType.NORMAL)
                    .sortBy(ConfigServerSet.Fields.ID, SortMode.ASCENDING).asList();
            ConfigServerSet serverSet = null;

            if (serverSets == null || serverSets.isEmpty())
            {
                // None exist... Create a specific one for this server with a specific set
                // Create a specific set for this server
                ConfigSet newConfigSet = new ConfigSet();
                newConfigSet.setName(this.getServerUID());
                newConfigSet.setDescription("Auto generated Configurations Set for: " + this.getServerUID());
                newConfigSet = this.getDB().getConfigSetDataSet().insert(newConfigSet);

                // Create a specific server for this server with the previous set
                serverSet = new ConfigServerSet();
                serverSet.setConfigSet(newConfigSet);
                serverSet.setServerUid(this.getServerUID());

                serverSet = this.getDB().getConfigServerSetDataSet().insert(serverSet);
            }
            else if (serverSets.size() > 1)
            {
                // More than one... try to find the specific set created for this server (serverUID = set.name)
                for (ConfigServerSet currentServerSet: serverSets)
                {
                    if (currentServerSet.getConfigSet().getName().equals(this.getServerUID()))
                    {
                        serverSet = currentServerSet;
                        break;
                    }
                }

                // Specific not found (serverUID must have changed in time... select the first one, no other choice!
                if (serverSet == null)
                    serverSet = serverSets.get(0);
            }
            else
            {
                // Only one exists, select it...
                serverSet = serverSets.get(0);
            }

            ConfigNode newNode = new ConfigNode();
            newNode.setConfigSet(serverSet.getConfigSet());
            newNode.setNodePath(nodePath);
            newNode = this.getDB().getConfigNodeDataSet().insert(newNode);

            if (!wasActive)
                DIFRepositoryFactory.getSession().getTransaction().commit();

            return newNode;
        }
    }

    /**
     * Gets the configuration point prefix.
     *
     * @return the configuration point prefix
     */
    public String getServerUID()
    {
        return serverUID;
    }

    /**
     * Initialize.
     *
     * @throws ConfigurationException
     *             the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#initialize()
     */
    public void initialize() throws ConfigurationException
    {
        this.initializeDBAuditContext();

        // Read current configuration module
        String currentConfigurationModule = IConfigurations.class.getCanonicalName();
        IConfigurations oldConfigurationsImpl = null;

        boolean migrateConfigs = false;

        if (StringUtils.isBlank(DIFStartupConfiguration.getModulesProps().getProperty(currentConfigurationModule)))
        {
            // No module defined and the this one is initializing => Means we want to use database configurations and
            // it's the first time we are using it. Do all the initializations and migrations for first time use

            // Migrate DB configs
            oldConfigurationsImpl = new ConfigurationsPreferencesImpl();
            Properties prefsDBConfig = oldConfigurationsImpl.readConfiguration(HibernateUtil.REPOSITORY_CONFIG_ID,
                    HibernateUtil.REPOSITORY_SECTION_ID + "/" + DIFRepositoryFactory.SESSION_FACTORY_NAME);
            DIFStartupConfiguration.setDbProps(prefsDBConfig);

            migrateConfigs = true;
        }

        // Initialize the Model in a controlled fashion (if not getSession may be called with will trigger the model
        // reading it's database configurations from the default API location, the ConfigurationsAPI
        IModelManager difModelManager = DIFIoCRegistry.getRegistry().getImplementation(IModelManager.class, "DIF");
        try
        {
            DIFInitializer.processModelManager(difModelManager);
            HibernateUtil.getSessionFactory(DIFRepositoryFactory.SESSION_FACTORY_NAME, null,
                    DIFStartupConfiguration.getDbProps());
        }
        catch (InternalFrameworkException e)
        {
            throw new ConfigurationException(e);
        }

        String serverUIDtemp = DIFStartupConfiguration.getMachineIDForConfigurations() + "|"
                + AbstractConfigurationsImpl.generalConfigurationPrefix;

        // Initialize the serverUID (check replica logic)
        if (DIFStartupConfiguration.getReplicaMode())
        {
            String serverReplicaUIDCalc = DIFStartupConfiguration.getReplicaModeAlternateMachineIDForConfigurations();

            if (StringUtils.isBlank(serverReplicaUIDCalc)
                    || DIFStartupConfiguration.EMPTY_PARAM_SLOT.equals(serverReplicaUIDCalc))
            {
                try
                {

                    IConfigurationsService configsService = DIFIoCRegistry.getRegistry()
                            .getImplementation(IConfigurationsService.class);
                    boolean hasConfigurationsForReplicaInDB = configsService.getConfigServerSetDataSet().query()
                            .equals(ConfigServerSet.Fields.SERVERUID, serverUIDtemp).count() > 0;

                    if (!hasConfigurationsForReplicaInDB)
                    {
                        // We have no parameters in DB for the serverUID. Probably the DB was overridden by the
                        // production one.
                        DIFLogger.getLogger().info("ReplicaMode: No parameters found in DB");

                        // Determine first server for this application (same context root) that is not this one.
                        // Hopefully will determine the production application and use it's parameters
                        Server productionServer = DIFIoCRegistry.getRegistry().getImplementation(IServersService.class)
                                .getServerDataSet().query()
                                .like(Server.Fields.CONTEXTROOT, AbstractConfigurationsImpl.generalConfigurationPrefix)
                                .notEquals(Server.Fields.MACHINESERVERUID,
                                        DIFStartupConfiguration.getMachineIDForConfigurations())
                                .singleValue();

                        if (productionServer != null)
                        {
                            DIFLogger.getLogger().info("ReplicaMode: Found an alternative server for \""
                                    + AbstractConfigurationsImpl.generalConfigurationPrefix + "\"");

                            // Alternative server found. Apply it's parameters to the replica.
                            serverUIDtemp = productionServer.getMachineServerUid() + "|"
                                    + AbstractConfigurationsImpl.generalConfigurationPrefix;

                            // Apply configuration for next restarts
                            DIFStartupConfiguration.setReplicaModeAlternateMachineIDForConfigurations(
                                    productionServer.getMachineServerUid());
                            DIFStartupConfiguration.updateConfig();

                            DIFLogger.getLogger()
                                    .info("ReplicaMode: Using alternative (production?) server for configurations \""
                                            + productionServer.getMachineServerUid() + "\"");
                        }
                    }
                }
                catch (DataSetException e)
                {
                    throw new ConfigurationException(e);
                }
            }
            else
                serverUIDtemp = serverReplicaUIDCalc + "|" + AbstractConfigurationsImpl.generalConfigurationPrefix;
        }

        serverUID = serverUIDtemp;

        if (migrateConfigs && oldConfigurationsImpl != null)
        {
            // Migrate from Preferences
            boolean configurationsMigrated = migrateFromOtherImplementation(oldConfigurationsImpl, true);

            if (configurationsMigrated)
            {
                // Set the module to DB in the configuration after it has finish initializing
                DIFLogger.getLogger().info("Activating Database Configurations module...");
                DIFStartupConfiguration.getModulesProps().setProperty(currentConfigurationModule, "Database");

                // Update all configurations
                DIFStartupConfiguration.updateConfig();
            }
            else
            {
                StringBuffer buffer = new StringBuffer();
                buffer.append("\n\n");
                buffer.append("Database Configurations module was not definitely activated.\n");
                buffer.append("Migration was not successful!\n");
                buffer.append("Check the configurations database content for the migration process to run.");
                buffer.append("\n\n");
                buffer.append("Possible solutions:\n");
                buffer.append(
                        "1) Create a new server from scratch => Change machine configuration prefix in the startup file \""
                                + DIFStartupConfiguration.CONFIGURATION_FILE
                                + "\", property \"machineIDForConfigurations\" to a non-existing one;\n");
                buffer.append(
                        "2) Reset database configurations => Delete previous entries from the configurations database for prefix \""
                                + this.getServerUID() + "\";\n");
                buffer.append("3) Use existing database configurations, do not migrate => In the startup file \""
                        + DIFStartupConfiguration.CONFIGURATION_FILE + ", set property '"
                        + AbstractConfigurationsImpl.generalConfigurationPrefix + ".module."
                        + IConfigurations.class.getCanonicalName() + "=Database';\n");
                buffer.append("4) Keep using previous legacy PreferencesAPI, do not migrate => In the startup file \""
                        + DIFStartupConfiguration.CONFIGURATION_FILE + ", set property '"
                        + AbstractConfigurationsImpl.generalConfigurationPrefix + ".module."
                        + IConfigurations.class.getCanonicalName() + "=preferences'.\n");

                // TODO: Here we should enable the bootstrap mode!
                buffer.append(
                        "\n\nFatal error! DIF will terminate. Please choose the solution from the options above!\n\n.");

                DIFLogger.getLogger().fatal(buffer.toString());

                throw new ConfigurationException(
                        "Database Configurations module was not definetly activated. Migration was not successful!");
            }
        }
        else
        {
            try
            {
                this.processServerUIDPendingNameChange();
            }
            catch (DataSetException e)
            {
                throw new ConfigurationException(e);
            }
        }

        // Trigger initialization so the DebugModel will be read on the next call to DIFModel API
        LoggingConfiguration.getInstance();
        PerformanceLoggerConfiguration.getInstance();
    }

    /**
     * Initialize DB audit context.
     */
    private void initializeDBAuditContext()
    {

        if (!AuditContext.hasContextDefined())
        {
            AuditContext.setProcessNameForCurrentThread("DBConfigurations");
            AuditContext.setUserForCurrentThread("DIF");
        }
    }

    /**
     * Checks if is persistent repository empty.
     *
     * @return true, if is persistent repository empty
     * @throws ConfigurationException
     *             the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#isPersistentRepositoryEmpty()
     */
    public boolean isPersistentRepositoryEmpty() throws ConfigurationException
    {
        this.initializeDBAuditContext();

        try
        {
            return this.getDB().getConfigurationDataSet().query()
                    .equals(Configuration.FK().configNode().configSet().configServerSets().SERVERUID(),
                            this.getServerUID())
                    .count() == 0;
        }
        catch (DataSetException e)
        {
            throw new ConfigurationException(e);
        }
    }

    /**
     * Process server UID pending name change.
     *
     * @throws DataSetException
     *             the data set exception
     */
    private void processServerUIDPendingNameChange() throws DataSetException
    {
        if (DIFStartupConfiguration.hasMachineIDNameChangePending())
        {
            String oldID = DIFStartupConfiguration.getMachineIDForConfigurations() + "|"
                    + AbstractConfigurationsImpl.generalConfigurationPrefix;
            String newID = DIFStartupConfiguration.getNewMachineIDForConfigurationsToApply() + "|"
                    + AbstractConfigurationsImpl.generalConfigurationPrefix;

            boolean wasActive = DIFRepositoryFactory.openTransaction();

            // Check if the new server already exists...
            boolean newServerUIDExists = DIFIoCRegistry.getRegistry().getImplementation(IConfigurationsService.class)
                    .getConfigServerSetDataSet().query().equals(ConfigServerSet.Fields.SERVERUID, newID).count() > 0;

            // Only change server UID for existing associations if the new server does not already exists...
            // If it exists will use them and keep old ones for recovery if needed...

            DIFLogger.getLogger().info("Changing machineServerUID for existing configurations on database from \""
                    + oldID + "\" to \"" + newID + "\"...");

            if (newServerUIDExists)
            {
                DIFLogger.getLogger().warn("Configurations for \"" + newID + "\" already exist!");
                DIFLogger.getLogger()
                        .warn("Associated the current server to the existing configurations under \"" + newID + "\".");
                DIFLogger.getLogger().warn(
                        "Kept the old configurations for recovery purposes (or another server might be using them) under \""
                                + oldID + "\".");
            }
            else
            {
                StringBuffer hqlBuffer = new StringBuffer();
                hqlBuffer.append("update " + ConfigServerSet.class.getSimpleName());
                hqlBuffer.append("   set " + ConfigServerSet.Fields.SERVERUID + " = '" + newID + "'");
                hqlBuffer.append(" where " + ConfigServerSet.Fields.SERVERUID + " = '" + oldID + "'");

                DIFRepositoryFactory.getSession().createQuery(hqlBuffer.toString()).executeUpdate();
                DIFLogger.getLogger().info("Configurations changed to \"" + newID + "\"...");
            }

            if (!wasActive)
                DIFRepositoryFactory.getSession().getTransaction().commit();

            // Apply new name. DIFInitializer will apply it definitely in the DIFStartup, but for the configurations API
            // the move is complete and the configurations are in the new node so we must search them there from now
            // on...
            serverUID = newID;
        }
    }

    /**
     * Read all configurations.
     *
     * @param configID
     *            the config ID
     * @param sectionID
     *            the section ID
     * @return the map
     * @throws ConfigurationException
     *             the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#readAllConfigurations(java.lang.String, java.lang.String)
     */
    public Map<String, Properties> readAllConfigurations(String configID, String sectionID)
            throws ConfigurationException
    {
        this.initializeDBAuditContext();

        String nodePath = configID;

        if (StringUtils.isNotBlank(sectionID))
            nodePath += "/" + sectionID;

        LinkedHashMap<String, Properties> configs = new LinkedHashMap<String, Properties>();

        try
        {
            Query<ConfigNode> nodesQuery = this.getDB().getConfigNodeDataSet().query()
                    .equals(ConfigNode.FK().configSet().configServerSets().SERVERUID(), this.getServerUID())
                    .like(ConfigNode.Fields.NODEPATH, nodePath + "%")
                    .addJoin(ConfigNode.FK().configurations(), JoinType.LEFT_OUTER_JOIN)
                    .sortBy(ConfigNode.Fields.NODEPATH, SortMode.ASCENDING);
            nodesQuery.setDistinctEntities(true);

            List<ConfigNode> nodes = nodesQuery.asList();

            for (ConfigNode node: nodes)
            {
                configs.put(node.getNodePath(), this.convertRecordToProperties(node.getConfigurations()));
            }

            return configs;
        }
        catch (DataSetException e)
        {
            throw new ConfigurationException(e);
        }
    }

    /**
     * Read all configurations.
     *
     * @param <T>
     *            the generic type
     * @param configID
     *            the config ID
     * @param sectionID
     *            the section ID
     * @param clazz
     *            the clazz
     * @return the map
     * @throws ConfigurationException
     *             the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#readAllConfigurations(java.lang.String, java.lang.String,
     *      java.lang.Class)
     */
    public <T> Map<String, T> readAllConfigurations(String configID, String sectionID, Class<T> clazz)
            throws ConfigurationException
    {
        this.initializeDBAuditContext();

        String nodePath = configID;

        if (StringUtils.isNotBlank(sectionID))
            nodePath += "/" + sectionID;

        Map<String, T> result = new HashMap<String, T>();

        try
        {
            Query<ConfigNode> nodesQuery = this.getDB().getConfigNodeDataSet().query()
                    .equals(ConfigNode.FK().configSet().configServerSets().SERVERUID(), this.getServerUID())
                    .like(ConfigNode.Fields.NODEPATH, nodePath + "%")
                    .sortBy(ConfigNode.Fields.NODEPATH, SortMode.ASCENDING);

            List<ConfigNode> nodes = nodesQuery.asList();

            for (ConfigNode node: nodes)
            {
                result.put(node.getNodePath().substring(nodePath.length() + 1),
                        this.readConfiguration(node.getNodePath(), null, clazz));
            }

            return result;
        }
        catch (DataSetException e)
        {
            throw new ConfigurationException(e);
        }
    }

    /**
     * Read configuration.
     *
     * @param configID
     *            the config ID
     * @param sectionID
     *            the section ID
     * @return the properties
     * @throws ConfigurationException
     *             the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#readConfiguration(java.lang.String, java.lang.String)
     */
    public Properties readConfiguration(String configID, String sectionID) throws ConfigurationException
    {
        this.initializeDBAuditContext();

        String nodePath = configID + (StringUtils.isBlank(sectionID) ? "" : "/" + sectionID);
        DIFLogger.getLogger().debug("Reading configuration point: " + nodePath + "...");

        List<Configuration> configProps;
        try
        {
            configProps = this.getDB().getConfigurationDataSet().query()
                    .equals(Configuration.FK().configNode().configSet().configServerSets().SERVERUID(),
                            this.getServerUID())
                    .equals(Configuration.FK().configNode().NODEPATH(), nodePath)
                    .sortBy(Configuration.Fields.ID, SortMode.ASCENDING).asList();

            return this.convertRecordToProperties(configProps);
        }
        catch (DataSetException e)
        {
            throw new ConfigurationException(e);
        }
    }

    /**
     * Removes the configuration.
     *
     * @param configID
     *            the config ID
     * @param sectionID
     *            the section ID
     * @return true, if successful
     * @throws ConfigurationException
     *             the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#removeConfiguration(java.lang.String, java.lang.String)
     */
    public boolean removeConfiguration(String configID, String sectionID) throws ConfigurationException
    {
        String nodePath = configID + (StringUtils.isBlank(sectionID) ? "" : "/" + sectionID);

        DIFLogger.getLogger().debug("Removing configuration point: " + nodePath + "...");

        this.initializeDBAuditContext();

        boolean wasActive = DIFRepositoryFactory.openTransaction();

        List<ConfigSet> listSets;
        try
        {
            listSets = this.getDB().getConfigSetDataSet().query()
                    .equals(ConfigSet.FK().configServerSets().SERVERUID(), this.getServerUID()).asList();

            if (listSets != null && !listSets.isEmpty())
            {
                // Only delete if there are nodes for this server. No nodes, no configurations...
                StringBuffer hqlBuffer = new StringBuffer();
                hqlBuffer.append("delete from " + Configuration.class.getSimpleName());
                hqlBuffer.append(" where " + Configuration.FK().configNode().ID() + " in (select "
                        + ConfigNode.Fields.ID + " from " + ConfigNode.class.getSimpleName() + " where "
                        + ConfigNode.FK().configSet().ID() + " in ("
                        + CollectionUtils.listToCommaSeparatedString(listSets, ConfigSet.Fields.ID) + ") and "
                        + ConfigNode.Fields.NODEPATH + " = '" + nodePath + "' )");

                DIFRepositoryFactory.getSession().createQuery(hqlBuffer.toString()).executeUpdate();
            }

            if (!wasActive)
                DIFRepositoryFactory.getSession().getTransaction().commit();

            return true;
        }
        catch (DataSetException e)
        {
            throw new ConfigurationException(e);
        }
    }

    /**
     * Write configuration.
     *
     * @param configID
     *            the config ID
     * @param sectionID
     *            the section ID
     * @param values
     *            the values
     * @return true, if successful
     * @throws ConfigurationException
     *             the configuration exception
     * @see pt.digitalis.utils.config.IConfigurations#writeConfiguration(java.lang.String, java.lang.String,
     *      java.util.Properties)
     */
    public boolean writeConfiguration(String configID, String sectionID, Properties values)
            throws ConfigurationException
    {
        this.initializeDBAuditContext();

        if (values != null && !values.isEmpty())
        {
            Properties originalValues = new Properties();
            originalValues.putAll(values);

            String nodePath = configID + (StringUtils.isBlank(sectionID) ? "" : "/" + sectionID);
            DIFLogger.getLogger().debug("Writing configuration point: " + nodePath + "...");

            boolean wasactive = DIFRepositoryFactory.openTransaction();
            IDataSet<Configuration> configDS = this.getDB().getConfigurationDataSet();

            // Get existing properties
            List<Configuration> configProps;
            try
            {
                configProps = configDS.query()
                        .equals(Configuration.FK().configNode().configSet().configServerSets().SERVERUID(),
                                this.getServerUID())
                        .equals(Configuration.FK().configNode().NODEPATH(), nodePath)
                        .sortBy(Configuration.Fields.ID, SortMode.ASCENDING).asList();

                // Update existing values
                for (Configuration configurationInDB: configProps)
                {
                    if (values.containsKey(configurationInDB.getKey()))
                    {
                        String propertyValue = values.getProperty(configurationInDB.getKey());

                        String value = null;
                        String valueXL = null;

                        if (propertyValue != null && propertyValue.length() > 4000)
                            valueXL = propertyValue;
                        else
                            value = propertyValue;

                        configurationInDB.setValue(value);
                        configurationInDB.setValueXL(valueXL);
                        configDS.update(configurationInDB);

                        // Taken care of. Discard it...
                        values.remove(configurationInDB.getKey());
                    }
                }

                // Insert new ones, if any remained...
                if (!values.isEmpty())
                {
                    // Get the node to write to
                    ConfigNode node = this.getOrCreateNode(nodePath);

                    for (Object propertyKey: values.keySet())
                    {
                        String propertyValue = values.getProperty(propertyKey.toString());

                        String value = null;
                        String valueXL = null;

                        if (propertyValue != null && propertyValue.length() > 4000)
                            valueXL = propertyValue;
                        else
                            value = propertyValue;

                        Configuration newConfigurationDB = new Configuration();
                        newConfigurationDB.setConfigNode(node);
                        newConfigurationDB.setKey(propertyKey.toString());
                        newConfigurationDB.setValue(value);
                        newConfigurationDB.setValueXL(valueXL);
                        newConfigurationDB = configDS.insert(newConfigurationDB);
                    }

                }

                if (!wasactive)
                    DIFRepositoryFactory.getSession().getTransaction().commit();

                if (DIFStartupConfiguration.getReplicaMode()
                        && DIFGeneralConfigurationParameters.getInstance().getKeepPrefsFileWhenInReplicaMode())
                {
                    // If in replica mode, the database parameter values can be overriden by the existing
                    Properties replicaProps = new Properties();
                    InputStream replicaConfigsFile = Thread.currentThread().getContextClassLoader()
                            .getResourceAsStream(DIFInitializer.REPLICA_PREFS);

                    try
                    {
                        // Load previous properties if any...
                        if (replicaConfigsFile != null)
                            replicaProps.load(replicaConfigsFile);

                        // Append/update new properties
                        for (Object key: originalValues.keySet())
                            replicaProps.put(nodePath + "." + key.toString(), originalValues.get(key));

                        // Save file with new properties
                        String difPropertiesFilePath = DIFStartupConfiguration
                                .getFileInConfigPath(DIFInitializer.REPLICA_PREFS);

                        DIFLogger.getLogger().debug("Saving configuration node \"" + nodePath
                                + "\" in replica preferences file \"" + DIFInitializer.REPLICA_PREFS + "\"...");
                        FileOutputStream outStream = new FileOutputStream(difPropertiesFilePath);
                        replicaProps.store(outStream, "DIF2 Replica configurations override file");
                    }
                    catch (IOException e)
                    {
                        new BusinessException(
                                "Error saving changed preferences to \"" + DIFInitializer.REPLICA_PREFS + "\"", e)
                                        .addToExceptionContext("node", nodePath).addToExceptionContext("prefs", values)
                                        .log(LogLevel.ERROR);
                        e.printStackTrace();
                    }
                }
            }
            catch (DataSetException e)
            {
                new BusinessException("Errors saving configurations", e).addToExceptionContext("NodePath", nodePath)
                        .addToExceptionContext("Properties", values).log(LogLevel.ERROR);
                e.printStackTrace();

                return false;
            }
        }

        return true;
    }
}
