/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.ecommerce;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.hibernate.exception.ConstraintViolationException;
import pt.digitalis.dif.dem.managers.impl.model.IECommerceService;
import pt.digitalis.dif.dem.managers.impl.model.data.EcommercePayments;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.model.dataset.DataSetException;
import pt.digitalis.dif.model.dataset.Filter;
import pt.digitalis.dif.model.dataset.FilterType;
import pt.digitalis.dif.model.dataset.Query;
import pt.digitalis.dif.utils.logging.IErrorLogManager;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.ConfigurationException;

import java.math.BigDecimal;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.sql.Timestamp;
import java.util.Date;
import java.util.List;

/**
 * The Class AbstractECommerce.
 *
 * @param <T> provider response type
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created May 30, 2012
 */
public abstract class AbstractECommerce<T> implements IECommerce<T>
{

    /**
     *
     */
    public static final String UNIQUE_ID_SEPARATOR = "-";

    /** Validation message: The payment amount received is different from the original request. */
    protected static final String PAYMENT_AMOUNT_CHANGED_MSG =
            "The payment amount received is different from the original request.";

    /** The Constant PAYMENT_BUSINESS_PROCESSOR_FAILED. */
    protected static final String PAYMENT_BUSINESS_PROCESSOR_FAILED = "The payment business proccess [\"#1\"] failed.";

    /** The Constant PAYMENT_BUSINESS_PROCESSOR_WARNING. */
    protected static final String PAYMENT_BUSINESS_PROCESSOR_WARNING =
            "The payment business proccess [\"#1\"] has warnings.";

    /** Validation message: The payment was not initialized. Process aborted */
    protected static final String PAYMENT_INIT_ERROR = "The payment was not initialized. Process aborted.";

    /** Validation message: The payment token received is different from the original request. */
    protected static final String PAYMENT_TOKEN_CHANGED_MSG =
            "The payment token received is different from the original request.";

    /** The MESSAG e_ separator. */
    public static String MESSAGE_SEPARATOR = "#SEP#";

    /** The e commerce service. */
    IECommerceService eCommerceService = DIFIoCRegistry.getRegistry().getImplementation(IECommerceService.class);

    /** {@link IErrorLogManager} the error Manager */
    IErrorLogManager errorLogManager = DIFIoCRegistry.getRegistry().getImplementation(IErrorLogManager.class);

    /**
     * Verify and trucate if the message is bigger than 4000 charateres
     *
     * @param newMessage    the new message
     * @param statusMessage the status message
     *
     * @return the message
     */
    public static String verifyMessage(String newMessage, String statusMessage)
    {
        String result = statusMessage;
        // TODO: Verificar isto quando se alterar a STATUS_MESSAGE para CLOB.
        if (result != null)
        {
            if (result.contains(newMessage))
                result = result.replace(newMessage, "");

            result += newMessage;

            if (result.length() >= 3999)
            {
                int position = result.lastIndexOf(newMessage);
                result = result.substring(position - 400, position - 1);
            }
        }

        return result;
    }

    /**
     * @see pt.digitalis.dif.ecommerce.IECommerce#canCancelPayments()
     */
    @Override
    public boolean canCancelPayments()
    {
        return true;
    }

    /**
     * Do init.
     *
     * @param payment         the payment
     * @param businessID      the business unique id
     * @param configurationId the configuration id
     *
     * @return the payment details
     *
     * @exception ConfigurationException the configuration exception
     */
    public abstract PaymentExecutionResult<T> doInit(PaymentRequest payment, String businessID, String configurationId)
            throws ConfigurationException;

    /**
     * Do process.
     *
     * @param businessID       the business ID to process
     * @param securityToken    the payment security token
     * @param configurationId  the configuration Id
     * @param providerResponse provider payment response type
     *
     * @return the payment details
     *
     * @exception ConfigurationException the configuration exception
     */
    public abstract PaymentExecutionResult<T> doProcess(String businessID, String securityToken, String configurationId,
            T providerResponse) throws ConfigurationException;

    /**
     * Gets the next id.
     *
     * @param partialBusinessId the partial BusinessId id (i.e. RU:XXXX-12345)
     *
     * @return the next id
     *
     * @exception DataSetException the data set exception
     */
    private String getNextID(String partialBusinessId) throws DataSetException
    {
        String result = "";

        Query<EcommercePayments> query = eCommerceService.getEcommercePaymentsDataSet().query();
        query.addFilter(new Filter(EcommercePayments.Fields.BUSINESSID.toString(), FilterType.LIKE, partialBusinessId));

        Long seq = query.count();

        result = partialBusinessId + UNIQUE_ID_SEPARATOR + (seq + 1);

        return result;
    }

    /**
     * Gets the payment record.
     *
     * @param businessId the business id
     *
     * @return the payment record
     *
     * @exception DataSetException the data set exception
     */
    private EcommercePayments getPaymentRecord(String businessId) throws DataSetException
    {
        Query<EcommercePayments> query = eCommerceService.getEcommercePaymentsDataSet().query();
        query.addFilter(new Filter(EcommercePayments.Fields.BUSINESSID.toString(), FilterType.EQUALS, businessId));
        return query.singleValue();
    }

    /**
     * @see pt.digitalis.dif.ecommerce.IECommerce#initWebPayment(pt.digitalis.dif.ecommerce.PaymentRequest,
     *         java.lang.String)
     */
    @Override
    public final EcommercePayments initWebPayment(PaymentRequest payment, String partialBusinessId)
            throws PaymentException, ConfigurationException
    {
        return initWebPayment(payment, partialBusinessId, null);
    }

    /**
     * @see pt.digitalis.dif.ecommerce.IECommerce#initWebPayment(pt.digitalis.dif.ecommerce.PaymentRequest,
     *         java.lang.String, java.lang.String)
     */
    @Override
    public EcommercePayments initWebPayment(PaymentRequest payment, String partialBusinessId, String configurationId)
            throws PaymentException, ConfigurationException
    {
        EcommercePayments ePaymentRecord = null;
        ePaymentRecord = new EcommercePayments();
        ePaymentRecord.setBusinessContext(payment.getBusinessContext());
        ePaymentRecord.setCreator(payment.getCreator());
        ePaymentRecord.setDateCreation(new Timestamp(new Date().getTime()));
        ePaymentRecord.setEcommerceProcessor(this.getIdentifier());
        ePaymentRecord.setPaymentValue(payment.getAmount());
        ePaymentRecord.setConfigurationId(configurationId);

        // Writes the ip in the payment menssage
        InetAddress ip;
        try
        {
            ip = InetAddress.getLocalHost();
            ePaymentRecord
                    .setStatusMessage(" Host Address:  " + ip.getHostAddress() + " Host Name: " + ip.getHostName());
        }
        catch (UnknownHostException e1)
        {
            // Do nothing, was not possible to obtain the ip address
        }

        // In case it has a fee will insert it on EcommercePayments and increment to amount
        if (payment.getFee() != null)
        {
            ePaymentRecord.setPaymentFee(payment.getFee());
            BigDecimal amount = payment.getAmount();
            amount = amount.add(payment.getFee());
            ePaymentRecord.setPaymentValue(amount);
            payment.setAmount(amount);
        }

        ePaymentRecord.setStatus(PaymentStatus.F.name()); /*
         * Failed: Preventive status. Will change to W if online payment
         * successful started. Otherwise will thus be discarded
         */
        ePaymentRecord.setPaymentContext(CollectionUtils.keyValueMapToString(payment.getPaymentContext()));

        boolean inserted = false;

        // Must insert the record before to make sure the businessID is not taken and thus reserving it.
        while (!inserted)
            try
            {
                // Will try this until a non-unique value is inserted.
                String id = this.getNextID(partialBusinessId);
                ePaymentRecord.setBusinessId(id);

                ePaymentRecord = eCommerceService.getEcommercePaymentsDataSet().insert(ePaymentRecord);
                inserted = true;
            }
            catch (Exception e)
            {
                boolean uniqueKeyException = false;

                if ((e instanceof ConstraintViolationException))
                {
                    ConstraintViolationException constraintViolation = (ConstraintViolationException) e.getCause();
                    // TODO: Add the correct constraint name after it exists
                    uniqueKeyException = constraintViolation.getConstraintName().toUpperCase()
                            .endsWith(".EPAYMENTRECORD_BUSINESS_ID_UK");
                }

                // Any other error will be thowned outside to stop the process
                if (!uniqueKeyException)
                    throw new PaymentException(e);
            }

        // Now that we have a reserved ID initialize the online payment service with it
        PaymentExecutionResult<T> paymentResult = doInit(payment, ePaymentRecord.getBusinessId(), configurationId);

        if (paymentResult.isSuccess())
        {
            // If success update the security token in our record and change it to waiting
            ePaymentRecord.setSecurityToken(paymentResult.getSecurityToken());
            ePaymentRecord.setRedirectUrl(paymentResult.getRedirectURL());
            ePaymentRecord.setTransactionId(paymentResult.getTransactionId());
            ePaymentRecord.setPaymentContext(CollectionUtils.keyValueMapToString(payment.getPaymentContext()));

            String message = "doInit: Payment '" + ePaymentRecord.getBusinessId() + "' initialized.";

            if (paymentResult.getMessage() != null)
                message += " Message:" + paymentResult.getMessage();
            ePaymentRecord.setStatusMessage(message);
            ePaymentRecord.setStatus(PaymentStatus.W.name()); /* Waiting */

            try
            {
                ePaymentRecord = eCommerceService.getEcommercePaymentsDataSet().update(ePaymentRecord);
            }
            catch (DataSetException e)
            {
                throw new PaymentException(e);
            }
        }
        else
        {
            String errorMessage;

            if (paymentResult.getMessage() == null)
                errorMessage = PAYMENT_INIT_ERROR;
            else
                errorMessage = PAYMENT_INIT_ERROR + "\n   Reason: " + paymentResult.getMessage();

            try
            {
                ePaymentRecord.setStatusMessage(errorMessage);
                ePaymentRecord = eCommerceService.getEcommercePaymentsDataSet().update(ePaymentRecord);
            }
            catch (DataSetException e)
            {
                throw new PaymentException(e);
            }

            throw new PaymentException(errorMessage);
        }

        return ePaymentRecord;
    }

    /**
     * @see pt.digitalis.dif.ecommerce.IECommerce#processBusinessPayment(pt.digitalis.dif.dem.managers.impl.model.data.EcommercePayments)
     */
    @Override
    public final EcommercePayments processBusinessPayment(EcommercePayments ePaymentRecord) throws PaymentException
    {
        try
        {
            String statusMessage = StringUtils.nvl(ePaymentRecord.getStatusMessage(), "");

            /* Executes the business associated with this payment, if exists */

            List<IECommerceBusiness> list = DIFIoCRegistry.getRegistry().getImplementations(IECommerceBusiness.class);
            Boolean markAsProcessed = true;
            Boolean updateRecord = false;

            // Writes the ip in the payment menssage
            InetAddress ip;
            try
            {
                ip = InetAddress.getLocalHost();
                ePaymentRecord
                        .setStatusMessage(" Host Address:  " + ip.getHostAddress() + " Host Name: " + ip.getHostName());
            }
            catch (UnknownHostException e1)
            {
                // Do nothing, was not possible to obtain the ip address
            }

            if (!list.isEmpty())
            {
                IECommerceBusiness business = list.get(0);
                if (business != null)
                {
                    String businessMessage = "";
                    BusinessProcessResult processResult;
                    try
                    {
                        processResult = business.processPayment(ePaymentRecord);
                        if (!processResult.isSuccess())
                            markAsProcessed = false;
                        businessMessage = processResult.getMessage();
                    }
                    catch (Exception e)
                    {
                        businessMessage = " " + e.getMessage();
                        markAsProcessed = false;
                    }
                    if (!markAsProcessed)
                    {
                        String newMessage = MESSAGE_SEPARATOR + PAYMENT_BUSINESS_PROCESSOR_FAILED
                                .replace("#1", business.getFullIdentifier()) + ": " + businessMessage;
                        // TODO: Verificar isto quando se alterar a STATUS_MESSAGE para CLOB.
                        statusMessage = verifyMessage(newMessage, statusMessage);
                        ePaymentRecord.setStatusMessage(statusMessage.trim());
                        updateRecord = true;
                    }
                    else if (businessMessage != null && !"".equals(businessMessage))
                    {
                        statusMessage += MESSAGE_SEPARATOR + PAYMENT_BUSINESS_PROCESSOR_WARNING
                                .replace("#1", business.getFullIdentifier()) + ": " + businessMessage;
                        updateRecord = true;

                        ePaymentRecord.setStatusMessage(statusMessage.trim());
                    }
                }
            }
            if (markAsProcessed)
            {
                /* Updates the payment to PROCESSED Status */
                ePaymentRecord.setStatus(PaymentStatus.P.name());
                ePaymentRecord.setDateProcessed(new Timestamp(new Date().getTime()));
                updateRecord = true;
            }

            if (updateRecord)
            {
                String message = ePaymentRecord.getStatusMessage();
                if (StringUtils.isNotBlank(message))
                {
                    message = StringUtils.right(message, 3999).trim();
                    ePaymentRecord.setStatusMessage(message);
                }
                ePaymentRecord = eCommerceService.getEcommercePaymentsDataSet().update(ePaymentRecord);
            }
        }
        catch (DataSetException e)
        {
            throw new PaymentException(e);
        }

        return ePaymentRecord;
    }

    /**
     * @see pt.digitalis.dif.ecommerce.IECommerce#processWebPayment(java.lang.String)
     */
    @Override
    public EcommercePayments processWebPayment(String businessID) throws PaymentException, ConfigurationException
    {
        return processWebPayment(businessID, null, null);
    }

    /**
     * @see pt.digitalis.dif.ecommerce.IECommerce#processWebPayment(java.lang.String, java.lang.String,
     *         java.lang.Object)
     */
    @Override
    public final EcommercePayments processWebPayment(String businessID, String securityToken, T providerResponse)
            throws PaymentException, ConfigurationException
    {
        // To prevent concurrent calls for the same payment. For example, the job run at same time as a business call
        synchronized (this)
        {

            EcommercePayments ePaymentRecord = null;
            try
            {
                ePaymentRecord = getPaymentRecord(businessID);

                if (ePaymentRecord == null)
                    throw new PaymentException(
                            "The requested payment wasn't found. The payment ID is \"" + businessID + "\"");
                // For the case the Epayment instance identifier is different from one in record, should not process
                if (!(ePaymentRecord.getEcommerceProcessor() != null &&
                      this.getIdentifier().equals(ePaymentRecord.getEcommerceProcessor())))
                {
                    String message =
                            (ePaymentRecord.getStatusMessage() != null ? ePaymentRecord.getStatusMessage() : "") +
                            " - " + "Error: " + "The processor identifier for this instance (" + this.getIdentifier() +
                            ") is different from the one in record (" + ePaymentRecord.getEcommerceProcessor() +
                            "). Please see error log and search for the business id(" + ePaymentRecord.getBusinessId() +
                            ").";

                    ePaymentRecord.setStatusMessage(message);

                    Exception e = new PaymentException(message);
                    message += " - " + ExceptionUtils.getStackTrace(e);

                    errorLogManager
                            .logError("EcommercePayments", this.getClass().getName(), new PaymentException(message));
                }
                else
                {
                    String statusMessage = StringUtils.nvl(ePaymentRecord.getStatusMessage(), "");

                    if (!PaymentStatus.W.name().equals(ePaymentRecord.getStatus()))
                        throw new PaymentException(
                                "The requested payment has allready been processed. The payment ID is \"" + businessID +
                                "\"");
                    else
                    {
                        PaymentExecutionResult<T> paymentExecutionResult = new PaymentExecutionResult<T>();

                        /* If no errors, process the payment */
                        if (ePaymentRecord.getSecurityToken() == null ||
                            ePaymentRecord.getSecurityToken().equals(securityToken) || securityToken == null)
                        {
                            if (securityToken == null)
                                securityToken = ePaymentRecord.getSecurityToken();
                            /* Process the Payment Gateway implementation */
                            paymentExecutionResult =
                                    this.doProcess(businessID, securityToken, ePaymentRecord.getConfigurationId(),
                                            providerResponse);
                            statusMessage += MESSAGE_SEPARATOR + " doProcess ";
                        }
                        else
                            statusMessage += " " + PAYMENT_TOKEN_CHANGED_MSG;

                        // If the payment had no token, and there is one now, update it!
                        if (ePaymentRecord.getSecurityToken() == null &&
                            paymentExecutionResult.getSecurityToken() != null)
                            ePaymentRecord.setSecurityToken(paymentExecutionResult.getSecurityToken());

                        if (paymentExecutionResult.isSuccess())
                        {
                            statusMessage += " Success ";
                            if (paymentExecutionResult.getMessage() != null)
                                statusMessage += paymentExecutionResult.getMessage();
                            paymentExecutionResult.setBusinessContext(ePaymentRecord.getBusinessContext());

                            if (paymentExecutionResult.getAmmount().compareTo(ePaymentRecord.getPaymentValue()) != 0)
                                statusMessage += MESSAGE_SEPARATOR + PAYMENT_AMOUNT_CHANGED_MSG;

                            /* Updates the payment to RECIEVED Status */
                            String message = StringUtils.right(statusMessage, 3999);
                            ePaymentRecord.setStatusMessage(StringUtils.nvl(message, "").trim());

                            ePaymentRecord.setStatus(PaymentStatus.R.name());
                            ePaymentRecord.setTransactionId(paymentExecutionResult.getTransactionId());
                            ePaymentRecord.setTransactionDate(
                                    new Timestamp(paymentExecutionResult.getTransactionDate().getTime()));
                            ePaymentRecord.setDateReceived(new Timestamp(new Date().getTime()));
                            ePaymentRecord.setPaymentValue(paymentExecutionResult.getAmmount());
                            ePaymentRecord.setAuthorizationId(paymentExecutionResult.getAuthorizationId());
                            ePaymentRecord = eCommerceService.getEcommercePaymentsDataSet().update(ePaymentRecord);

                            /* Process the client extra business */
                            ePaymentRecord = processBusinessPayment(ePaymentRecord);
                        }
                        else
                        {

                            if (paymentExecutionResult.getMessage() != null)
                            {
                                String newMessage = MESSAGE_SEPARATOR + paymentExecutionResult.getMessage();
                                // TODO: Verificar isto quando se alterar a STATUS_MESSAGE para CLOB.
                                statusMessage = verifyMessage(newMessage, statusMessage);
                            }

                            // In case there's a error message more friendly to display to user
                            if (StringUtils.isNotEmpty(paymentExecutionResult.getMessageForUser()))
                            {
                                String errorFriendlyMessage = paymentExecutionResult.getMessageForUser();
                                errorFriendlyMessage = StringUtils.right(errorFriendlyMessage, 3990);
                                ePaymentRecord.setErrorFriendlyUserMessage(errorFriendlyMessage.trim());
                            }
                            statusMessage = StringUtils.right(statusMessage, 3990);

                            ePaymentRecord.setStatusMessage(statusMessage.trim());
                            if (paymentExecutionResult.isPaymentFailed())
                            {
                                ePaymentRecord.setStatus(PaymentStatus.F.name());
                                ePaymentRecord.setDateReceived(new Timestamp(new Date().getTime()));
                            }
                            ePaymentRecord = eCommerceService.getEcommercePaymentsDataSet().update(ePaymentRecord);
                        }
                    }
                }
            }
            catch (DataSetException e)
            {
                throw new PaymentException(e.getMessage());
            }

            return ePaymentRecord;
        }
    }
}
