/**
 * 2019, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 http://www.digitalis.pt
 */
package pt.digitalis.dif.servermanager;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import pt.digitalis.dif.dem.managers.impl.model.DIFRepositoryFactory;
import pt.digitalis.dif.dem.managers.impl.model.IServersService;
import pt.digitalis.dif.dem.managers.impl.model.data.Server;
import pt.digitalis.dif.dem.managers.impl.model.data.ServerActivityLog;
import pt.digitalis.dif.dem.managers.impl.model.data.ServerMessage;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.model.dataset.DataSetException;
import pt.digitalis.dif.model.dataset.JoinType;
import pt.digitalis.dif.servermanager.messages.IServerMessage;
import pt.digitalis.dif.startup.DIFInitializer;
import pt.digitalis.dif.startup.DIFStartupConfiguration;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.AbstractConfigurationsImpl;
import pt.digitalis.utils.config.ConfigurationException;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

// TODO: Auto-generated Javadoc

/**
 * The Class ServerManagerDBImpl.
 *
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created Apr 23, 2019
 */
public class ServerManagerDBImpl extends AbstractServerManager implements IServerManager
{

    /** The db. */
    private IServersService db;

    /** The servers list. */
    private List<ServerApplicationNodeInstance> serversList = new ArrayList<ServerApplicationNodeInstance>();

    /** The servers map indexed by server.ID */
    private Map<Long, ServerApplicationNodeInstance> serversMapByServerID =
            new HashMap<Long, ServerApplicationNodeInstance>();

    /**
     * Allow server communication.
     *
     * @return true, if successful
     *
     * @see pt.digitalis.dif.servermanager.IServerManager#allowServerCommunication()
     */
    public boolean allowServerCommunication()
    {
        return true;
    }

    /**
     * Apply name change.
     *
     * @exception ServerManagerException the server manager exception
     * @see pt.digitalis.dif.servermanager.IServerManager#applyNameChange()
     */
    public void applyNameChange() throws ServerManagerException
    {
        if (DIFStartupConfiguration.hasMachineIDNameChangePending())
        {
            String oldID = DIFStartupConfiguration.getMachineIDForConfigurations() + "|" +
                           AbstractConfigurationsImpl.generalConfigurationPrefix;
            String newID = DIFStartupConfiguration.getNewMachineIDForConfigurationsToApply() + "|" +
                           AbstractConfigurationsImpl.generalConfigurationPrefix;

            boolean wasActive = DIFRepositoryFactory.openTransaction();

            StringBuffer hqlBuffer = new StringBuffer();
            hqlBuffer.append("update " + Server.class.getSimpleName());
            hqlBuffer.append("   set " + Server.Fields.MACHINESERVERUID + " = '" + newID + "'");
            hqlBuffer.append(" where " + Server.Fields.MACHINESERVERUID + " = '" + oldID + "'");
            hqlBuffer.append("   and " + Server.Fields.IPADDRESS + " = '" + this.getServerNode().getServerIP() + "'");
            hqlBuffer.append("   and " + Server.Fields.PORT + " = '" + this.getServerNode().getPort() + "'");

            DIFRepositoryFactory.getSession().createQuery(hqlBuffer.toString()).executeUpdate();

            if (wasActive)
                DIFRepositoryFactory.getSession().getTransaction().commit();

            DIFLogger.getLogger().info("Server serverUID changed to \"" + newID + "\" in control database...");
        }
    }

    /**
     * Discover servers.
     *
     * @exception ServerManagerException the server manager exception
     * @see pt.digitalis.dif.servermanager.IServerManager#discoverServers()
     */
    public synchronized void discoverServers() throws ServerManagerException
    {
        try
        {
            DIFLogger.getLogger().debug("Refreshing servers list from control database...");

            ArrayList<ServerApplicationNodeInstance> results = new ArrayList<ServerApplicationNodeInstance>();
            HashMap<Long, ServerApplicationNodeInstance> indexedResults =
                    new HashMap<Long, ServerApplicationNodeInstance>();
            Server currentServerRecord = (Server) this.getServerNode().getServerManagementObject();

            for (Server server : this.getDBService().getServerDataSet().query().equals(Server.Fields.ACTIVE, "true")
                    .asList())
            {
                if (currentServerRecord != null && server.getId().equals(currentServerRecord.getId()))
                {
                    // Skip current server
                }
                else
                {
                    Long currentTime = System.currentTimeMillis();
                    Long minutesElapsed = (currentTime - server.getLastSync().getTime()) / 1000 / 60;

                    if (minutesElapsed > ServerManagerConfigurations.getInstance().getPurgeInactiveServerAfterMinutes())
                    {
                        // Purge server, missing for too long
                        server.setMissing(true);
                        server.setActive(false);
                        server = this.getDBService().getServerDataSet().update(server);
                        DIFLogger.getLogger().info("Server: " + server.getName() + " was marked as INACTIVE after " +
                                                   minutesElapsed + " minutes of inactivity...");
                    }
                    else
                    {
                        // Still relevant, lets analize it...
                        if (minutesElapsed > INACTIVE_AFTER_MINUTES && !server.isMissing())
                        {
                            // Inactive: Mark as missing
                            server.setMissing(true);
                            server = this.getDBService().getServerDataSet().update(server);

                            DIFLogger.getLogger()
                                    .info("Server: " + server.getName() + " was marked as MISSING " + minutesElapsed +
                                          " minutes of inactivity...");
                        }

                        ServerApplicationNodeInstance tmpServerNode =
                                new ServerApplicationNodeInstance(server.getMachineServerUid(), server.getIpAddress(),
                                        server.getContextRoot(), server.getPort(), server.getEndpointBaseUrl());
                        tmpServerNode.setServerManagementObject(server);
                        results.add(tmpServerNode);
                        indexedResults.put(server.getId(), tmpServerNode);

                        DIFLogger.getLogger()
                                .debug("Server added to ServerSync nodes " + server.getName() + " [" + server.getId() +
                                       "]...");
                    }
                }
            }

            this.serversList = results;
            this.serversMapByServerID = indexedResults;
        }
        catch (DataSetException e)
        {
            throw new ServerManagerException(e);
        }
        catch (ConfigurationException e)
        {
            throw new ServerManagerException(e);
        }
    }

    /**
     * Gets the all servers.
     *
     * @return the all servers
     *
     * @exception ServerManagerException the server manager exception
     * @see pt.digitalis.dif.servermanager.IServerManager#getAllServers()
     */
    public List<ServerApplicationNodeInstance> getAllServers() throws ServerManagerException
    {
        // Will be refresh periodically by the background job
        return this.serversList;
    }

    /**
     * Gets the DB service.
     *
     * @return the DB service
     */
    private IServersService getDBService()
    {
        if (db == null)
            db = DIFIoCRegistry.getRegistry().getImplementation(IServersService.class);

        return db;
    }

    /**
     * Gets the server application node instance by server ID.
     *
     * @param id the id
     *
     * @return the server application node instance by server ID
     */
    private ServerApplicationNodeInstance getServerApplicationNodeInstanceByServerID(Long id)
    {
        return this.serversMapByServerID.get(id);
    }

    /**
     * Internal initialize.
     *
     * @exception ServerManagerException the server manager exception
     */
    @Override
    void internalInitialize() throws ServerManagerException
    {
        // Nothing to do...
    }

    /**
     * @see pt.digitalis.dif.servermanager.IServerManager#isValidServerIPAddress(java.lang.String)
     */
    public boolean isValidServerIPAddress(String callerIP)
    {
        if (StringUtils.isNotBlank(callerIP))
        {
            for (ServerApplicationNodeInstance server : serversList)
            {
                if (callerIP.equals(server.getServerIP()))
                {
                    DIFLogger.getLogger().debug("Caller valid: Server " + server.getServerBaseURL() +
                                                " is registered for the caller IP " + callerIP);
                    return true;
                }
            }
        }

        DIFLogger.getLogger().debug("Caller INVALID: No server registeres for the caller IP " + callerIP);

        return false;
    }

    /**
     * Keep server alive.
     *
     * @exception ServerManagerException the server manager exception
     * @see pt.digitalis.dif.servermanager.IServerManager#keepServerAlive()
     */
    public synchronized void keepServerAlive() throws ServerManagerException
    {
        Server serverRecord = (Server) this.getServerNode().getServerManagementObject();

        if (serverRecord == null)
        {
            registerServer();
            serverRecord = (Server) this.getServerNode().getServerManagementObject();
        }

        if (serverRecord != null)
        {
            DIFLogger.getLogger().debug("Performing keep alive for the current server...");

            try
            {
                serverRecord.setLastSync(new Timestamp(new Date().getTime()));
                serverRecord.setActive(true);
                serverRecord.setMissing(false);
                serverRecord.setLastSync(new Timestamp(new Date().getTime()));
                serverRecord = this.getDBService().getServerDataSet().update(serverRecord);
            }
            catch (DataSetException e)
            {
                throw new ServerManagerException(e);
            }
        }
    }

    /**
     * Migrate messages between server entries.
     *
     * @param previousServerNode  the previous server node
     * @param newServerNode       the new server node
     * @param onlyPendingMessages the only pending messages
     *
     * @see pt.digitalis.dif.servermanager.AbstractServerManager#migrateMessagesBetweenServerEntries(pt.digitalis.dif.servermanager.ServerApplicationNodeInstance,
     *         pt.digitalis.dif.servermanager.ServerApplicationNodeInstance, boolean)
     */
    @Override
    protected synchronized void migrateMessagesBetweenServerEntries(ServerApplicationNodeInstance previousServerNode,
            ServerApplicationNodeInstance newServerNode, boolean onlyPendingMessages)
    {
        Server oldServer = ((Server) previousServerNode.getServerManagementObject());
        Server newServer = ((Server) newServerNode.getServerManagementObject());

        if (oldServer != null && newServer != null && !oldServer.getId().equals(newServer.getId()))
        {
            String oldServerID = oldServer.getId().toString();
            String newServerID = newServer.getId().toString();

            boolean wasActive = DIFRepositoryFactory.openTransaction();

            StringBuffer hqlBuffer = new StringBuffer();
            hqlBuffer.append("update " + ServerMessage.class.getSimpleName());
            hqlBuffer.append("   set " + ServerMessage.FK().serverByServerReceiverId() + " = " + newServerID);
            hqlBuffer.append(" where " + ServerMessage.FK().serverByServerReceiverId() + " = " + oldServerID);
            if (onlyPendingMessages)
                hqlBuffer.append("   and " + ServerMessage.Fields.PROCESSED + " = false");
            DIFRepositoryFactory.getSession().createQuery(hqlBuffer.toString()).executeUpdate();

            hqlBuffer = new StringBuffer();
            hqlBuffer.append("update " + ServerMessage.class.getSimpleName());
            hqlBuffer.append("   set " + ServerMessage.FK().serverByServerSenderId() + " = " + newServerID);
            hqlBuffer.append(" where " + ServerMessage.FK().serverByServerSenderId() + " = " + oldServerID);
            if (onlyPendingMessages)
                hqlBuffer.append("   and " + ServerMessage.Fields.PROCESSED + " = false");
            DIFRepositoryFactory.getSession().createQuery(hqlBuffer.toString()).executeUpdate();

            if (wasActive)
                DIFRepositoryFactory.getSession().getTransaction().commit();

            DIFLogger.getLogger()
                    .info("Server messages migrated from server \"" + oldServerID + "\" to " + newServerID + " ('" +
                          newServer.getName() + "') in control database...");
        }
    }

    /**
     * Process server messages.
     *
     * @exception ServerManagerException the server manager exception
     * @see pt.digitalis.dif.servermanager.IServerManager#processServerMessages()
     */
    public synchronized void processServerMessages() throws ServerManagerException
    {
        Server serverRecord = (Server) this.getServerNode().getServerManagementObject();

        if (serverRecord != null)
        {
            DIFLogger.getLogger().debug("Processing pending server messages");

            try
            {
                for (ServerMessage message : this.getDBService().getServerMessageDataSet().query()
                        .equals(ServerMessage.FK().serverByServerReceiverId().ID(), serverRecord.getId().toString())
                        .equals(ServerMessage.Fields.PROCESSED, "false")
                        .addJoin(ServerMessage.FK().serverByServerSenderId(), JoinType.NORMAL).asList())
                {
                    Server senderRecord = message.getServerByServerSenderId();
                    ServerApplicationNodeInstance sender =
                            getServerApplicationNodeInstanceByServerID(senderRecord.getId());

                    DIFLogger.getLogger()
                            .debug("Processing message from " + senderRecord.getName() + " [" + senderRecord.getId() +
                                   "] - Content: " + message.getMessage());

                    ServerMessageExecutionResult result = processServerMessage(message.getMessage(), sender);

                    DIFLogger.getLogger()
                            .info("Processed message from " + senderRecord.getName() + " [" + senderRecord.getId() +
                                  "] - " + (result.isSuccess() ? "success!" : "problems ocurred, success = false!"));
                    DIFLogger.getLogger().debug("Message timestamp: " + message.getWhen().toString());
                    DIFLogger.getLogger().debug("Message processed: " + message.getMessage());
                    DIFLogger.getLogger().debug("Message result: " + result.toJSONString());

                    message.setProcessed(true);
                    message.setSuccess(result.isSuccess());
                    message.setAnswer(result.toJSONString());
                    message.setElapsedTime(System.currentTimeMillis() - message.getWhen().getTime());

                    message = this.getDBService().getServerMessageDataSet().update(message);
                }

                DIFLogger.getLogger().debug("Purging old processed messages...");
                boolean wasActive = DIFRepositoryFactory.openTransaction();
                Date olderThan = new Date();
                Calendar c = Calendar.getInstance();
                c.setTime(olderThan);
                c.add(Calendar.MONTH,
                        -ServerManagerConfigurations.getInstance().getPurgeInactiveServerAfterMinutes().intValue());
                olderThan = c.getTime();

                StringBuffer hqlBuffer = new StringBuffer();
                hqlBuffer.append("delete from " + ServerMessage.class.getSimpleName() + " t");
                hqlBuffer.append(" where " + ServerMessage.FK().serverByServerReceiverId() + " = " +
                                 serverRecord.getId());
                hqlBuffer.append("   and t." + ServerMessage.Fields.WHEN + " = :olderThan");

                int deletedMessages = DIFRepositoryFactory.getSession().createQuery(hqlBuffer.toString())
                        .setTimestamp("olderThan", olderThan).executeUpdate();

                if (wasActive)
                    DIFRepositoryFactory.getSession().getTransaction().commit();

                if (deletedMessages > 0)
                    DIFLogger.getLogger()
                            .info("Deleted " + deletedMessages + " messages in the control database (older than " +
                                  olderThan + ")...");
            }
            catch (DataSetException e)
            {
                throw new ServerManagerException(e);
            }
            catch (ConfigurationException e)
            {
                throw new ServerManagerException(e);
            }
            catch (JsonProcessingException e)
            {
                throw new ServerManagerException(e);
            }
        }
    }

    /**
     * Register server.
     *
     * @exception ServerManagerException the server manager exception
     * @see pt.digitalis.dif.servermanager.IServerManager#registerServer()
     */
    public synchronized void registerServer() throws ServerManagerException
    {
        Server serverRecord;
        Server currentServerRecord = (Server) getServerNode().getServerManagementObject();

        try
        {
            if (StringUtils.isBlank(getServerNode().getPort()))
            {
                DIFLogger.getLogger()
                        .info("Server was not registered in control database. Will wait for server port identification...");
            }
            else if (currentServerRecord == null || !getServerNode().getPort().equals(currentServerRecord.getPort()))
            {
                DIFLogger.getLogger().info("Registering server in control database...");

                serverRecord = this.getDBService().getServerDataSet().query()
                        .equals(Server.Fields.IPADDRESS, getServerNode().getServerIP())
                        .equals(Server.Fields.PORT, StringUtils.toStringOrNull(getServerNode().getPort()))
                        .equals(Server.Fields.CONTEXTROOT, getServerNode().getContextRootID()).singleValue();

                if (serverRecord == null)
                {
                    // First time this server logs into the database
                    serverRecord = new Server();
                    serverRecord.setIpAddress(getServerNode().getServerIP());
                    serverRecord.setPort(StringUtils.toStringOrNull(getServerNode().getPort()));
                    serverRecord.setContextRoot(getServerNode().getContextRootID());
                    serverRecord.setMachineServerUid(
                            DIFStartupConfiguration.hasMachineIDNameChangePending() ? DIFStartupConfiguration
                                    .getNewMachineIDForConfigurationsToApply() : DIFStartupConfiguration
                                    .getMachineIDForConfigurations());
                    serverRecord.setName(StringUtils
                            .nvl(ServerManagerConfigurations.getInstance().getServerCustomName(),
                                    serverRecord.getMachineServerUid() + "|" + serverRecord.getContextRoot()));
                    serverRecord.setEndpointBaseUrl(getServerNode().getServerBaseURL());

                    serverRecord.setActive(true);
                    serverRecord.setMissing(false);
                    serverRecord.setBootTime(new Timestamp(DIFInitializer.bootTime.getTime()));
                    serverRecord.setLastSync(new Timestamp(new Date().getTime()));

                    serverRecord = this.getDBService().getServerDataSet().insert(serverRecord);

                    getServerNode().setServerManagementObject(serverRecord);
                    DIFLogger.getLogger().info("New server registered in control database...");
                }
                else
                {
                    // Existing record. New boot so log previous up time activity log and reinitialize
                    ServerActivityLog serverActivityLog = new ServerActivityLog();
                    serverActivityLog.setServer(serverRecord);
                    serverActivityLog.setBootTime(serverRecord.getBootTime());
                    serverActivityLog.setLastSync(serverRecord.getLastSync());
                    serverActivityLog = this.getDBService().getServerActivityLogDataSet().insert(serverActivityLog);

                    serverRecord.setActive(true);
                    serverRecord.setMissing(false);
                    serverRecord.setBootTime(new Timestamp(DIFInitializer.bootTime.getTime()));
                    serverRecord.setLastSync(new Timestamp(new Date().getTime()));
                    serverRecord.setMachineServerUid(
                            DIFStartupConfiguration.hasMachineIDNameChangePending() ? DIFStartupConfiguration
                                    .getNewMachineIDForConfigurationsToApply() : DIFStartupConfiguration
                                    .getMachineIDForConfigurations());
                    serverRecord.setName(StringUtils
                            .nvl(ServerManagerConfigurations.getInstance().getServerCustomName(),
                                    serverRecord.getMachineServerUid() + "|" + serverRecord.getContextRoot()));
                    serverRecord.setEndpointBaseUrl(getServerNode().getServerBaseURL());

                    serverRecord = this.getDBService().getServerDataSet().update(serverRecord);

                    getServerNode().setServerManagementObject(serverRecord);
                    DIFLogger.getLogger()
                            .info("Previous server registered in control database (restart processed in activity log)...");
                }

                // Refresh server list. May need to purge the current server from the list after registration if not
                // previously identified
                this.discoverServers();
            }
        }
        catch (DataSetException e)
        {
            throw new ServerManagerException(e);
        }
        catch (ConfigurationException e)
        {
            throw new ServerManagerException(e);
        }
    }

    /**
     * Send message.
     *
     * @param serverMessage the server message
     *
     * @return true, if successful
     *
     * @exception ServerManagerException the server manager exception
     * @see pt.digitalis.dif.servermanager.IServerManager#sendMessage(pt.digitalis.dif.servermanager.messages.IServerMessage)
     */
    public boolean sendMessage(IServerMessage serverMessage) throws ServerManagerException
    {
        Server serverRecord = (Server) this.getServerNode().getServerManagementObject();

        if (serverRecord == null)
        {
            DIFLogger.getLogger().warn("Server message " + serverMessage.getMessageTypeID() +
                                       " was not sent since this server is stil not registered!");

            return false;
        }
        else
        {
            try
            {
                serverMessage.setSenderServerID(serverRecord.getId());
                ObjectMapper mapper = new ObjectMapper();
                String messageString = mapper.writeValueAsString(serverMessage);

                return this.sendMessageToServers(messageString, serverMessage.getDestinationServersType());
            }
            catch (JsonProcessingException e)
            {
                throw new ServerManagerException(e);
            }
        }
    }

    /**
     * Send message to servers.
     *
     * @param messageString          the message string
     * @param destinationServersType the destination servers type
     *
     * @return true, if successful
     *
     * @exception ServerManagerException the server manager exception
     */
    private boolean sendMessageToServers(String messageString, ServerMessageDestinationServers destinationServersType)
            throws ServerManagerException
    {
        Server serverRecord = (Server) this.getServerNode().getServerManagementObject();

        if (serverRecord == null)
        {
            DIFLogger.getLogger().warn("Server message " + messageString +
                                       " was not sent since this server is stil not registered!");

            return false;
        }
        else
        {
            List<ServerApplicationNodeInstance> serversToSend;

            switch (destinationServersType)
            {
                case OTHER_APP_INSTANCES:
                    serversToSend = this.getOtherInstancesOfThisApp();
                    break;
                case SAME_SERVER_INSTANCE:
                    serversToSend = this.getAppsRunningOnThisServerInstance();
                    break;
                default:
                    serversToSend = serversList;
            }

            Timestamp when = new Timestamp(System.currentTimeMillis());

            for (ServerApplicationNodeInstance server : serversToSend)
            {
                if (server.getServerManagementObject() != null)
                {
                    ServerMessage messageRecord = new ServerMessage();
                    messageRecord.setServerByServerSenderId(serverRecord);
                    messageRecord.setMessage(messageString);
                    messageRecord.setWhen(when);
                    messageRecord.setProcessed(false);
                    messageRecord.setSuccess(false);
                    messageRecord.setElapsedTime(0L);
                    messageRecord.setServerByServerReceiverId((Server) server.getServerManagementObject());

                    try
                    {
                        this.getDBService().getServerMessageDataSet().insert(messageRecord);
                    }
                    catch (DataSetException e)
                    {
                        throw new ServerManagerException(e);
                    }
                }
            }

            // Send a notification for immediate actions by the notified servers when possible
            new NotifyServersOfNewMessagesToProcess(serversToSend).start();

            return true;
        }
    }
}
