package pt.digitalis.dif.utils.logging;

import org.hibernate.Query;
import org.hibernate.Session;
import pt.digitalis.dif.dem.managers.impl.audit.model.DIFDBAuditFactory;
import pt.digitalis.dif.dem.managers.impl.audit.model.data.AuditLog;
import pt.digitalis.dif.dem.managers.impl.model.DIFRepositoryFactory;
import pt.digitalis.dif.dem.managers.impl.model.data.AccessLog;
import pt.digitalis.dif.dem.managers.impl.model.data.ErrorLog;
import pt.digitalis.dif.startup.DIFErrorHandingConfiguration;
import pt.digitalis.dif.utils.jobs.RecurrentJob;
import pt.digitalis.dif.utils.logging.performance.PerformanceLoggerConfiguration;
import pt.digitalis.utils.common.Chronometer;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.DateUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * The type Logger purge job.
 */
public class LoggerPurgeJob extends RecurrentJob
{

    /**
     * The constant DELETE_CHUNK.
     */
    private final static long DELETE_CHUNK = 1000;

    /**
     * Execute each time boolean.
     *
     * @return the boolean
     */
    @Override
    protected boolean executeEachTime()
    {
        DIFLogger.getLogger().info("Purging DIF logs from database...");

        // Purge Audit Database Logs
        if (LoggingConfiguration.getInstance().getAuditDatabase())
            purgeOldEntries(DIFDBAuditFactory.getSession(), AuditLog.Fields.OPERATIONDATE,
                    AuditLog.class.getSimpleName(),
                    LoggingConfiguration.getInstance().getAuditDatabasePurgeMaxEntries(),
                    LoggingConfiguration.getInstance().getAuditDatabasePurgeMaxDays());

        // Purge Performance Logs
        if (PerformanceLoggerConfiguration.getInstance().getAuditRequests())
            purgeOldEntries(DIFRepositoryFactory.getSession(), AccessLog.Fields.ACCESSDATE,
                    AccessLog.class.getSimpleName(),
                    PerformanceLoggerConfiguration.getInstance().getAuditRequestsPurgeMaxEntries(),
                    PerformanceLoggerConfiguration.getInstance().getAuditRequestsPurgeMaxDays());

        // Purge Error Logs
        purgeOldEntries(DIFRepositoryFactory.getSession(), ErrorLog.Fields.ERRODATE, ErrorLog.class.getSimpleName(),
                DIFErrorHandingConfiguration.getInstance().getErrorLogTablePurgeMaxEntries(),
                DIFErrorHandingConfiguration.getInstance().getErrorLogTablePurgeMaxDays());

        return true;
    }

    /**
     * Gets default run interval in seconds.
     *
     * @return the default run interval in seconds
     */
    @Override
    protected Long getDefaultRunIntervalInSeconds()
    {
        return LoggingConfiguration.getInstance().getPurgeJobInterval();
    }

    /**
     * Has entries to clean boolean.
     *
     * @param session           the session
     * @param dateField         the date field
     * @param tableName         the table name
     * @param maxEntriesAllowed the max entries allowed
     * @param maxDaysAllowed    the max days allowed
     */
    private void purgeOldEntries(Session session, String dateField, String tableName, Long maxEntriesAllowed,
            Long maxDaysAllowed)
    {
        if (maxEntriesAllowed > 0 || maxDaysAllowed > 0)
        {
            Chronometer totalTime = new Chronometer();
            Chronometer commandTime = new Chronometer();

            totalTime.start();

            boolean wasActive = session.getTransaction().isActive();

            if (!wasActive)
                session.beginTransaction();

            DIFLogger.getLogger()
                    .info("   => Analyzing " + tableName + " logs from database [maxDays:" + maxDaysAllowed +
                          " | maxEntries:" + maxEntriesAllowed + "]...");

            Object[] queryResult =
                    (Object[]) session.createQuery("select count(*), max(id), min(" + dateField + ") from " + tableName)
                            .list().get(0);

            Long totalEntries = (Long) queryResult[0];
            Long minID;
            Long maxID = (Long) queryResult[1];
            Date minDate = (Date) queryResult[2];
            Date now = new Date();

            DIFLogger.getLogger().debug("       => Oldest log found: " + DateUtils.simpleDateTimeToString(minDate));
            DIFLogger.getLogger().debug("       => Total logs found: " + totalEntries);

            boolean purgeRecordsTooManyEntries = maxEntriesAllowed > 0 && totalEntries > maxEntriesAllowed;
            boolean purgeRecordsEntriesTooOld =
                    maxDaysAllowed > 0 && DateUtils.getDateDiffInDays(minDate, now) > maxDaysAllowed;

            if (purgeRecordsEntriesTooOld || purgeRecordsTooManyEntries)
            {
                long entriesPurged;
                Date oldestDateToKeep = null;

                List<String> filters = new ArrayList<String>();
                if (purgeRecordsTooManyEntries)
                {
                    // Last existing ID minus max to keep. Delete bellow this ID
                    filters.add("id < " + (maxID - maxEntriesAllowed));
                }

                if (purgeRecordsEntriesTooOld)
                {
                    // Max date to keep is today minus max days to keep. Delete previous to this date
                    oldestDateToKeep = DateUtils.addDays(now, -maxDaysAllowed.intValue());
                    filters.add(dateField + " < :oldestDateToKeep");
                }

                // Determine the min and max of the records to delete
                String hql = "select min(id), max(id) from " + tableName + " where " +
                             CollectionUtils.listToSeparatedString(filters, " and ");
                Query query = session.createQuery(hql);

                if (oldestDateToKeep != null)
                    query.setParameter("oldestDateToKeep", oldestDateToKeep);

                queryResult = (Object[]) query.uniqueResult();
                minID = (Long) queryResult[0];
                maxID = (Long) queryResult[1];

                // Delete DELETE_CHUNK at a time with commits in between
                long totalEntriesPurged = 0;

                while (minID < maxID)
                {
                    long maxIDToDelete = minID + DELETE_CHUNK;
                    if (maxIDToDelete > maxID)
                        maxIDToDelete = maxID;

                    commandTime.start();
                    entriesPurged = session.createQuery("delete from " + tableName + " where id < " + maxIDToDelete)
                            .executeUpdate();

                    // Partial commit to prevent long transaction and undo files
                    session.getTransaction().commit();
                    session = session.getSessionFactory().getCurrentSession();
                    session.beginTransaction();

                    totalEntriesPurged += entriesPurged;
                    minID += DELETE_CHUNK;

                    DIFLogger.getLogger()
                            .info("  => Log PURGE (" + tableName + "): total records purged: " + entriesPurged +
                                  " in " + commandTime.getTimePassedAsFormattedString());
                }

                if (totalEntriesPurged > 0)
                    DIFLogger.getLogger()
                            .info("   => Log " + tableName + " purged. " + totalEntriesPurged + " deleted in " +
                                  totalTime.getTimePassedAsFormattedString() + ".");
            }

            if (wasActive)
                session.getTransaction().commit();
        }
    }
}
