/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.documents.repository;

import net.sf.jmimemagic.Magic;
import net.sf.jmimemagic.MagicMatch;
import org.hibernate.Session;
import pt.digitalis.dif.documents.model.DocumentsFactory;
import pt.digitalis.dif.documents.model.IDocumentsService;
import pt.digitalis.dif.documents.model.data.Documents;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.model.dataset.DataSetException;
import pt.digitalis.dif.model.dataset.Filter;
import pt.digitalis.dif.model.dataset.FilterType;
import pt.digitalis.dif.model.dataset.Query;
import pt.digitalis.dif.utils.extensions.document.AbstractDocumentRepository;
import pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry;
import pt.digitalis.dif.utils.extensions.document.DocumentRepositoryException;
import pt.digitalis.dif.utils.extensions.document.MaximumDocumentSizeException;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.ConfigurationException;

import java.io.IOException;
import java.sql.Blob;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a><br/>
 * @created Nov 6, 2009
 */
public class DocumentRepositoryDBImpl extends AbstractDocumentRepository
{

    /** The document size limit */
    private final Integer maxDocumentSize;

    /**
     * Constructor
     *
     * @exception ConfigurationException
     */
    public DocumentRepositoryDBImpl() throws ConfigurationException
    {
        this(DocumentRepositoryConfiguration.getInstance().getMaxDocumentSize());
    }

    /**
     * Constructor
     *
     * @param maxDocumentSize
     */
    public DocumentRepositoryDBImpl(Integer maxDocumentSize)
    {
        super();
        this.maxDocumentSize = maxDocumentSize;
    }

    /**
     * Convert the {@link DocumentRepositoryEntry} object to {@link Documents}
     *
     * @param document
     *
     * @return {@link Documents}
     */
    private synchronized static Documents convertToDocument(DocumentRepositoryEntry document)
    {

        Documents entry = new Documents();
        entry.setDocument(DocumentsFactory.getLobHelper().createBlob(document.getBytes()));
        entry.setCreationDate(document.getCreationDate());
        entry.setExpireOn(document.getExpireOn());
        entry.setCreator(document.getCreatorID());
        entry.setDescription(document.getDescription());
        entry.setFilename(document.getFileName());
        entry.setId(document.getId());
        entry.setMimeType(document.getMimeType());
        entry.setName(document.getName());

        return entry;
    }

    /**
     * Convert the {@link Documents} object to {@link DocumentRepositoryEntry}.
     *
     * @param document the document
     *
     * @return {@link DocumentRepositoryEntry}
     *
     * @exception SQLException the SQL exception
     * @exception IOException  Signals that an I/O exception has occurred.
     */
    private synchronized static DocumentRepositoryEntry convertToDocumentRepositoryEntry(Documents document)
            throws SQLException, IOException
    {
        return convertToDocumentRepositoryEntry(document, false);
    }

    /**
     * Convert the {@link Documents} object to {@link DocumentRepositoryEntry}.
     *
     * @param document     the document
     * @param lazyLoadFile the lazy load file
     *
     * @return {@link DocumentRepositoryEntry}
     *
     * @exception SQLException the SQL exception
     * @exception IOException  Signals that an I/O exception has occurred.
     */
    private synchronized static DocumentRepositoryEntry convertToDocumentRepositoryEntry(Documents document,
            boolean lazyLoadFile) throws SQLException, IOException
    {
        DocumentRepositoryEntryDB entry = new DocumentRepositoryEntryDB();
        if (!lazyLoadFile)
        {
            Blob aBlob = document.getDocument();

            byte[] bytes = new byte[new Long(aBlob.length()).intValue()];
            aBlob.getBinaryStream().read(bytes);

            entry.setBytes(bytes);

            if (StringUtils.isNotBlank(document.getMimeType()))
            {
                entry.setMimeType(document.getMimeType());
            }
            else
            {
                try
                {
                    MagicMatch match = Magic.getMagicMatch(entry.getBytes());
                    entry.setMimeType(match.getMimeType());
                }
                catch (Exception e)
                {
                    return null;
                }
            }
        }
        else
        {
            entry.setaBlob(document.getDocument());
            entry.setMimeType(document.getMimeType());
        }
        entry.setCreationDate(document.getCreationDate());
        entry.setExpireOn(document.getExpireOn());
        entry.setCreatorID(document.getCreator());
        entry.setDescription(document.getDescription());
        entry.setFileName(document.getFilename());
        entry.setId(document.getId());

        entry.setName(document.getName());

        return entry;
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#addDocument(pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry)
     */
    @Override
    public synchronized DocumentRepositoryEntry addDocument(DocumentRepositoryEntry arg0)
            throws DocumentRepositoryException
    {
        return this.addDocument(arg0, false);
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#addDocument(pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry,
     *         java.lang.Boolean)
     */
    @Override
    public synchronized DocumentRepositoryEntry addDocument(DocumentRepositoryEntry documentEntry,
            Boolean ignoreSizeLimit) throws DocumentRepositoryException
    {

        /* Validates if document exceeds the maximum size defined */
        if (ignoreSizeLimit || maxDocumentSize == null ||
            (documentEntry.getBytes().length / 1024) <= maxDocumentSize.intValue())
        {
            Documents document;
            document = convertToDocument(documentEntry);
            IDocumentsService documentsService =
                    DIFIoCRegistry.getRegistry().getImplementation(IDocumentsService.class);
            boolean isActive = documentsService.getDocumentsDataSet().getSession().getTransaction().isActive();
            try
            {
                if (!isActive)
                {
                    documentsService.getDocumentsDataSet().getSession().beginTransaction();
                }
                documentsService.getDocumentsDataSet().insert(document);

                documentEntry = convertToDocumentRepositoryEntry(document);
                if (!isActive)
                {
                    documentsService.getDocumentsDataSet().getSession().getTransaction().commit();
                }
            }
            catch (Exception e)
            {
                if (!isActive)
                {
                    documentsService.getDocumentsDataSet().getSession().getTransaction().rollback();
                }
                throw new DocumentRepositoryException("There was a problem inserting document on database " + e);
            }
        }
        else
        {
            throw new MaximumDocumentSizeException("The Document exceeds the maximum size of " + maxDocumentSize + "Kb",
                    maxDocumentSize);
        }
        return documentEntry;
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#deleteDocument(java.lang.Long)
     */
    @Override
    public synchronized void deleteDocument(Long documentID) throws DocumentRepositoryException
    {
        IDocumentsService documentsService = DIFIoCRegistry.getRegistry().getImplementation(IDocumentsService.class);
        try
        {
            documentsService.getDocumentsDataSet().delete(documentID.toString());
        }
        catch (DataSetException e)
        {
            throw new DocumentRepositoryException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#getDocument(java.lang.Long)
     */
    @Override
    public synchronized DocumentRepositoryEntry getDocument(Long documentID) throws DocumentRepositoryException
    {
        return this.getDocument(documentID, true);
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#getDocument(java.lang.Long,
     *         boolean)
     */
    @Override
    public DocumentRepositoryEntry getDocument(Long documentID, boolean lazyLoadFile) throws DocumentRepositoryException
    {
        if (documentID == null)
        {
            return null;
        }

        IDocumentsService documentsService = DIFIoCRegistry.getRegistry().getImplementation(IDocumentsService.class);

        Session session = documentsService.getDocumentsDataSet().getSession();
        Boolean wasActive = session.getTransaction().isActive();

        try
        {

            if (!wasActive)
            {
                documentsService.getDocumentsDataSet().getSession().beginTransaction();
            }

            Documents document = documentsService.getDocumentsDataSet().get(documentID.toString());
            DocumentRepositoryEntry entry = null;
            if (document != null)
            {
                entry = convertToDocumentRepositoryEntry(document, lazyLoadFile);
            }

            if (!wasActive)
            {
                documentsService.getDocumentsDataSet().getSession().getTransaction().commit();
            }

            return entry;
        }
        catch (SQLException e)
        {
            if (!wasActive)
            {
                documentsService.getDocumentsDataSet().getSession().getTransaction().rollback();
            }

            throw new DocumentRepositoryException(e);
        }
        catch (IOException e)
        {
            if (!wasActive)
            {
                documentsService.getDocumentsDataSet().getSession().getTransaction().rollback();
            }
            throw new DocumentRepositoryException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#getDocumentByOriginalFileName(java.lang.String)
     */
    @Override
    public synchronized List<DocumentRepositoryEntry> getDocumentByOriginalFileName(String fileName)
            throws DocumentRepositoryException
    {
        IDocumentsService documentsService = DIFIoCRegistry.getRegistry().getImplementation(IDocumentsService.class);

        try
        {
            Query<Documents> query = documentsService.getDocumentsDataSet().query();
            query.addFilter(new Filter(Documents.Fields.FILENAME.toString(), FilterType.EQUALS, fileName));

            List<DocumentRepositoryEntry> result = new ArrayList<DocumentRepositoryEntry>();
            for (Documents document : query.asList())
            {
                result.add(convertToDocumentRepositoryEntry(document));
            }
            return result;
        }
        catch (SQLException e)
        {
            throw new DocumentRepositoryException(e);
        }
        catch (IOException e)
        {
            throw new DocumentRepositoryException(e);
        }
        catch (DataSetException e)
        {
            throw new DocumentRepositoryException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#getDocumentsByCreator(java.lang.String)
     */
    @Override
    public synchronized List<DocumentRepositoryEntry> getDocumentsByCreator(String creatorID)
            throws DocumentRepositoryException
    {
        IDocumentsService documentsService = DIFIoCRegistry.getRegistry().getImplementation(IDocumentsService.class);

        try
        {
            Query<Documents> query = documentsService.getDocumentsDataSet().query();
            query.addFilter(new Filter(Documents.Fields.CREATOR.toString(), FilterType.EQUALS, creatorID));

            List<DocumentRepositoryEntry> result = new ArrayList<DocumentRepositoryEntry>();
            for (Documents document : query.asList())
            {
                result.add(convertToDocumentRepositoryEntry(document));
            }
            return result;
        }
        catch (SQLException e)
        {
            throw new DocumentRepositoryException(e);
        }
        catch (IOException e)
        {
            throw new DocumentRepositoryException(e);
        }
        catch (DataSetException e)
        {
            throw new DocumentRepositoryException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.AbstractDocumentRepository#getDocumentsByIds(java.util.List)
     */
    @Override
    public synchronized List<DocumentRepositoryEntry> getDocumentsByIds(List<Long> documentIds)
            throws DocumentRepositoryException
    {

        IDocumentsService documentsService = DIFIoCRegistry.getRegistry().getImplementation(IDocumentsService.class);
        List<DocumentRepositoryEntry> documents = new ArrayList<DocumentRepositoryEntry>();

        try
        {
            Query<Documents> query = documentsService.getDocumentsDataSet().query();
            query.addFilter(new Filter(Documents.Fields.ID.toString(), FilterType.IN,
                    CollectionUtils.listToCommaSeparatedString(documentIds)));

            for (Documents document : query.asList())
            {
                DocumentRepositoryEntry entry = convertToDocumentRepositoryEntry(document);
                documents.add(entry);
            }
        }
        catch (SQLException e)
        {
            throw new DocumentRepositoryException(e);
        }
        catch (IOException e)
        {
            throw new DocumentRepositoryException(e);
        }
        catch (DataSetException e)
        {
            throw new DocumentRepositoryException(e);
        }

        return documents;
    }

    /**
     * Gets documents by mime type.
     *
     * @param mimeType the mime type
     *
     * @return the documents by mime type
     *
     * @exception DocumentRepositoryException the document repository exception
     */
    @Override
    public synchronized List<DocumentRepositoryEntry> getDocumentsByMimeType(String mimeType)
            throws DocumentRepositoryException
    {
        IDocumentsService documentsService = DIFIoCRegistry.getRegistry().getImplementation(IDocumentsService.class);

        try
        {

            Query<Documents> query = documentsService.getDocumentsDataSet().query();
            query.addFilter(new Filter(Documents.Fields.MIMETYPE.toString(), FilterType.EQUALS, mimeType));

            List<DocumentRepositoryEntry> result = new ArrayList<DocumentRepositoryEntry>();
            for (Documents document : query.asList())
            {
                result.add(convertToDocumentRepositoryEntry(document));
            }
            return result;
        }

        catch (SQLException e)
        {
            throw new DocumentRepositoryException(e);
        }
        catch (IOException e)
        {
            throw new DocumentRepositoryException(e);
        }
        catch (DataSetException e)
        {
            throw new DocumentRepositoryException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#updateDocument(pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry)
     */
    @Override
    public synchronized DocumentRepositoryEntry updateDocument(DocumentRepositoryEntry documentEntry)
    {
        IDocumentsService documentsService = DIFIoCRegistry.getRegistry().getImplementation(IDocumentsService.class);
        Documents document = convertToDocument(documentEntry);

        boolean isActive = documentsService.getDocumentsDataSet().getSession().getTransaction().isActive();
        if (!isActive)
        {
            documentsService.getDocumentsDataSet().getSession().beginTransaction();
        }
        documentsService.getDocumentsDataSet().getSession().merge(document);
        // Hack: Must set the BLOB again so that hibernate does not maintain the previous content of the BLOB
        document.setDocument(DocumentsFactory.getLobHelper().createBlob(documentEntry.getBytes()));

        if (!isActive)
        {
            documentsService.getDocumentsDataSet().getSession().getTransaction().commit();
        }

        try
        {
            documentEntry = getDocument(documentEntry.getId());
        }
        catch (DocumentRepositoryException e)
        {
            e.printStackTrace();
        }

        return documentEntry;
    }
}
