/**
 * 2009, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.documents.repository;

import java.io.File;
import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.CheckIndex;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.TopDocCollector;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.LockObtainFailedException;

import pt.digitalis.dif.utils.extensions.document.AbstractDocumentRepository;
import pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry;
import pt.digitalis.dif.utils.extensions.document.MaximumDocumentSizeException;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.log.ILogWrapper;
import pt.digitalis.utils.common.DateUtils;

/**
 * @author Galaio da Silva <a href="mailto:jgalaio@digitalis.pt">jgalaio@digitalis.pt</a><br/>
 * @created Jun 15, 2009
 */
public class DocumentRepositoryLuceneImpl extends AbstractDocumentRepository {

    /**
     * Current Document id.
     */
    private static Long documentId;

    /**
     * The suffix for index dir
     */
    private static final String INDEX_SUFFIX_DIR = "lucene_index";

    /** The writer to access index */
    public static IndexWriter writer;

    /** A <code>StandardAnalyzer</code> */
    StandardAnalyzer analizer = new StandardAnalyzer();

    /** the indexFile Directory */
    File indexDirectory;

    /**
     * The index path
     */
    private String indexPath;

    /** The logger instance */
    protected ILogWrapper logger;

    /** The document size limit */
    private Integer maxDocumentSize;

    /**
     * Constructor for <code>DocumentRepositoryLuceneImpl</code>.
     */
    public DocumentRepositoryLuceneImpl()
    {
        this("".equals(LuceneConfiguration.getInstance().getIndexPath()) ? System.getProperty("user.home") + "/"
                + INDEX_SUFFIX_DIR : LuceneConfiguration.getInstance().getIndexPath(), LuceneConfiguration
                .getInstance().getMaxDocumentSize());
    }

    /**
     * Constructor for <code>DocumentRepositoryLuceneImpl</code>
     * 
     * @param maxDocumentSize
     */
    public DocumentRepositoryLuceneImpl(Integer maxDocumentSize)
    {
        this("".equals(LuceneConfiguration.getInstance().getIndexPath()) ? System.getProperty("user.home") + "/"
                + INDEX_SUFFIX_DIR : LuceneConfiguration.getInstance().getIndexPath(), maxDocumentSize);
    }

    /**
     * Constructor for <code>DocumentRepositoryLuceneImpl</code>
     * 
     * @param indexPath
     *            the idex path
     * @param maxDocumentSize
     */
    public DocumentRepositoryLuceneImpl(String indexPath, Integer maxDocumentSize)
    {
        this.indexPath = indexPath;
        this.logger = DIFLogger.getLogger();
        this.maxDocumentSize = maxDocumentSize;
        this.initialize();
    }

    /**
     * @throws MaximumDocumentSizeException
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#addDocument(pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry)
     */
    public synchronized DocumentRepositoryEntry addDocument(DocumentRepositoryEntry document)
            throws MaximumDocumentSizeException
    {
        return addDocument(document, false);
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#addDocument(pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry,
     *      java.lang.Boolean)
     */
    public DocumentRepositoryEntry addDocument(DocumentRepositoryEntry document, Boolean ignoreSizeLimit)
            throws MaximumDocumentSizeException
    {
        /*
         * StandardAnalyzer analyzer = new StandardAnalyzer(); File indexDirectory = new File(this.indexPath); boolean
         * exists = indexDirectory.exists();
         */
        document.setId(getNextDocumentId());
        Document doc = this.buildDocument(document);

        /* Validates if document exceeds the maximum size defined */
        if (ignoreSizeLimit || this.maxDocumentSize == null
                || (document.getBytes().length / 1024) <= this.maxDocumentSize.intValue())
        {
            try
            {
                // IndexWriter writer = new IndexWriter(this.indexPath, analyzer,
                // !exists,IndexWriter.MaxFieldLength.UNLIMITED);

                writer.addDocument(doc);
                writer.commit();
                // writer.close();
            }
            catch (CorruptIndexException e)
            {
                e.printStackTrace();
            }
            catch (LockObtainFailedException e)
            {
                this.recoverFromCorrupt();
            }
            catch (IOException e)
            {
                e.printStackTrace();
            }
            finally
            {
                document.setId(new Long(doc.get(DocumentRepositoryEntry.Fields.ID.toString())));
            }
        }
        else
        {
            throw new MaximumDocumentSizeException(
                    "The Document exceeds the maximum size of " + maxDocumentSize + "Kb", maxDocumentSize);
        }

        return document;
    }

    /**
     * Transforms a <code>DocumentRepositoryEntry</code> into Lucene <code>Document</code>.
     * 
     * @param document
     *            the <code>DocumentRepositoryEntry</code>.
     * @return Lucene <code>Document</code>.
     */
    synchronized private Document buildDocument(DocumentRepositoryEntry document)
    {
        Document luceneDocument = new Document();
        luceneDocument.add(new Field(DocumentRepositoryEntry.Fields.ID.toString(), document.getId().toString(),
                Field.Store.YES, Field.Index.ANALYZED));
        if (document.getCreatorID() != null)
            luceneDocument.add(new Field(DocumentRepositoryEntry.Fields.CREATOR.toString(), document.getCreatorID(),
                    Field.Store.YES, Field.Index.ANALYZED));
        if (document.getName() != null)
            luceneDocument.add(new Field(DocumentRepositoryEntry.Fields.NAME.toString(), document.getName(),
                    Field.Store.YES, Field.Index.ANALYZED));
        if (document.getDescription() != null)
            luceneDocument.add(new Field(DocumentRepositoryEntry.Fields.DESCRIPTION.toString(), document
                    .getDescription(), Field.Store.YES, Field.Index.ANALYZED));
        luceneDocument.add(new Field(DocumentRepositoryEntry.Fields.FILENAME.toString(), document.getFileName(),
                Field.Store.YES, Field.Index.ANALYZED));
        luceneDocument.add(new Field(DocumentRepositoryEntry.Fields.CREATION_DATE.toString(), DateUtils
                .dateToString(document.getCreationDate()), Field.Store.YES, Field.Index.ANALYZED));
        luceneDocument.add(new Field(DocumentRepositoryEntry.Fields.MIMETYPE.toString(), document.getMimeType(),
                Field.Store.YES, Field.Index.ANALYZED));
        luceneDocument.add(new Field(DocumentRepositoryEntry.Fields.BYTE_CONTENT.toString(), document.getBytes(),
                Field.Store.YES));
        return luceneDocument;
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#deleteDocument(java.lang.Long)
     */
    public synchronized void deleteDocument(Long id)
    {
        // IndexReader indexReader;
        try
        {
            Term term = new Term(DocumentRepositoryEntry.Fields.ID.toString(), id.toString());

            writer.deleteDocuments(term);
            writer.commit();
            /*
             * indexReader = IndexReader.open(indexPath); indexReader.deleteDocument(new Integer(id).intValue());
             * indexReader.close();
             */
        }
        catch (CorruptIndexException e)
        {
            this.recoverFromCorrupt();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }

    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#getDocument(java.lang.Long)
     */
    public synchronized DocumentRepositoryEntry getDocument(Long id)
    {
        ArrayList<DocumentRepositoryEntry> docs = search(DocumentRepositoryEntry.Fields.ID.toString(), id.toString());
        if (docs.size() != 0)
        {
            return docs.get(0);
        }
        return null;
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#getDocumentByOriginalFileName(java.lang.String)
     */
    public List<DocumentRepositoryEntry> getDocumentByOriginalFileName(String originalFileName)
    {
        return search(DocumentRepositoryEntry.Fields.FILENAME.toString(), originalFileName);
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#getDocumentsByCreator(java.lang.String)
     */
    public List<DocumentRepositoryEntry> getDocumentsByCreator(String creatorUserID)
    {
        ArrayList<DocumentRepositoryEntry> documents = search(DocumentRepositoryEntry.Fields.CREATOR.toString(),
                creatorUserID);
        return documents;
    }

    /**
     * Increments the document identifier
     * 
     * @return the next document identifier
     */
    synchronized private Long getNextDocumentId()
    {
        do
        {
            documentId = documentId.longValue() + 1;
        }
        while (getDocument(documentId) != null);

        /* Writer Optimization when 50 documents are generated */
        if ((documentId % 50) == 0)
        {
            try
            {
                writer.optimize();
            }
            catch (CorruptIndexException e)
            {
                logger.debug("Error while trying to optimize the lucene index");
            }
            catch (IOException e)
            {
                logger.debug("Error while trying to optimize the lucene index");
            }
        }

        return documentId;
    }

    // Keep it for future implementation. We don't yet know if we will need it.
    // /**
    // * Transform <code>Document</code> into <code>DocumentRepositoryEntry</code>.
    // *
    // * @param document
    // * the lucene <code>Document</code>
    // * @param id
    // * the document internal id.
    // * @return <code>DocumentRepositoryEntry</code>
    // * @throws ParseException
    // */
    // private DocumentRepositoryEntry buildDocumentRepositoryEntry(Document document, String id) throws ParseException
    // {
    // DocumentRepositoryEntry documentRepEntry = new DocumentRepositoryEntry();
    // documentRepEntry.setId(id);
    // documentRepEntry.setCreatorID(document.get(DocumentRepositoryEntry.Fields.CREATOR.toString()));
    // documentRepEntry.setName(document.get(DocumentRepositoryEntry.Fields.NAME.toString()));
    // documentRepEntry.setDescription(document.get(DocumentRepositoryEntry.Fields.DESCRIPTION.toString()));
    // documentRepEntry.setFileName(document.get(DocumentRepositoryEntry.Fields.FILENAME.toString()));
    // documentRepEntry.setCreationDate(DateUtils.stringToDate(document
    // .get(DocumentRepositoryEntry.Fields.CREATION_DATE.toString())));
    // documentRepEntry.setMimeType(document.get(DocumentRepositoryEntry.Fields.MIMETYPE.toString()));
    // documentRepEntry.setBytes(document.getBinaryValue(DocumentRepositoryEntry.Fields.BYTE_CONTENT.toString()));
    // return documentRepEntry;
    // }

    /**
     * Initialize the document identifier
     * 
     * @param value
     *            the initial value
     */
    synchronized private void initializaDocumentId(Long value)
    {
        documentId = value;
    }

    /**
     * Initialize this Lucene Implementation.
     */
    private void initialize()
    {
        this.logger = DIFLogger.getLogger();
        if (writer == null)
        {
            IndexWriter tempWriter = null;
            indexDirectory = new File(indexPath);
            boolean exists = indexDirectory.exists();
            try
            {
                if (exists)
                {
                    this.unLockIndex();
                }
                tempWriter = new IndexWriter(this.indexPath, this.analizer, !exists, IndexWriter.MaxFieldLength.LIMITED);
                tempWriter.setMaxFieldLength(1);
                tempWriter.optimize();

                initializaDocumentId(new Long(tempWriter.maxDoc()));
                writer = tempWriter;
            }
            catch (CorruptIndexException e)
            {
                this.recoverFromCorrupt();
            }
            catch (LockObtainFailedException e)
            {
                e.printStackTrace();
            }
            catch (IOException e)
            {
                e.printStackTrace();
            }
            finally
            {
                logger.debug("The writer was created");
            }
        }
    }

    /**
     * Recover a corrupt index.
     */
    private synchronized void recoverFromCorrupt()
    {
        logger.debug("The Index is corrupt, starting recover!");
        Directory directory;
        CheckIndex.Status status = null;
        try
        {
            directory = FSDirectory.getDirectory(indexPath);
            CheckIndex checkIndex = new CheckIndex(directory);
            status = checkIndex.checkIndex();
            checkIndex.fixIndex(status);
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
        finally
        {
            logger.debug("The Index recovery was finished!");
            logger.debug("Bad Segments:" + status.numBadSegments);
        }
    }

    /**
     * Generic search method.
     * 
     * @param field
     *            the field to search
     * @param value
     *            the value to search
     * @return <code>DocumentRepositoryEntry</code>.
     */
    private ArrayList<DocumentRepositoryEntry> search(String field, String value)
    {
        ArrayList<DocumentRepositoryEntry> documents = new ArrayList<DocumentRepositoryEntry>();
        int hitsPerPage = 1000;
        try
        {
            IndexSearcher searcher;
            searcher = new IndexSearcher(this.indexPath);
            TopDocCollector collector = new TopDocCollector(hitsPerPage);

            Query query = new QueryParser("", this.analizer).parse(field + ":" + value);
            searcher.search(query, collector);
            ScoreDoc[] hits = collector.topDocs().scoreDocs;

            for (int i = 0; i < hits.length; ++i)
            {
                int docId = hits[i].doc;
                Document doc = searcher.doc(docId);

                try
                {
                    documents.add(new DocumentRepositoryEntry(new Long(doc.get(DocumentRepositoryEntry.Fields.ID
                            .toString())), doc.get(DocumentRepositoryEntry.Fields.CREATOR.toString()), doc
                            .get(DocumentRepositoryEntry.Fields.NAME.toString()), doc
                            .get(DocumentRepositoryEntry.Fields.DESCRIPTION.toString()), doc
                            .get(DocumentRepositoryEntry.Fields.FILENAME.toString()), DateUtils.stringToDate(doc
                            .get(DocumentRepositoryEntry.Fields.CREATION_DATE.toString())), doc
                            .get(DocumentRepositoryEntry.Fields.MIMETYPE.toString()), doc
                            .getBinaryValue(DocumentRepositoryEntry.Fields.BYTE_CONTENT.toString())));
                }
                catch (ParseException e)
                {
                    e.printStackTrace();
                }
            }
        }
        catch (CorruptIndexException e)
        {
            this.recoverFromCorrupt();
        }
        catch (org.apache.lucene.queryParser.ParseException e)
        {
            e.printStackTrace();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
        return documents;
    }

    /**
     * Unlock the writer.
     */
    private synchronized void unLockIndex()
    {
        try
        {
            if (IndexWriter.isLocked(indexPath))
            {
                this.logger.debug("Starting unlock Writer!");
                Directory directory = FSDirectory.getDirectory(indexPath);
                IndexWriter.unlock(directory);
                this.logger.debug("The writer was unlocked successfully!");
            }
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
    }

    /**
     * @see pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager#updateDocument(pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry)
     */
    public synchronized DocumentRepositoryEntry updateDocument(DocumentRepositoryEntry document)
    {
        try
        {
            /*
             * IndexWriter writer = new IndexWriter(this.indexPath, this.analizer, true,
             * IndexWriter.MaxFieldLength.UNLIMITED);
             */
            writer.updateDocument(new Term(DocumentRepositoryEntry.Fields.ID.toString(), document.getId().toString()),
                    this.buildDocument(document));
            writer.commit();
            // writer.close();
        }
        catch (CorruptIndexException e)
        {
            this.recoverFromCorrupt();
        }
        catch (LockObtainFailedException e)
        {
            e.printStackTrace();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
        return null;
    }

}
