/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.identity.repository;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import pt.digitalis.dif.controller.security.managers.impl.AbstractIdentityManager;
import pt.digitalis.dif.controller.security.objects.DIFGroupImpl;
import pt.digitalis.dif.controller.security.objects.DIFUserImpl;
import pt.digitalis.dif.controller.security.objects.IDIFGroup;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.dem.managers.impl.model.IIdentityService;
import pt.digitalis.dif.dem.managers.impl.model.data.Groups;
import pt.digitalis.dif.dem.managers.impl.model.data.UserAttributes;
import pt.digitalis.dif.dem.managers.impl.model.data.UserGroups;
import pt.digitalis.dif.dem.managers.impl.model.data.Users;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.model.dataset.DataSetException;
import pt.digitalis.dif.model.dataset.Filter;
import pt.digitalis.dif.model.dataset.FilterType;
import pt.digitalis.dif.model.dataset.JoinType;
import pt.digitalis.dif.model.dataset.Query;
import pt.digitalis.dif.utils.Pagination;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.common.collections.CaseInsensitiveHashMap;
import pt.digitalis.utils.crypto.IEncryptor;
import pt.digitalis.utils.crypto.exeption.CryptoException;
import pt.digitalis.utils.crypto.impl.EncryptorBase64Impl;

/**
 * Identity manager implementation based on Database.
 *
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a>
 * @created Oct 11, 2012
 */
public class IdentityManagerDBImpl extends AbstractIdentityManager {

    /** Active Encryptor implementation. */
    private static IEncryptor encryptor = null;

    /** just a variable */
    private static final char[] hideInPlainSight = {(char) 97, (char) 115, (char) 100, (char) 97, (char) 97, (char) 115,
                        (char) 100, (char) 102, (char) 56, (char) 115, (char) 97, (char) 100, (char) 57, (char) 102,
                        (char) 56, (char) 55, (char) 97, (char) 115};

    /** The Constant NAME. */
    public static final String NAME = "Database";

    /** The variable to return */
    private static final String variableToReturn = new String(hideInPlainSight);

    /** The identity service. */
    IIdentityService identityService = DIFIoCRegistry.getRegistry().getImplementation(IIdentityService.class);

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#addGroup(pt.digitalis.dif.controller.security.objects.IDIFGroup)
     */
    public void addGroup(IDIFGroup newGroup) throws IdentityManagerException
    {
        Groups group = convertFromDIFGroupToDBGroup(newGroup);
        try
        {
            identityService.getGroupsDataSet().insert(group);
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#addUser(pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public void addUser(IDIFUser newUser) throws IdentityManagerException
    {
        // If the user password is not defined, by default the password is "password"
        if (((DIFUserImpl) newUser).getPassword() == null)
        {
            newUser.setPassword("password");
        }

        Users dbUser = convertFromDIFUserToDBUser(newUser);
        try
        {
            dbUser = identityService.getUsersDataSet().insert(dbUser);
            processFromDIFUserToDBUserAttributes(newUser, dbUser);
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#addUserToGroup(java.lang.String,
     *      java.lang.String)
     */
    public void addUserToGroup(String userID, String groupID) throws IdentityManagerException
    {
        try
        {
            Users user = identityService.getUsersDataSet().get(userID);
            Groups group = identityService.getGroupsDataSet().get(groupID);

            UserGroups userGroups = new UserGroups();
            userGroups.setGroups(group);
            userGroups.setUsers(user);

            identityService.getUserGroupsDataSet().insert(userGroups);
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }

    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#changePassword(java.lang.String,
     *      java.lang.String)
     */
    public void changePassword(String userID, String newPassword) throws IdentityManagerException
    {
        try
        {
            Users user = identityService.getUsersDataSet().get(userID);
            user.setPassword(getEncryptator().encrypt(newPassword));
            identityService.getUsersDataSet().update(user);
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
        catch (CryptoException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * Converts an DB group into a DIF group.
     *
     * @param dbGroup
     *            the DB group
     * @return the DIF group
     * @throws IdentityManagerException
     *             if the operation does not succeed
     */
    private IDIFGroup convertFromDBGroupToDIFGroup(Groups dbGroup) throws IdentityManagerException
    {
        IDIFGroup difGroup = null;

        if (dbGroup != null)
        {
            difGroup = new DIFGroupImpl();

            difGroup.setID(dbGroup.getId());

            if (dbGroup.getName() != null)
                difGroup.setName(dbGroup.getName());

            difGroup.setParentGroupID(N_A);
            if (dbGroup.getGroups() != null)
                difGroup.setParentGroupID(dbGroup.getGroups().getId());

            if (dbGroup.getDescription() != null)
                difGroup.setDescription(dbGroup.getDescription());

        }

        return difGroup;
    }

    /**
     * Converts an BD user into a DIF user.
     *
     * @param dbUser
     *            the DB user
     * @return the DIF user
     * @throws IdentityManagerException
     *             if the operation does not succeed
     */
    private IDIFUser convertFromDBUserToDIFUser(Users dbUser) throws IdentityManagerException
    {

        IDIFUser difUser = null;

        if (dbUser != null)
        {
            difUser = new DIFUserImpl();

            if (dbUser.getId() != null)
                difUser.setID(dbUser.getId());

            if (dbUser.getGroups() != null)
            {
                difUser.setProfileID(dbUser.getGroups().getId());
            }

            if (dbUser.getId() != null)
                difUser.setNick(dbUser.getId());

            if (dbUser.getName() != null)
                difUser.setName(dbUser.getName());

            if (dbUser.getPassword() != null)
            {
                try
                {
                    difUser.setPassword(getEncryptator().decrypt(dbUser.getPassword()));
                }
                catch (CryptoException e1)
                {
                    throw new IdentityManagerDBException(e1);
                }
            }

            if (dbUser.getEmail() != null)
                difUser.setEmail(dbUser.getEmail());

            difUser.setDefault(dbUser.isDefaultUser());
            difUser.setEnabled(dbUser.isEnabled());

            /* Parameters */
            Query<UserAttributes> queryUserAttributes = identityService.getUserAttributesDataSet().query();
            try
            {
                queryUserAttributes.addFilter(new Filter(
                        StringUtils.toLowerFirstChar(Users.class.getSimpleName() + "." + Users.Fields.ID.toString()),
                        FilterType.EQUALS, dbUser.getId()));

                List<UserAttributes> userAttributes = queryUserAttributes.asList();

                // Fetch parameters
                CaseInsensitiveHashMap<Object> difAttributes = new CaseInsensitiveHashMap<Object>();

                for (UserAttributes attrib: userAttributes)
                {
                    difAttributes.put(attrib.getAttributeKey(), attrib.getAttributeValue());
                }

                difUser.initializeAttributes(difAttributes);
            }
            catch (DataSetException e)
            {
                throw new IdentityManagerDBException(e);
            }
        }

        return difUser;
    }

    /**
     * Converts a DIF group into an DB group.
     *
     * @param difGroup
     *            the DIF group
     * @return the DB group
     * @throws IdentityManagerException
     *             if a mandatory attribute is not present
     */
    private Groups convertFromDIFGroupToDBGroup(IDIFGroup difGroup) throws IdentityManagerException
    {
        Groups dbGroup = new Groups();

        if (difGroup.getID() != null)
        {
            dbGroup.setId(difGroup.getID());
        }
        else
            throw new IdentityManagerException("Group has no ID!");

        if (difGroup.getName() != null)
        {
            dbGroup.setName(difGroup.getName());
        }
        else
        {
            dbGroup.setName(difGroup.getID());
        }

        if (difGroup.getParentGroupID() != null)
            dbGroup.setGroups(getParentGroupDNFromProfileID(difGroup.getParentGroupID()));

        if (difGroup.getDescription() != null)
            dbGroup.setDescription(difGroup.getDescription());

        dbGroup.setDefaultGroup(difGroup.isDefault());

        return dbGroup;
    }

    /**
     * Converts a DIF user into an DB user.
     *
     * @param difUser
     *            the user to convert
     * @return the converted user
     * @throws IdentityManagerException
     *             if a mandatory attribute is not present
     */
    private Users convertFromDIFUserToDBUser(IDIFUser difUser) throws IdentityManagerException
    {
        Users dbUser = null;

        if (difUser.getID() != null)
        {
            try
            {
                dbUser = identityService.getUsersDataSet().get(difUser.getID());
            }
            catch (DataSetException e)
            {
                throw new IdentityManagerDBException(e);
            }
            if (dbUser == null)
            {
                dbUser = new Users();
                dbUser.setId(difUser.getID());
            }
            dbUser.setName(difUser.getName());
        }
        else
            throw new IdentityManagerDBException("User has no ID!");

        if (difUser.getProfileID() != null)
            dbUser.setGroups(getParentGroupDNFromProfileID(difUser.getProfileID()));

        if (difUser.getNick() != null)
            dbUser.setNick(difUser.getNick());

        if (difUser.getName() != null)
            dbUser.setName(difUser.getName());

        try
        {
            if (((DIFUserImpl) difUser).getPassword() != null)
                dbUser.setPassword(getEncryptator().encrypt((((DIFUserImpl) difUser).getPassword())));
        }
        catch (CryptoException e)
        {
            throw new IdentityManagerDBException(e);
        }
        if (difUser.getEmail() != null)
            dbUser.setEmail(difUser.getEmail());

        dbUser.setDefaultUser(difUser.isDefault());
        dbUser.setEnabled(difUser.isEnabled());

        return dbUser;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countAllGroups()
     */
    public int countAllGroups() throws IdentityManagerException
    {
        try
        {
            return (int) identityService.getGroupsDataSet().query().count();
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countAllGroupsOfUser(java.lang.String)
     */
    public int countAllGroupsOfUser(String userId) throws IdentityManagerException
    {
        try
        {
            int result = 0;
            // User profile
            IDIFUser user = getUser(userId);

            if (user.getProfileID() != null)
                result++;

            Query<UserGroups> query = identityService.getUserGroupsDataSet().query();
            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Users.class.getSimpleName() + "." + Users.Fields.ID),
                            FilterType.EQUALS, userId));

            result += (int) query.count();

            return result;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countAllUsers()
     */
    public int countAllUsers() throws IdentityManagerException
    {
        try
        {
            return (int) identityService.getUsersDataSet().query().count();
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countAllUsers(java.lang.String)
     */
    public int countAllUsers(String groupId) throws IdentityManagerException
    {
        try
        {
            Query<UserGroups> query = identityService.getUserGroupsDataSet().query();
            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Groups.class.getSimpleName() + "." + Groups.Fields.ID),
                            FilterType.EQUALS, groupId));

            return (int) query.count();
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countUsers(java.util.Map)
     */
    public int countUsers(Map<String, String> attributes) throws IdentityManagerException
    {
        try
        {
            Query<UserAttributes> query = getQueryUsersByAllAttributes(attributes, null);
            return (int) query.count();
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.impl.AbstractIdentityManager#gatherManagedAttributes()
     */
    @Override
    public List<String> gatherManagedAttributes()
    {
        List<String> result = new ArrayList<String>();
        try
        {
            Query<UserAttributes> query = identityService.getUserAttributesDataSet().query();
            query.addField(UserAttributes.Fields.ATTRIBUTEKEY.toString());
            query.setDistinct(true);

            for (UserAttributes userAttribute: query.asList())
            {
                result.add(userAttribute.getAttributeKey());
            }
        }
        catch (DataSetException e)
        {
            e.printStackTrace();
        }

        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getAllGroups()
     */
    public Set<IDIFGroup> getAllGroups() throws IdentityManagerException
    {
        try
        {
            Query<Groups> query = identityService.getGroupsDataSet().query();

            Set<IDIFGroup> result = new HashSet<IDIFGroup>();

            for (Groups group: query.asList())
            {
                result.add(this.convertFromDBGroupToDIFGroup(group));
            }

            return result;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getAllUsers()
     */
    public Set<IDIFUser> getAllUsers() throws IdentityManagerException
    {
        try
        {
            Query<Users> query = identityService.getUsersDataSet().query();

            Set<IDIFUser> result = new HashSet<IDIFUser>();

            for (Users user: query.asList())
            {
                result.add(this.convertFromDBUserToDIFUser(user));
            }

            return result;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * Get the active encryptor instance.
     *
     * @return the encryptor instance
     */
    private IEncryptor getEncryptator()
    {
        if (encryptor == null)
        {
            encryptor = new EncryptorBase64Impl();
            encryptor.setSeed(variableToReturn);
        }
        return encryptor;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getExclusionCharaters()
     */
    public String getExclusionCharaters()
    {
        return "";
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroup(java.lang.String)
     */
    public IDIFGroup getGroup(String groupID) throws IdentityManagerException
    {
        if (groupID == null || N_A.equals(groupID))
            return null;
        else
        {
            try
            {
                Groups group = identityService.getGroupsDataSet().get(groupID);
                return this.convertFromDBGroupToDIFGroup(group);
            }
            catch (DataSetException e)
            {
                throw new IdentityManagerDBException(e);
            }
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroupAttributeName()
     */
    public String getGroupAttributeName()
    {
        return null;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroupGroups(java.lang.String)
     */
    public Map<String, IDIFGroup> getGroupGroups(String parentGroupID) throws IdentityManagerException
    {
        try
        {
            Query<Groups> query = identityService.getGroupsDataSet().query();

            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Groups.class.getSimpleName() + "." + Groups.Fields.ID),
                            FilterType.EQUALS, parentGroupID));

            Map<String, IDIFGroup> result = new HashMap<String, IDIFGroup>();

            for (Groups group: query.asList())
            {
                result.put(group.getId(), this.convertFromDBGroupToDIFGroup(group));
                result.putAll(this.getGroupGroups(group.getId()));
            }

            return result;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroups(pt.digitalis.dif.utils.Pagination)
     */
    public Set<IDIFGroup> getGroups(Pagination page) throws IdentityManagerException
    {

        try
        {
            Query<Groups> query = identityService.getGroupsDataSet().query();
            Integer start = page.getStartRow();
            query.filterResults(start, page.getRowsPerPage());

            Set<IDIFGroup> result = new HashSet<IDIFGroup>();

            for (Groups group: query.asList())
                result.add(this.convertFromDBGroupToDIFGroup(group));

            return result;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroupUsers(java.lang.String)
     */
    public Map<String, IDIFUser> getGroupUsers(String groupID) throws IdentityManagerException
    {
        try
        {
            Query<UserGroups> query = identityService.getUserGroupsDataSet().query();
            query.addField(StringUtils.toLowerFirstChar(Users.class.getSimpleName() + "." + Users.Fields.ID));
            query.addJoin(UserGroups.FK().users(), JoinType.NORMAL);
            query.addFilter(new Filter(UserGroups.FK().groups().ID(), FilterType.EQUALS, groupID));

            Map<String, IDIFUser> result = new HashMap<String, IDIFUser>();

            for (UserGroups group: query.asList())
            {
                result.put(group.getUsers().getId(), this.convertFromDBUserToDIFUser(group.getUsers()));
            }

            return result;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getMailAttributeName()
     */
    public String getMailAttributeName()
    {
        return "mail";
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getNameAttributeName()
     */
    public String getNameAttributeName()
    {
        return "name";
    }

    /**
     * Gets the parent group DN from the profile ID.
     *
     * @param profileID
     *            the profile ID
     * @return the parent group DN
     * @throws IdentityManagerException
     *             if the parent group can't be accessed
     */
    final private Groups getParentGroupDNFromProfileID(String profileID) throws IdentityManagerException
    {
        Groups result = null;
        try
        {
            result = identityService.getGroupsDataSet().get(profileID);
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException("Could not fetch parent group for profile ID: " + profileID, e);
        }
        return result;
    }

    /**
     * Return the User Attributes (all attributes match) query distinct by User
     *
     * @param attributes
     * @param page
     *            The Pagination
     * @return the {@link Query} object
     * @throws DataSetException
     */
    private Query<UserAttributes> getQueryUsersByAllAttributes(Map<String, String> attributes, Pagination page)
            throws DataSetException
    {
        Query<UserAttributes> query = identityService.getUserAttributesDataSet().query();
        query.addField(StringUtils.toLowerFirstChar(Users.class.getSimpleName() + "." + Users.Fields.ID));
        query.setDistinct(true);

        if (attributes.containsKey(this.getMailAttributeName()))
        {
            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Users.class.getSimpleName()) + "." + Users.Fields.EMAIL,
                            FilterType.EQUALS, attributes.get(this.getMailAttributeName())));
            attributes.remove(this.getMailAttributeName());
        }

        if (attributes.containsKey(this.getUserLoginAttributeName()))
        {
            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Users.class.getSimpleName()) + "." + Users.Fields.ID,
                            FilterType.EQUALS, attributes.get(this.getUserLoginAttributeName())));
            attributes.remove(this.getUserLoginAttributeName());
        }

        if (attributes.containsKey(this.getNameAttributeName()))
        {
            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Users.class.getSimpleName()) + "." + Users.Fields.NAME,
                            FilterType.EQUALS, attributes.get(this.getNameAttributeName())));
            attributes.remove(this.getNameAttributeName());
        }

        if (attributes.containsKey(this.getUserParentGroupAttributeName()))
        {
            query.addFilter(new Filter(
                    StringUtils.toLowerFirstChar(Users.class.getSimpleName()) + "."
                            + StringUtils.toLowerFirstChar(Groups.class.getSimpleName()) + "." + Groups.Fields.ID,
                    FilterType.EQUALS, attributes.get(this.getUserParentGroupAttributeName())));
            attributes.remove(this.getUserParentGroupAttributeName());
        }

        if (page != null)
        {
            Integer start = page.getStartRow();
            if (start == 0)
            {
                start = null;
            }
            query.filterResults(start, page.getEndRow());
        }

        for (Map.Entry<String, String> entry: attributes.entrySet())
        {
            query.addFilter(
                    new Filter(UserAttributes.Fields.ATTRIBUTEKEY.toString(), FilterType.EQUALS, entry.getKey()));
            query.addFilter(
                    new Filter(UserAttributes.Fields.ATTRIBUTEVALUE.toString(), FilterType.EQUALS, entry.getValue()));

        }

        return query;
    }

    /**
     * Return the User Attributes (any attribute match) query distinct by User
     *
     * @param attributes
     * @param page
     *            The Pagination
     * @return the {@link Query} object
     * @throws DataSetException
     */
    private Query<UserAttributes> getQueryUsersByAnyAttribute(Map<String, String> attributes, Pagination page)
            throws DataSetException
    {
        Map<String, String> attributesToProcess = new HashMap<String, String>();
        attributesToProcess.putAll(attributes);

        Query<UserAttributes> query = identityService.getUserAttributesDataSet().query();
        query.addField(UserAttributes.FK().users().ID());
        query.setDistinct(true);
        query.addJoin(UserAttributes.FK().users(), JoinType.NORMAL);

        List<String> filters = new ArrayList<String>();

        if (attributesToProcess.containsKey(this.getMailAttributeName()))
        {
            filters.add("users1_." + Users.Fields.EMAIL + " = '" + attributesToProcess.get(this.getMailAttributeName())
                    + "'");
            attributesToProcess.remove(this.getMailAttributeName());
        }

        if (attributesToProcess.containsKey(this.getUserLoginAttributeName()))
        {
            filters.add("users1_." + Users.Fields.ID + " = '"
                    + attributesToProcess.get(this.getUserLoginAttributeName()) + "'");
            attributesToProcess.remove(this.getUserLoginAttributeName());
        }

        if (attributesToProcess.containsKey(this.getNameAttributeName()))
        {
            filters.add("users1_." + Users.Fields.NAME + " = '" + attributesToProcess.get(this.getNameAttributeName())
                    + "'");
            attributesToProcess.remove(this.getNameAttributeName());
        }

        if (attributesToProcess.containsKey(this.getUserParentGroupAttributeName()))
        {
            filters.add("profileid = '" + attributesToProcess.get(this.getUserParentGroupAttributeName()) + "'");
            attributesToProcess.remove(this.getUserParentGroupAttributeName());
        }

        if (page != null)
        {
            Integer start = page.getStartRow();
            if (start == 0)
            {
                start = null;
            }
            query.filterResults(start, page.getEndRow());
        }

        for (Map.Entry<String, String> entry: attributesToProcess.entrySet())
        {
            filters.add(
                    "(attribute_key = '" + entry.getKey() + "' and " + "attribute_value = '" + entry.getValue() + "')");
        }

        if (!filters.isEmpty())
            query.addFilter(new Filter(FilterType.SQL, CollectionUtils.listToSeparatedString(filters, " or ")));

        return query;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUser(java.lang.String)
     */
    public IDIFUser getUser(String userID) throws IdentityManagerException
    {
        try
        {
            Users user = identityService.getUsersDataSet().get(userID);
            return this.convertFromDBUserToDIFUser(user);
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserGroups(java.lang.String)
     */
    public Map<String, IDIFGroup> getUserGroups(String userID) throws IdentityManagerException
    {
        return getUserGroupsPagination(userID, null);
    }

    /**
     * Returns the list of group IDs of a given user
     *
     * @param userID
     *            the user
     * @param page
     *            the pagination (optional)
     * @return the list of groups
     * @throws IdentityManagerException
     *             if the group's users can't be found
     */
    private Set<Groups> getUserGroups(String userID, Pagination page) throws IdentityManagerException
    {
        try
        {
            Query<UserGroups> query = identityService.getUserGroupsDataSet().query();
            query.addField(StringUtils.toLowerFirstChar(Groups.class.getSimpleName() + "." + Groups.Fields.ID));
            query.addField(StringUtils.toLowerFirstChar(Groups.class.getSimpleName()) + "."
                    + StringUtils.toLowerFirstChar(Groups.class.getSimpleName() + "." + Groups.Fields.ID));
            query.addField(StringUtils.toLowerFirstChar(Users.class.getSimpleName() + "." + Users.Fields.ID));
            query.addJoin(StringUtils.toLowerFirstChar(Groups.class.getSimpleName()), JoinType.NORMAL);
            query.addJoin(StringUtils.toLowerFirstChar(Groups.class.getSimpleName()) + "."
                    + StringUtils.toLowerFirstChar(Groups.class.getSimpleName()), JoinType.LEFT_OUTER_JOIN);
            query.setDistinct(true);

            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Users.class.getSimpleName() + "." + Users.Fields.ID),
                            FilterType.EQUALS, userID));

            if (page != null)
            {
                Integer start = page.getStartRow();
                if (start == 0)
                {
                    start = null;
                }
                query.filterResults(start, page.getEndRow());
            }

            Set<Groups> result = new HashSet<Groups>();

            List<UserGroups> userGroups = query.asList();
            if (userGroups.size() > 0)
            {
                for (UserGroups userGroup: userGroups)
                {
                    result.add(userGroup.getGroups());
                }
            }

            if (page == null)
            {
                // Add the user profile as a group, only if no pagination is requested
                Users user = identityService.getUsersDataSet().get(userID);
                if (user.getGroups() != null)
                {
                    result.add(user.getGroups());
                }
            }

            return result;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserGroupsIDs(java.lang.String)
     */
    public Set<String> getUserGroupsIDs(String userID) throws IdentityManagerException
    {

        Set<String> result = new HashSet<String>();
        for (Groups group: this.getUserGroups(userID, null))
        {
            result.add(group.getId());
        }
        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserGroupsPagination(java.lang.String,
     *      pt.digitalis.dif.utils.Pagination)
     */
    public Map<String, IDIFGroup> getUserGroupsPagination(String userID, Pagination page)
            throws IdentityManagerException
    {
        Map<String, IDIFGroup> result = new HashMap<String, IDIFGroup>();
        boolean wasActive = identityService.getUserGroupsDAO().getSession().getTransaction().isActive();
        if (!wasActive)
        {
            identityService.getUserGroupsDAO().getSession().beginTransaction();
        }

        try
        {
            for (Groups group: this.getUserGroups(userID, page))
            {
                result.put(group.getId(), convertFromDBGroupToDIFGroup(group));
            }

            if (!wasActive)
            {
                identityService.getUserGroupsDAO().getSession().getTransaction().commit();
            }
        }
        catch (Exception e)
        {
            if (!wasActive)
            {
                identityService.getUserGroupsDAO().getSession().getTransaction().rollback();
            }
        }

        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserIDsInGroup(java.lang.String)
     */
    public Set<String> getUserIDsInGroup(String groupID) throws IdentityManagerException
    {
        try
        {
            Query<UserGroups> query = identityService.getUserGroupsDataSet().query();
            query.addField(StringUtils.toLowerFirstChar(Users.class.getSimpleName() + "." + Users.Fields.ID));
            query.setDistinct(true);

            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Groups.class.getSimpleName() + "." + Groups.Fields.ID),
                            FilterType.EQUALS, groupID));

            Set<String> result = new HashSet<String>();

            for (UserGroups user: query.asList())
            {
                result.add(user.getUsers().getId());
            }

            return result;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserLoginAttributeName()
     */
    public String getUserLoginAttributeName()
    {
        return Users.Fields.ID.toString();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserParentGroupAttributeName()
     */
    public String getUserParentGroupAttributeName()
    {
        return StringUtils.toLowerFirstChar(Groups.class.getSimpleName() + "." + Groups.Fields.ID.toString());
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsers(pt.digitalis.dif.utils.Pagination)
     */
    public Set<IDIFUser> getUsers(Pagination page) throws IdentityManagerException
    {
        try
        {
            Query<Users> query = identityService.getUsersDataSet().query();

            if (page != null)
            {
                Integer first = page.getStartRow();
                Integer last = page.getStartRow() + page.getRowsPerPage() - 1;

                if (first == 0)
                    first = null;

                query.filterResults(first, last);
            }

            Set<IDIFUser> result = new HashSet<IDIFUser>();

            for (Users user: query.asList())
            {
                result.add(this.convertFromDBUserToDIFUser(user));
            }

            return result;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByAnyAttribute(java.util.Map)
     */
    public Set<IDIFUser> getUsersByAnyAttribute(Map<String, String> attributes) throws IdentityManagerException
    {
        try
        {
            Query<UserAttributes> query = getQueryUsersByAnyAttribute(attributes, null);

            Set<IDIFUser> result = new HashSet<IDIFUser>();

            for (UserAttributes user: query.asList())
            {
                result.add(convertFromDBUserToDIFUser(user.getUsers()));
            }

            return result;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByAttribute(java.lang.String,
     *      java.lang.String)
     */
    public Set<IDIFUser> getUsersByAttribute(String attribute, String value) throws IdentityManagerException
    {
        try
        {
            Query<UserAttributes> query = identityService.getUserAttributesDataSet().query();

            query.addField(UserAttributes.FK().users().ID());
            query.addField(UserAttributes.FK().users().NAME());
            query.addField(UserAttributes.FK().users().NICK());
            query.addField(UserAttributes.FK().users().EMAIL());
            query.setDistinct(true);

            if (attribute.equalsIgnoreCase(Users.Fields.ID))
                query.like(UserAttributes.FK().users().ID(), value);
            else if (attribute.equalsIgnoreCase(Users.Fields.NAME))
                query.like(UserAttributes.FK().users().NAME(), value);
            else if (attribute.equalsIgnoreCase(Users.Fields.NICK))
                query.like(UserAttributes.FK().users().NICK(), value);
            else if (attribute.equalsIgnoreCase(Users.Fields.EMAIL))
                query.like(UserAttributes.FK().users().EMAIL(), value);
            else
            {
                query.equals(UserAttributes.Fields.ATTRIBUTEKEY.toString(), attribute);
                query.like(UserAttributes.Fields.ATTRIBUTEVALUE.toString(), value);
            }

            Set<IDIFUser> result = new HashSet<IDIFUser>();

            for (UserAttributes userAttribute: query.asList())
            {
                result.add(this.convertFromDBUserToDIFUser(userAttribute.getUsers()));
            }

            return result;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * Gets the users by attributes.
     *
     * @param attributes
     *            the attributes
     * @return the users by attributes
     * @throws IdentityManagerException
     *             the identity manager exception
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByAttributes(java.util.Map)
     */
    public Set<IDIFUser> getUsersByAttributes(Map<String, String> attributes) throws IdentityManagerException
    {
        return this.getUsersByAttributes(attributes, null);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByAttributes(java.util.Map,
     *      pt.digitalis.dif.utils.Pagination)
     */
    public Set<IDIFUser> getUsersByAttributes(Map<String, String> attributes, Pagination page)
            throws IdentityManagerException
    {
        try
        {
            Query<UserAttributes> query = getQueryUsersByAllAttributes(attributes, page);

            Set<IDIFUser> result = new HashSet<IDIFUser>();

            for (UserAttributes user: query.asList())
            {
                result.add(convertFromDBUserToDIFUser(user.getUsers()));
            }

            return result;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByEmail(java.lang.String)
     */
    public Set<IDIFUser> getUsersByEmail(String value) throws IdentityManagerException
    {
        try
        {
            Query<Users> query = identityService.getUsersDataSet().query();
            query.addFilter(new Filter(Users.Fields.EMAIL.toString(), FilterType.EQUALS, value));

            Set<IDIFUser> result = new HashSet<IDIFUser>();

            for (Users user: query.asList())
            {
                result.add(this.convertFromDBUserToDIFUser(user));
            }

            return result;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#groupExists(java.lang.String)
     */
    public boolean groupExists(String groupID) throws IdentityManagerException
    {
        try
        {
            Groups group = identityService.getGroupsDataSet().get(groupID);
            return group != null;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#isIdentityValid(java.lang.String,
     *      java.lang.String)
     */
    public boolean isIdentityValid(String userID, String suppliedPassword) throws IdentityManagerException
    {
        try
        {
            Users user = identityService.getUsersDataSet().get(userID);
            return user != null && suppliedPassword != null
                    && suppliedPassword.equals(getEncryptator().decrypt(user.getPassword()));
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
        catch (CryptoException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#isReadOnly()
     */
    public boolean isReadOnly()
    {
        return false;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#isUserInGroup(java.lang.String,
     *      java.lang.String)
     */
    public boolean isUserInGroup(String userID, String groupID) throws IdentityManagerException
    {
        try
        {
            Query<UserGroups> query = identityService.getUserGroupsDataSet().query();
            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Users.class.getSimpleName() + "." + Users.Fields.ID),
                            FilterType.EQUALS, userID));

            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Groups.class.getSimpleName() + "." + Groups.Fields.ID),
                            FilterType.EQUALS, groupID));

            return query.count() > 0;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManagerPrivate#persistUserAttribute(java.lang.String,
     *      java.lang.String, java.lang.Object)
     */
    public void persistUserAttribute(String userID, String attributeID, Object attributeValue)
            throws IdentityManagerException
    {
        try
        {
            Query<UserAttributes> query = identityService.getUserAttributesDataSet().query();
            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Users.class.getSimpleName() + "." + Users.Fields.ID),
                            FilterType.EQUALS, userID));

            query.addFilter(new Filter(UserAttributes.Fields.ATTRIBUTEKEY.toString(), FilterType.EQUALS, attributeID));

            UserAttributes userAttribute = query.singleValue();

            if (userAttribute != null)
            {
                userAttribute.setAttributeValue(attributeValue.toString());
                identityService.getUserAttributesDataSet().update(userAttribute);
            }
            else
            {
                userAttribute = new UserAttributes();
                Users user = identityService.getUsersDataSet().get(userID);
                if (user == null)
                {
                    throw new IdentityManagerException("User " + userID + " does not exists on the IdentityManager!! ");
                }

                userAttribute.setUsers(user);
                userAttribute.setAttributeKey(attributeID);
                userAttribute.setAttributeValue(attributeValue.toString());
                identityService.getUserAttributesDataSet().insert(userAttribute);

            }
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManagerPrivate#persistUserAttributes(java.lang.String,
     *      java.util.Map)
     */
    public void persistUserAttributes(String userID, Map<String, Object> attributes) throws IdentityManagerException
    {
        try
        {
            Users dbUser = identityService.getUsersDataSet().get(userID);
            if (dbUser == null)
            {
                throw new IdentityManagerDBException("Could not update attributes for user with ID: " + userID + "!");
            }
            Query<UserAttributes> query = identityService.getUserAttributesDataSet().query();
            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Users.class.getSimpleName() + "." + Users.Fields.ID),
                            FilterType.EQUALS, userID));

            for (UserAttributes userAttribute: query.asList())
            {
                String value = "";

                if (attributes.containsKey(userAttribute.getAttributeKey()))
                {
                    value = attributes.get(userAttribute.getAttributeKey()).toString();
                }
                userAttribute.setAttributeValue(value);
                identityService.getUserAttributesDataSet().update(userAttribute);
                attributes.remove(userAttribute.getAttributeKey());
            }

            /* If there's still attributes inside, it means that a insert have to be made */
            if (attributes.size() > 0)
            {
                for (String attribKey: attributes.keySet())
                {
                    UserAttributes newAttribute = new UserAttributes();
                    newAttribute.setAttributeKey(attribKey);
                    newAttribute.setAttributeValue(attributes.get(attribKey).toString());

                    newAttribute.setUsers(dbUser);
                    newAttribute = identityService.getUserAttributesDataSet().insert(newAttribute);
                }
            }
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }

    }

    /**
     * Process a DIF user into an DB user attributes.
     *
     * @param difUser
     *            the DIF user to process
     * @param dbUser
     *            the DB user
     * @throws IdentityManagerException
     *             if a mandatory attribute is not present
     */
    private void processFromDIFUserToDBUserAttributes(IDIFUser difUser, Users dbUser) throws IdentityManagerException
    {
        try
        {
            /* Parameters */
            // Fetch parameters
            CaseInsensitiveHashMap<Object> attributes = difUser.getAttributes();

            Query<UserAttributes> queryUserAttributes = identityService.getUserAttributesDataSet().query();
            queryUserAttributes.addFilter(new Filter(
                    StringUtils.toLowerFirstChar(Users.class.getSimpleName() + "." + Users.Fields.ID.toString()),
                    FilterType.EQUALS, dbUser.getId()));

            List<UserAttributes> dbAttributes = null;

            // Check if there are parameters to process
            if (attributes != null && attributes.size() > 0)
            {
                // Load the User Attributes
                dbAttributes = queryUserAttributes.asList();

                for (String attributeName: attributes.keySet())
                {
                    UserAttributes existentAttribute = null;
                    for (UserAttributes attribute: dbAttributes)
                    {
                        if (attribute.getAttributeKey().equalsIgnoreCase(attributeName))
                        {
                            existentAttribute = attribute;
                            break;
                        }
                    }

                    if (existentAttribute != null)
                    {
                        existentAttribute.setAttributeValue(attributes.get(attributeName).toString());
                        identityService.getUserAttributesDataSet().update(existentAttribute);
                    }
                    else
                    {
                        UserAttributes newAttribute = new UserAttributes();
                        newAttribute.setAttributeKey(attributeName);
                        newAttribute.setAttributeValue(attributes.get(attributeName).toString());
                        newAttribute.setUsers(dbUser);
                        newAttribute = identityService.getUserAttributesDataSet().insert(newAttribute);
                    }
                }
            }

            if (difUser.getAttributesToRemove() != null && difUser.getAttributesToRemove().size() > 0)
            {
                if (dbAttributes == null)
                {
                    dbAttributes = queryUserAttributes.asList();
                }
                for (String attributeName: difUser.getAttributesToRemove())
                {
                    for (UserAttributes attribute: dbAttributes)
                    {
                        if (attribute.getAttributeKey().equals(attributeName))
                        {
                            identityService.getUserAttributesDataSet().delete(attribute.getId().toString());
                            break;
                        }
                    }
                }
            }
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#removeGroup(java.lang.String)
     */
    public void removeGroup(String groupID) throws IdentityManagerException
    {
        try
        {
            identityService.getGroupsDataSet().delete(groupID);
        }
        catch (Exception e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#removeUser(java.lang.String)
     */
    public void removeUser(String userID) throws IdentityManagerException
    {
        try
        {
            identityService.getUsersDataSet().delete(userID);
        }
        catch (Exception e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#removeUserFromGroup(java.lang.String,
     *      java.lang.String)
     */
    public void removeUserFromGroup(String userID, String groupID) throws IdentityManagerException
    {
        try
        {

            Query<UserGroups> query = identityService.getUserGroupsDataSet().query();
            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Users.class.getSimpleName() + "." + Users.Fields.ID),
                            FilterType.EQUALS, userID));
            query.addFilter(
                    new Filter(StringUtils.toLowerFirstChar(Groups.class.getSimpleName() + "." + Groups.Fields.ID),
                            FilterType.EQUALS, groupID));

            UserGroups userGroups = query.singleValue();
            if (userGroups != null)
            {
                identityService.getUserGroupsDataSet().delete(userGroups.getId().toString());
            }
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#resetIdentityManager()
     */
    public void resetIdentityManager()
    {}

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#updateGroup(pt.digitalis.dif.controller.security.objects.IDIFGroup)
     */
    public void updateGroup(IDIFGroup existingGroup) throws IdentityManagerException
    {
        Groups dbGroup = convertFromDIFGroupToDBGroup(existingGroup);

        try
        {
            identityService.getGroupsDataSet().update(dbGroup);
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#updateUser(pt.digitalis.dif.controller.security.objects.IDIFUser,
     *      java.lang.String)
     */
    public void updateUser(IDIFUser existingUser, String userID) throws IdentityManagerException
    {
        Users dbUser = convertFromDIFUserToDBUser(existingUser);

        try
        {
            dbUser = identityService.getUsersDataSet().update(dbUser);
            processFromDIFUserToDBUserAttributes(existingUser, dbUser);
        }
        catch (Exception e)
        {
            throw new IdentityManagerDBException(e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#updateUserAttribute(java.lang.String,
     *      java.lang.String, java.lang.Object)
     */
    public void updateUserAttribute(String userID, String attributeID, Object attributeValue)
            throws IdentityManagerException
    {
        this.persistUserAttribute(userID, attributeID, attributeValue);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#updateUserAttributes(java.lang.String,
     *      java.util.Map)
     */
    public void updateUserAttributes(String userID, Map<String, Object> attributes) throws IdentityManagerException
    {
        this.persistUserAttributes(userID, attributes);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#userExists(java.lang.String)
     */
    public boolean userExists(String userID) throws IdentityManagerException
    {
        try
        {
            Users user = identityService.getUsersDataSet().get(userID);
            return user != null;
        }
        catch (DataSetException e)
        {
            throw new IdentityManagerDBException(e);
        }
    }
}
