/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.identity.ldap;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.naming.NamingException;

import pt.digitalis.dif.controller.security.managers.IAuthorizationManager;
import pt.digitalis.dif.controller.security.managers.impl.AbstractIdentityManager;
import pt.digitalis.dif.controller.security.objects.DIFGroupImpl;
import pt.digitalis.dif.controller.security.objects.DIFUserImpl;
import pt.digitalis.dif.controller.security.objects.IDIFGroup;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.utils.Pagination;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.ConfigurationsPreferencesImpl;
import pt.digitalis.utils.config.IConfigurations;
import pt.digitalis.utils.ldap.ILDAPUtils;
import pt.digitalis.utils.ldap.LDAPConfigurations;
import pt.digitalis.utils.ldap.LDAPGroup;
import pt.digitalis.utils.ldap.LDAPUser;
import pt.digitalis.utils.ldap.exception.LDAPOperationException;

/**
 * Identity manager implementation based on LDAP.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Fbio Souto <a href="mailto:fsouto@digitalis.pt">fsouto@digitalis.pt</a><br/>
 * @created Mar 26, 2008
 */
public class IdentityManagerLDAPImpl extends AbstractIdentityManager {

    /** The authorization manager. */
    static private IAuthorizationManager authorizationManager = DIFIoCRegistry.getRegistry()
            .getImplementation(IAuthorizationManager.class);

    /** The key used to publish the LDAP Domain Name into the users details */
    private static String DOMAIN_NAME_USER_ATTRIBUTE = "domainName";

    /**
     * A cache for mapping between LDAP DN group names and DIF group names: Goal is simply reduce LDAP searches for
     * performance gain
     */
    static private Map<String, String> ldapGroupDNMappingCache = new HashMap<String, String>();

    /** The Constant NAME. */
    public static final String NAME = "LDAP";

    /** The configurations object. */
    private IdentityManagerLDAPConfigurations configurations = null;

    /** The ILDAPUtils current implementation (obtained from the IoC) for method execution. */
    private ILDAPUtils ldapUtils = null;

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#addGroup(pt.digitalis.dif.controller.security.objects.IDIFGroup)
     */
    public void addGroup(IDIFGroup newGroup) throws IdentityManagerException
    {
        try
        {
            LDAPGroup ldapGroup = convertFromDIFGroupToLDAPGroup(newGroup);

            getLDAPUtils().addGroup(ldapGroup);

            if (ldapGroup != null)
                // Update the cache;
                ldapGroupDNMappingCache.put(ldapGroup.getDistinguishedName(), newGroup.getID());

        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Couldn't add group " + newGroup.getID() + " to LDAP server!",
                    ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#addUser(pt.digitalis.dif.controller.security.objects.IDIFUser)
     */
    public void addUser(IDIFUser newUser) throws IdentityManagerException
    {
        try
        {
            // If the user password is not defined, by default the password is "password"
            if (((DIFUserImpl) newUser).getPassword() == null)
            {
                newUser.setPassword("password");
            }
            getLDAPUtils().addUser(convertFromDIFUserToLDAPUser(newUser));
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException(
                    "Couldn't add user to LDAP server! " + ldapOperationException.getMessage(), ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#addUserToGroup(java.lang.String,
     *      java.lang.String)
     */
    public void addUserToGroup(String userID, String groupID) throws IdentityManagerException
    {
        try
        {
            getLDAPUtils().addUserToGroup(groupID, userID);
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Coud not add user " + userID + " to group " + groupID + "!",
                    ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#changePassword(java.lang.String,
     *      java.lang.String)
     */
    public void changePassword(String userID, String newPassword) throws IdentityManagerException
    {
        try
        {
            getLDAPUtils().changePassword(userID, newPassword);
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not change password for user " + userID + "!",
                    ldapOperationException);
        }
    }

    /**
     * Converts a DIF group into an LDAP group.
     * 
     * @param difGroup
     *            the DIF group
     * @return the LDAP group
     * @throws IdentityManagerException
     *             if a mandatory attribute is not present
     */
    private LDAPGroup convertFromDIFGroupToLDAPGroup(IDIFGroup difGroup) throws IdentityManagerException
    {
        LDAPGroup ldapGroup = new LDAPGroup();

        if (difGroup.getID() != null)
        {
            ldapGroup.setCommonName(difGroup.getID());
        }
        else
            throw new IdentityManagerLDAPException("Group has no ID! The ID is needed to serve as LDAP cn...");

        if (difGroup.getName() != null)
            ldapGroup.setName(difGroup.getName());

        if (difGroup.getParentGroupID() != null)
            ldapGroup.setParentGroupDN(getParentGroupDNFromProfileID(difGroup.getParentGroupID()));

        if (difGroup.getDescription() != null)
            ldapGroup.setDescription(difGroup.getDescription());

        return ldapGroup;
    }

    /**
     * Converts a DIF user into an LDAP user.
     * 
     * @param difUser
     *            the user to convert
     * @return the converted user
     * @throws IdentityManagerException
     *             if a mandatory attribute is not present
     */
    private LDAPUser convertFromDIFUserToLDAPUser(IDIFUser difUser) throws IdentityManagerException
    {
        LDAPUser ldapUser = new LDAPUser();

        // DIFUser#ID <-> LDAPUser#loginName
        if (difUser.getID() != null)
        {
            ldapUser.setLoginName(difUser.getID());
            ldapUser.setName(difUser.getName());
            ldapUser.setUserName(difUser.getID());
        }
        else
            throw new IdentityManagerLDAPException("User has no ID! The ID is needed to serve as LDAP login...");

        if (difUser.getProfileID() != null)
            ldapUser.setParentGroupDN(getParentGroupDNFromProfileID(difUser.getProfileID()));

        if (difUser.getNick() != null)
            ldapUser.setDisplayName(difUser.getNick());

        if (difUser.getName() != null)
            ldapUser.setGivenName(difUser.getName());

        if (((DIFUserImpl) difUser).getPassword() != null)
            ldapUser.setPassword(((DIFUserImpl) difUser).getPassword());

        if (difUser.getEmail() != null)
            ldapUser.setEmail(difUser.getEmail());

        ldapUser.setDescription("DiF2 user");

        /* Parameters */
        // Fetch parameters
        Map<String, Object> attributes = difUser.getAttributes();

        // Check if there are parameters to process
        if (attributes != null && attributes.size() > 0)
        {
            for (String attributeName: attributes.keySet())
            {
                // ...add it to the LDAPUser parameters (K=LDAP attribute name)
                ldapUser.setParameter(attributeName, attributes.get(attributeName).toString());
            }
        }

        for (String attributeName: difUser.getAttributesToRemove())
        {
            ldapUser.removeParameter(attributeName);
        }

        return ldapUser;
    }

    /**
     * Converts an LDAP group into a DIF group.
     * 
     * @param ldapGroup
     *            the LDAP group
     * @return the DIF group
     * @throws IdentityManagerException
     *             if the operation does not succeed
     */
    private IDIFGroup convertFromLDAPGroupToDIFGroup(LDAPGroup ldapGroup) throws IdentityManagerException
    {
        IDIFGroup difGroup = null;

        if (ldapGroup != null)
        {
            difGroup = new DIFGroupImpl();

            if (ldapGroup.getCommonName() != null)
                difGroup.setID(ldapGroup.getCommonName());

            if (ldapGroup.getName() != null)
                difGroup.setName(ldapGroup.getName());

            difGroup.setParentGroupID(N_A);
            try
            {
                if (ldapGroup.getParentGroupDN() != null && !ldapGroup.getParentGroupDN().equals(N_A))
                    difGroup.setParentGroupID(
                            getLDAPUtils().findGroupByDistinguishedName(ldapGroup.getParentGroupDN()).getCommonName());
            }
            catch (LDAPOperationException ldapOperationException)
            {
                // Untestable exception: impossible to force an illegal parent group DN on the LDAP group (group is
                // returned by the server and is not possible, at this time, to force the illegal parent group DN
                // through Java).
                DIFLogger.getLogger()
                        .warn("Could not access parent group \"" + ldapGroup.getParentGroupDN() + "\", of the group \""
                                + ldapGroup.getCommonName() + "\"! Exception: " + ldapOperationException.getMessage());
            }

            if (ldapGroup.getDescription() != null)
                difGroup.setDescription(ldapGroup.getDescription());

        }

        return difGroup;
    }

    /**
     * Converts an LDAP user into a DIF user.
     * 
     * @param ldapUser
     *            the LDAP user
     * @return the DIF user
     * @throws IdentityManagerException
     *             if the operation does not succeed
     */
    private IDIFUser convertFromLDAPUserToDIFUser(LDAPUser ldapUser) throws IdentityManagerException
    {

        IDIFUser difUser = null;

        if (ldapUser != null)
        {
            difUser = new DIFUserImpl();

            if (ldapUser.getLoginName() != null)
                difUser.setID(ldapUser.getLoginName());

            try
            {
                if (ldapUser.getParentGroupDN() != null && !N_A.equals(ldapUser.getParentGroupDN()))
                {
                    if (ldapGroupDNMappingCache.containsKey(ldapUser.getParentGroupDN()))
                    {
                        // Cache group, set ID

                        difUser.setProfileID(ldapGroupDNMappingCache.get(ldapUser.getParentGroupDN()));

                    }
                    else
                    {
                        // Not in cache. Search it in LDAP...

                        String profileID = convertFromLDAPGroupToDIFGroup(
                                getLDAPUtils().findGroupByDistinguishedName(ldapUser.getParentGroupDN())).getID();
                        difUser.setProfileID(profileID);

                        ldapGroupDNMappingCache.put(ldapUser.getParentGroupDN(), profileID);
                    }
                }
            }
            catch (LDAPOperationException ldapOperationException)
            {
                // Untestable exception: impossible to force an illegal parent group DN on the LDAP user (user is
                // returned by the server and is not possible, at this time, to force the illegal parent group DN
                // through Java).

                // TODO: Viegas: define what to do when the profile can't be set
            }

            if (ldapUser.getLoginName() != null)
                difUser.setNick(ldapUser.getLoginName());

            if (ldapUser.getGivenName() != null && !ldapUser.getGivenName().equals(ldapUtils.getNonAvailableValue()))
                difUser.setName(ldapUser.getGivenName());
            if (ldapUser.getDisplayName() != null && !ldapUser.getDisplayName().equals(ldapUtils.getNonAvailableValue())
                    && (StringUtils.isBlank(difUser.getName())
                            || difUser.getName().length() < ldapUser.getDisplayName().length()))
                difUser.setName(ldapUser.getDisplayName());

            // LDAP server does not return passwords.
            // If-branch is here since this is an helper method that might be used on other operations in the future
            if (ldapUser.getPassword() != null)
                difUser.setPassword(ldapUser.getPassword());

            if (ldapUser.getEmail() != null)
                difUser.setEmail(ldapUser.getEmail());

            /* Parameters */
            Map<String, String> parameters = ldapUser.getParameters();
            Map<String, Object> difAttributes = new HashMap<String, Object>();

            difAttributes.putAll(parameters);

            /* The DOMAIN_NAME_USER_ATTRIBUTE is always filled */
            difAttributes.put(DOMAIN_NAME_USER_ATTRIBUTE, ldapUser.getDistinguishedName());

            difUser.initializeAttributes(difAttributes);
        }

        return difUser;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countAllGroups()
     */
    public int countAllGroups() throws IdentityManagerException
    {
        try
        {
            return this.getLDAPUtils().countAllGroups(true);
        }
        catch (LDAPOperationException op)
        {
            throw new IdentityManagerLDAPException("Could not count all groups: ", op);
        }
        catch (NamingException e)
        {
            throw new IdentityManagerLDAPException("Could not count all groups: ", e);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.impl.AbstractIdentityManager#countAllGroupsOfUser(java.lang.String)
     */
    public int countAllGroupsOfUser(String userId) throws IdentityManagerException
    {
        try
        {
            int result = 0;
            // User profile
            IDIFUser user = getUser(userId);

            if (user.getProfileID() != null)
                result++;

            result += getLDAPUtils().countAllGroupsOfUser(userId);

            return result;
        }
        catch (NamingException ne)
        {
            throw new IdentityManagerLDAPException("Could not retrieve information for user " + userId + "!", ne);
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not retrieve information for user " + userId + "!",
                    ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countAllUsers()
     */
    public int countAllUsers() throws IdentityManagerException
    {
        try
        {
            return this.getLDAPUtils().countAllUsers();
        }
        catch (LDAPOperationException op)
        {
            throw new IdentityManagerLDAPException("Could not count all users: ", op);
        }
        catch (NamingException ne)
        {
            throw new IdentityManagerLDAPException("Could not count all users: ", ne);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countAllUsers(java.lang.String)
     */
    public int countAllUsers(String groupID) throws IdentityManagerException
    {
        try
        {
            return getLDAPUtils().countAllUsers(groupID);
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException(
                    "Couldn't add user to LDAP server! " + ldapOperationException.getMessage(), ldapOperationException);
        }
    }

    /**
     * @throws IdentityManagerLDAPException
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#countUsers(java.util.Map)
     */
    public int countUsers(Map<String, String> attributes) throws IdentityManagerException
    {
        try
        {
            return getLDAPUtils().countUsers(attributes);
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException(
                    "Could not retrieve counting for users with attributes " + attributes + "!",
                    ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.impl.AbstractIdentityManager#gatherManagedAttributes()
     */
    @Override
    public List<String> gatherManagedAttributes()
    {
        LDAPConfigurations ldapconfig = DIFIoCRegistry.getRegistry().getImplementation(IConfigurations.class)
                .readConfiguration(LDAPConfigurations.class);
        List<String> result = new ArrayList<String>(ldapconfig.getAttributesMapping().keySet());
        return result;
    }

    /**
     * @throws IdentityManagerLDAPException
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getAllGroups()
     */
    public Set<IDIFGroup> getAllGroups() throws IdentityManagerException
    {
        Set<LDAPGroup> ldapGroups = null;
        Set<IDIFGroup> result = null;

        try
        {
            ldapGroups = getLDAPUtils().findAllGroups();
            result = new HashSet<IDIFGroup>();

            // convert from LDAPGroup to IDIFGroup, for returning
            for (LDAPGroup g: ldapGroups)
            {
                result.add(this.convertFromLDAPGroupToDIFGroup(g));

            }

            return result;
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not get all groups from LDAP: ", ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getAllUsers()
     */
    public Set<IDIFUser> getAllUsers() throws IdentityManagerException
    {
        Set<LDAPUser> ldapUsers = null;
        Set<IDIFUser> result = null;

        try
        {
            ldapUsers = getLDAPUtils().findAllUsers();
            result = new HashSet<IDIFUser>();

            // convert from LDAPUser to IDIFUser, for returning
            for (LDAPUser u: ldapUsers)
                result.add(this.convertFromLDAPUserToDIFUser(u));
            return result;
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not get all users from LDAP: ", ldapOperationException);
        }

    }

    /**
     * Configurations accessor.
     * 
     * @return the configurations
     */
    protected IdentityManagerLDAPConfigurations getConfigurations()
    {
        if (this.configurations == null)
            configurations = new ConfigurationsPreferencesImpl()
                    .readConfiguration(IdentityManagerLDAPConfigurations.class);

        return this.configurations;

    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getExclusionCharaters()
     */
    public String getExclusionCharaters()
    {
        return this.getLDAPUtils().getConfigurations().getExclusionCharaters();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroup(java.lang.String)
     */
    public IDIFGroup getGroup(String groupID) throws IdentityManagerException
    {

        if (groupID == null || N_A.equals(groupID))
            return null;

        try
        {
            LDAPGroup ldapGroup = getLDAPUtils().findGroupByCommonName(groupID);
            IDIFGroup difGroup = convertFromLDAPGroupToDIFGroup(ldapGroup);

            if (difGroup != null)
                // Update the cache;
                ldapGroupDNMappingCache.put(ldapGroup.getDistinguishedName(), difGroup.getID());

            return difGroup;

        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not fetch data for group " + groupID + "!",
                    ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroupAttributeName()
     */
    public String getGroupAttributeName()
    {
        return getLDAPUtils().getGroupAttributeName();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroupGroups(java.lang.String)
     */
    public Map<String, IDIFGroup> getGroupGroups(String parentGroupID) throws IdentityManagerException
    {

        Set<LDAPGroup> childGroups = null;

        try
        {
            childGroups = getLDAPUtils().getChildGroupsByDN(getParentGroupDNFromProfileID(parentGroupID));
        }
        catch (LDAPOperationException ldapOperationException)
        {
            // Untestable exception: getChildGroupsByDN never throws the LDAPOperationException since passing an invalid
            // parentGroupID will yield an IdentityManagerException
            throw new IdentityManagerLDAPException(
                    "Could not access LDAP server to find child groups of group with ID: " + parentGroupID + "!",
                    ldapOperationException);
        }

        Map<String, IDIFGroup> childGroupsMap = new HashMap<String, IDIFGroup>(childGroups.size());

        for (LDAPGroup group: childGroups)
        {
            IDIFGroup difGroup = convertFromLDAPGroupToDIFGroup(group);

            childGroupsMap.put(difGroup.getID(), difGroup);
        }

        return childGroupsMap;
    }

    /**
     * @throws IdentityManagerException
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroups(Pagination)
     */
    public Set<IDIFGroup> getGroups(Pagination page) throws IdentityManagerException
    {
        Set<LDAPGroup> ldapGroups = null;
        Set<IDIFGroup> result = null;

        try
        {
            ldapGroups = getLDAPUtils().findGroups(page.getRowsPerPage(), page.getPage());
            result = new HashSet<IDIFGroup>();

            // convert from LDAPGroup to IDIFGroup, for returning
            for (LDAPGroup g: ldapGroups)
                result.add(this.convertFromLDAPGroupToDIFGroup(g));
            return result;
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not retrieve subset of groups (" + page.getRowsPerPage()
                    + " rows per page, " + page.getPage() + " page)", ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getGroupUsers(java.lang.String)
     */
    public Map<String, IDIFUser> getGroupUsers(String groupID) throws IdentityManagerException
    {
        Map<String, IDIFUser> difUsers = new HashMap<String, IDIFUser>();

        Map<String, LDAPUser> ldapUsers = null;
        try
        {
            ldapUsers = getLDAPUtils().findUsersInGroup(groupID);
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException(ldapOperationException);
        }

        for (String userID: ldapUsers.keySet())
            difUsers.put(userID, convertFromLDAPUserToDIFUser(ldapUsers.get(userID)));

        return difUsers;
    }

    /**
     * Returns the active LDAP Utils implementation. Uses lazy-loading to instantiate the configured implementation.
     * 
     * @return the active LDAP Utils implementation
     */
    public ILDAPUtils getLDAPUtils()
    {
        if (this.ldapUtils == null)
        {
            if (LDAPUtilsImplementation.AD.equals(getConfigurations().getLdapUtilsImplementation()))
                this.ldapUtils = DIFIoCRegistry.getRegistry().getImplementation(ILDAPUtils.class,
                        LDAPUtilsImplementation.AD.getIdTag());
            else if (LDAPUtilsImplementation.OPEN_LDAP.equals(getConfigurations().getLdapUtilsImplementation()))
                this.ldapUtils = DIFIoCRegistry.getRegistry().getImplementation(ILDAPUtils.class,
                        LDAPUtilsImplementation.OPEN_LDAP.getIdTag());
            else if (LDAPUtilsImplementation.OID.equals(getConfigurations().getLdapUtilsImplementation()))
                throw new RuntimeException("Oracle OID LDAP Utils implementation is not supported on this version! ");
        }

        ldapUtils.setLogger(DIFLogger.getLogger());
        return ldapUtils;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getMailAttributeName()
     */
    public String getMailAttributeName()
    {
        return getLDAPUtils().getMailAttributeName();

    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getNameAttributeName()
     */
    public String getNameAttributeName()
    {
        return getLDAPUtils().getNameAttributeName();
    }

    /**
     * Gets the parent group DN from the profile ID.
     * 
     * @param profileID
     *            the profile ID
     * @return the parent group DN
     * @throws IdentityManagerException
     *             if the parent group can't be accessed
     */
    final private String getParentGroupDNFromProfileID(String profileID) throws IdentityManagerException
    {
        String result = null;
        try
        {
            LDAPGroup group = getLDAPUtils().findGroupByCommonName(profileID);
            if (group != null)
            {
                result = group.getDistinguishedName();
            }
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not fetch parent group for profile ID: " + profileID,
                    ldapOperationException);
        }
        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUser(java.lang.String)
     */
    public IDIFUser getUser(String userID) throws IdentityManagerException
    {
        try
        {
            return convertFromLDAPUserToDIFUser(getLDAPUtils().findUserByLogin(userID));
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not fetch data for user " + userID + "!",
                    ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserGroups(java.lang.String)
     */
    public Map<String, IDIFGroup> getUserGroups(String userID) throws IdentityManagerException
    {
        Map<String, IDIFGroup> groupsOfUser = new HashMap<String, IDIFGroup>();

        IDIFGroup difGroup = null;

        try
        {
            // User profile
            IDIFUser user = getUser(userID);
            /* If the user doesn't exists, should return a list of empty groups */
            if (user != null)
            {
                IDIFGroup profile = getGroup(user.getProfileID());

                if (profile != null)
                    groupsOfUser.put(profile.getID(), profile);

                // Groups of user
                for (LDAPGroup ldapGroup: getLDAPUtils().findGroupsOfUser(userID))
                {
                    difGroup = convertFromLDAPGroupToDIFGroup(ldapGroup);

                    if (difGroup != null)
                    {
                        groupsOfUser.put(difGroup.getID(), difGroup);

                        // Update the cache;
                        ldapGroupDNMappingCache.put(ldapGroup.getDistinguishedName(), difGroup.getID());
                    }
                }
            }
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not get groups of user " + userID, ldapOperationException);
        }

        return groupsOfUser;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserGroupsIDs(java.lang.String)
     */
    public Set<String> getUserGroupsIDs(String userID) throws IdentityManagerException
    {
        return getUserGroups(userID).keySet();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserGroupsPagination(java.lang.String,
     *      pt.digitalis.dif.utils.Pagination)
     */
    public Map<String, IDIFGroup> getUserGroupsPagination(String userID, Pagination page)
            throws IdentityManagerException
    {
        Map<String, IDIFGroup> groupsOfUser = new HashMap<String, IDIFGroup>();

        IDIFGroup difGroup = null;

        try
        {

            int rowsPerPage = page.getRowsPerPage();

            // Groups of user
            for (LDAPGroup ldapGroup: getLDAPUtils().findGroupsOfUserPagination(userID, rowsPerPage, page.getPage()))
            {
                difGroup = convertFromLDAPGroupToDIFGroup(ldapGroup);

                if (difGroup != null)
                {
                    groupsOfUser.put(difGroup.getID(), difGroup);

                    // Update the cache;
                    ldapGroupDNMappingCache.put(ldapGroup.getDistinguishedName(), difGroup.getID());
                }
            }
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not get groups of user " + userID, ldapOperationException);
        }

        return groupsOfUser;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserIDsInGroup(java.lang.String)
     */
    public Set<String> getUserIDsInGroup(String groupID) throws IdentityManagerException
    {
        return getGroupUsers(groupID).keySet();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserLoginAttributeName()
     */
    public String getUserLoginAttributeName()
    {
        return getLDAPUtils().getUserLoginAttributeName();

    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUserParentGroupAttributeName()
     */
    public String getUserParentGroupAttributeName()
    {
        return getLDAPUtils().getUserParentGroupAttributeName();
    }

    /**
     * @throws IdentityManagerException
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsers(Pagination)
     */
    public Set<IDIFUser> getUsers(Pagination page) throws IdentityManagerException
    {
        Set<LDAPUser> ldapUsers = null;
        Set<IDIFUser> result = null;

        try
        {
            // HISTORY: FSouto wrote this: add 1 to the page to return, since findUsers starts counting from 1 instead
            // of 0
            // Viegas: I found that this was not correct and 1 is received and thus this code skiped the first page!
            ldapUsers = getLDAPUtils().findUsers(page.getRowsPerPage(), page.getPage());
            result = new HashSet<IDIFUser>();

            // convert from LDAPUser to IDIFUser, for returning
            for (LDAPUser u: ldapUsers)
                result.add(this.convertFromLDAPUserToDIFUser(u));
            return result;
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not get all users from LDAP: ", ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByAnyAttribute(java.util.Map)
     */
    public Set<IDIFUser> getUsersByAnyAttribute(Map<String, String> attributes) throws IdentityManagerException
    {
        return this.internalGetUsersByAttributes(attributes, false);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByAttribute(java.lang.String,
     *      java.lang.String)
     */
    public Set<IDIFUser> getUsersByAttribute(String attribute, String value) throws IdentityManagerException
    {
        // Replace all * syntax (LDAP notation) to SQL like syntax
        value = value.replaceAll("\\%", "*");

        Set<IDIFUser> difUsers = new HashSet<IDIFUser>();
        try
        {
            for (LDAPUser ldapUser: getLDAPUtils().findUsersByAttribute(attribute, value))
                difUsers.add(convertFromLDAPUserToDIFUser(ldapUser));
            return difUsers;
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not fetch data for users!", ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByAttributes(java.util.Map)
     */
    public Set<IDIFUser> getUsersByAttributes(Map<String, String> attributes) throws IdentityManagerException
    {
        return this.internalGetUsersByAttributes(attributes, true);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByAttributes(java.util.Map,
     *      Pagination)
     */
    public Set<IDIFUser> getUsersByAttributes(Map<String, String> attributes, Pagination page)
            throws IdentityManagerException
    {
        // Replace all * syntax (LDAP notation) to SQL like syntax
        for (Entry<String, String> entry: attributes.entrySet())
            attributes.put(entry.getKey(), entry.getValue().replaceAll("\\%", "*"));

        Set<LDAPUser> ldapUsers = null;
        Set<IDIFUser> result = null;

        try
        {
            // HISTORY: FSouto: add 1 to the page to return, since findUsers starts counting from 1 instead of 0
            // Viegas: I found that this was not correct and 1 is received and thus this code skiped the first page!
            ldapUsers = this.getLDAPUtils().findUsersByAttributes(attributes, page.getRowsPerPage(), page.getPage());
            result = new HashSet<IDIFUser>();

            // convert from LDAPUser to IDIFUser, for returning
            for (LDAPUser u: ldapUsers)
                result.add(this.convertFromLDAPUserToDIFUser(u));
            return result;
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not get all users from LDAP: ", ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#getUsersByEmail(java.lang.String)
     */
    public Set<IDIFUser> getUsersByEmail(String value) throws IdentityManagerException
    {
        return getUsersByAttribute(getLDAPUtils().getMailAttributeName(), value);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#groupExists(java.lang.String)
     */
    public boolean groupExists(String groupID) throws IdentityManagerException
    {
        try
        {
            return getLDAPUtils().groupExists(groupID);
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not retrieve information for group " + groupID + "!",
                    ldapOperationException);
        }
    }

    /**
     * Inner logic to fiend users by attributes
     * 
     * @param attributes
     *            A map of attributes to be searched, where the keys are the attribute names, and the values are the
     *            attribute values.
     * @param allFields
     *            if T will return users that match ALL attribute values. If F will return users that match at least one
     *            attribute value
     * @return A set containing the users that obey all of the specified criteria
     * @throws IdentityManagerException
     *             If the operation can't be executed
     */
    private Set<IDIFUser> internalGetUsersByAttributes(Map<String, String> attributes, boolean allFields)
            throws IdentityManagerException
    {
        // Replace all * syntax (LDAP notation) to SQL like syntax
        for (Entry<String, String> entry: attributes.entrySet())
            attributes.put(entry.getKey(), entry.getValue().replaceAll("\\%", "*"));

        Set<LDAPUser> ldapUsers = null;
        Set<IDIFUser> result = null;
        try
        {
            // HISTORY: FSouto: add 1 to the page to return, since findUsers starts counting from 1 instead of 0
            // Viegas: I found that this was not correct and 1 is received and thus this code skipped the first page!
            if (allFields)
                ldapUsers = this.getLDAPUtils().findUsersByAttributes(attributes);
            else
                ldapUsers = this.getLDAPUtils().findUsersByAnyAttribute(attributes);

            result = new HashSet<IDIFUser>();

            // convert from LDAPUser to IDIFUser, for returning
            for (LDAPUser u: ldapUsers)
                result.add(this.convertFromLDAPUserToDIFUser(u));
            return result;
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not get all users from LDAP: ", ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#isIdentityValid(java.lang.String,
     *      java.lang.String)
     */
    public boolean isIdentityValid(String userID, String suppliedPassword) throws IdentityManagerException
    {
        try
        {
            return getLDAPUtils().isIdentityValid(userID, suppliedPassword);
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not verify identity of user " + userID + "!",
                    ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#isReadOnly()
     */
    public boolean isReadOnly()
    {
        return this.getLDAPUtils().isReadOnly();
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#isUserInGroup(java.lang.String,
     *      java.lang.String)
     */
    public boolean isUserInGroup(String userID, String groupID) throws IdentityManagerException
    {
        try
        {
            return getLDAPUtils().isUserInGroup(groupID, userID);
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException(
                    "Could not check if user " + userID + " belongs to group " + groupID + "!", ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManagerPrivate#persistUserAttribute(java.lang.String,
     *      java.lang.String, java.lang.Object)
     */
    synchronized public void persistUserAttribute(String userID, String attributeID, Object attributeValue)
            throws IdentityManagerException
    {
        /* Never saves the attribute DOMAIN_NAME_USER_ATTRIBUTE, there no need to persist it. */
        if (!DOMAIN_NAME_USER_ATTRIBUTE.equals(attributeID))
        {
            try
            {
                // ...add it to the LDAPUser parameters (K=LDAP attribute name)
                getLDAPUtils().setUserAttribute(userID, attributeID, attributeValue.toString());
            }
            catch (LDAPOperationException ldapOperationException)
            {
                throw new IdentityManagerLDAPException("Could not update attribute with ID: " + attributeID
                        + " with value : " + attributeValue + " for user with ID: " + userID + "!",
                        ldapOperationException);
            }
        }

    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManagerPrivate#persistUserAttributes(java.lang.String,
     *      java.util.Map)
     */
    public void persistUserAttributes(String userID, Map<String, Object> attributes) throws IdentityManagerException
    {

        for (String attributeID: attributes.keySet())
        {
            updateUserAttribute(userID, attributeID, attributes.get(attributeID));
        }

    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#removeGroup(java.lang.String)
     */
    public void removeGroup(String groupID) throws IdentityManagerException
    {
        try
        {
            LDAPGroup ldapGroup = getLDAPUtils().findGroupByCommonName(groupID);

            if (ldapGroup != null)
            {

                getLDAPUtils().removeGroup(groupID);
                authorizationManager.revokeAllAccessFromGroup(groupID);

                // Update the cache;
                ldapGroupDNMappingCache.remove(ldapGroup.getDistinguishedName());
            }
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not remove group " + groupID + "!", ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#removeUser(java.lang.String)
     */
    public void removeUser(String userID) throws IdentityManagerException
    {
        try
        {
            getLDAPUtils().removeUser(userID);
            authorizationManager.revokeAllAccessFromUser(userID);
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not remove user " + userID + "!", ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#removeUserFromGroup(java.lang.String,
     *      java.lang.String)
     */
    public void removeUserFromGroup(String userID, String groupID) throws IdentityManagerException
    {
        try
        {
            getLDAPUtils().removeUserFromGroup(groupID, userID);
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not remove user " + userID + " from group " + groupID + "!",
                    ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#resetIdentityManager() Forces LDAPUtils
     *      re-instantiation (when the LDAP utils changes in runtime, e.g. unit tests) and configurations reloading.
     */
    public void resetIdentityManager()
    {
        if (this.ldapUtils != null)
        {
            this.ldapUtils.resetConfigurations();
        }
        this.configurations = null;
        this.ldapUtils = null;
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#updateGroup(pt.digitalis.dif.controller.security.objects.IDIFGroup)
     */
    public void updateGroup(IDIFGroup existingGroup) throws IdentityManagerException
    {
        try
        {
            getLDAPUtils().updateGroup(convertFromDIFGroupToLDAPGroup(existingGroup), existingGroup.getID());
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not update group " + existingGroup.getID() + "!",
                    ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#updateUser(pt.digitalis.dif.controller.security.objects.IDIFUser,
     *      java.lang.String)
     */
    public void updateUser(IDIFUser existingUser, String userID) throws IdentityManagerException
    {
        try
        {
            existingUser.cleanCache();

            getLDAPUtils().updateUser(this.convertFromDIFUserToLDAPUser(existingUser), userID);
            existingUser.refresh();

        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not update user " + existingUser.getID() + "!",
                    ldapOperationException);
        }
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#updateUserAttribute(java.lang.String,
     *      java.lang.String, java.lang.Object)
     */
    synchronized public void updateUserAttribute(String userID, String attributeID, Object attributeValue)
            throws IdentityManagerException
    {
        persistUserAttribute(userID, attributeID, attributeValue);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#updateUserAttributes(java.lang.String,
     *      java.util.Map)
     */
    synchronized public void updateUserAttributes(String userID, Map<String, Object> attributes)
            throws IdentityManagerException
    {
        this.persistUserAttributes(userID, attributes);
    }

    /**
     * @see pt.digitalis.dif.controller.security.managers.IIdentityManager#userExists(java.lang.String)
     */
    public boolean userExists(String userID) throws IdentityManagerException
    {
        try
        {
            return (userID == null ? false : getLDAPUtils().userExists(userID));
        }
        catch (LDAPOperationException ldapOperationException)
        {
            throw new IdentityManagerLDAPException("Could not retrieve information for user " + userID + "!",
                    ldapOperationException);
        }
    }

}
