/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.identity.ldap;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.utils.config.ConfigurationException;
import pt.digitalis.utils.config.IConfigurations;
import pt.digitalis.utils.config.annotations.ConfigIgnore;
import pt.digitalis.utils.ldap.LDAPConfigurations;

/**
 * Holds the business-to-LDAP-attributes mapping configuration.
 *
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created 2008/09/18
 */
public class AttributeConfigurations {

    /** The name of the Properties key that holds the name of the LDAP attribute to store bulk parameters. */
    final static public String BULK_PARAMETER_PROPERTY_KEY = "BulkParameterAttributeName";

    /** Holds the name of the bulk parameter attribute name. */
    private String bulkParameterAttributeName;

    /**
     * Maps the business-to-LDAP attributes.<K=business, V=LDAP>
     */
    private Map<String, String> business2LDAPMap = new HashMap<String, String>();

    /**
     * Maps the LDAP-attributes-to-business parameters <K=LDAP, V=business>
     */
    private Map<String, String> LDAP2BusinessMap = new HashMap<String, String>();

    /**
     * Default ctor.
     * 
     * @throws ConfigurationException
     */
    public AttributeConfigurations() throws ConfigurationException
    {
        IConfigurations configAPI = DIFIoCRegistry.getRegistry().getImplementation(IConfigurations.class);

        Properties props = new Properties();
        props = configAPI.readConfiguration("dif2", "IdManagerConfigurations");

        if (props.containsKey(BULK_PARAMETER_PROPERTY_KEY))
        {
            bulkParameterAttributeName = props.getProperty(BULK_PARAMETER_PROPERTY_KEY);
        }
        else
            bulkParameterAttributeName = DIFIoCRegistry.getRegistry().getImplementation(IConfigurations.class)
                    .readConfiguration(LDAPConfigurations.class).getBulkParametersAttributeName();

        for (Object key: props.keySet())
        {
            // Exclude bulk parameter attribute name from maps
            if (!key.equals(bulkParameterAttributeName))
            {
                addMappingPair(key.toString(), props.get(key).toString());
            }
        }
    }

    /**
     * Adds a new business-to-LDAP mapping to the configurations, if such mapping does not exists yet.
     *
     * @param businessParameterName
     *            the business parameter name
     * @param ldapAttributeName
     *            the LDAP attribute name
     */
    public void addMappingPair(String businessParameterName, String ldapAttributeName)
    {
        if (!this.business2LDAPMap.containsKey(businessParameterName)
                && !this.LDAP2BusinessMap.containsKey(ldapAttributeName))
        {
            this.business2LDAPMap.put(businessParameterName, ldapAttributeName);
            this.LDAP2BusinessMap.put(ldapAttributeName, businessParameterName);
        }
    }

    /**
     * Checks if the supplied business parameter name is mapped.
     *
     * @param businessParameterName
     *            the business parameter name
     * @return T if the supplied business parameter name is mapped, F otherwise
     */
    public boolean containsBusinessParameter(String businessParameterName)
    {
        return this.business2LDAPMap.containsKey(businessParameterName);
    }

    /**
     * Checks if the supplied LDAP attribute name is mapped.
     *
     * @param ldapAttributeName
     *            the LDAP attribute name
     * @return T if the supplied LDAP attribute name is mapped, F otherwise
     */
    public boolean containsLDAPAttribute(String ldapAttributeName)
    {
        return this.LDAP2BusinessMap.containsKey(ldapAttributeName);
    }

    /**
     * Inspector for the 'bulkParameterAttributeName' attribute.
     *
     * @return the bulkParameterAttributeName value
     */
    public String getBulkParameterAttributeName()
    {
        return this.bulkParameterAttributeName;
    }

    /**
     * Returns the name of the LDAP attribute that maps to the supplied business parameter name.
     *
     * @param businessParameterName
     *            the business parameter name
     * @return the LDAP attribute name that maps to the supplied business parameter name
     */
    @ConfigIgnore
    public String getBusiness2LDAPMapping(String businessParameterName)
    {
        return this.business2LDAPMap.get(businessParameterName);
    }

    /**
     * Returns the name of the business parameter that maps to the supplied LDAP attribute name.
     *
     * @param ldapAttributeName
     *            the LDAP attribute name
     * @return the business parameter that maps to the supplied LDAP attribute name.
     */
    @ConfigIgnore
    public String getLDAP2BusinessMapping(String ldapAttributeName)
    {

        return this.LDAP2BusinessMap.get(ldapAttributeName);
    }

    /**
     * Removes an existing mapping.
     *
     * @param businessParameterName
     *            the business parameter name
     * @param ldapAttributeName
     *            the LDAP attribute name
     */
    public void removeMapping(String businessParameterName, String ldapAttributeName)
    {
        if (this.business2LDAPMap.containsKey(businessParameterName)
                && this.LDAP2BusinessMap.containsKey(ldapAttributeName))
        {
            this.business2LDAPMap.remove(businessParameterName);
            this.LDAP2BusinessMap.remove(ldapAttributeName);
        }
    }

    /**
     * Modifier for the 'bulkParameterAttributeName' attribute.
     *
     * @param bulkParameterAttributeName
     *            the new bulkParameterAttributeName value to set
     */
    public void setBulkParameterAttributeName(String bulkParameterAttributeName)
    {
        this.bulkParameterAttributeName = bulkParameterAttributeName;
    }
}
