package pt.digitalis.dif.identity.ldap.sanitycheck;

import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.identity.ldap.IdentityManagerLDAPImpl;
import pt.digitalis.dif.ioc.DIFDefaultModulesConfiguration;
import pt.digitalis.dif.sanitycheck.AbstractSanityCheckTestSuite;
import pt.digitalis.dif.sanitycheck.ExecutionResult;
import pt.digitalis.dif.sanitycheck.TestResult;
import pt.digitalis.dif.sanitycheck.annotations.SanityCheckTest;
import pt.digitalis.utils.config.ConfigurationException;
import pt.digitalis.utils.ldap.exception.LDAPOperationException;

import java.util.Map;
import java.util.Map.Entry;

/**
 * @author Galaio da Silva <a href="mailto:jgalaio@digitalis.pt">jgalaio@digitalis.pt</a><br/>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 6 de Jul de 2011
 */
public class CheckDIF2LDAPConfigurations extends AbstractSanityCheckTestSuite
{

    /** IdentityManager */
    private IdentityManagerLDAPImpl identityManagerImpl = new IdentityManagerLDAPImpl();

    /**
     * @see pt.digitalis.dif.sanitycheck.AbstractSanityCheckTestSuite#isActive()
     */
    @Override
    public boolean isActive()
    {
        return IdentityManagerLDAPImpl.NAME.equals(DIFDefaultModulesConfiguration.getInstance()
                .getDefaultImplementationIDFor(IIdentityManager.class));
    }

    /**
     * Test sanity for <i>Ldap DIF2 Attributes</i> Group
     *
     * @return <code>SanityUnitResult</code>
     *
     * @exception IdentityManagerException
     * @exception ConfigurationException
     */
    @SanityCheckTest
    public TestResult test1LdapAtttributesDIF2() throws IdentityManagerException, ConfigurationException
    {
        String errorMessage = "";
        TestResult testResult;
        pt.digitalis.utils.ldap.LDAPConfigurations configurations =
                identityManagerImpl.getLDAPUtils().getConfigurations();

        if (configurations.getBaseSearchDN() == null || "".equals(configurations.getBaseSearchDN()))
            errorMessage += "BaseSearchDN, ";

        if (configurations.getBaseUserDN() == null || "".equals(configurations.getBaseUserDN()))
            errorMessage += "BaseUserDN, ";

        if (configurations.getBulkParametersAttributeName() == null ||
            "".equals(configurations.getBulkParametersAttributeName()))
            errorMessage += "ParametersAttributeName, ";

        if (configurations.getDefaultGroupDN() == null || "".equals(configurations.getDefaultGroupDN()))
            errorMessage += "DefaultGroupDN, ";

        if (configurations.getDefaultProfileDN() == null || "".equals(configurations.getDefaultProfileDN()))
            errorMessage += "DefaultProfileDN, ";

        if (configurations.getHostName() == null || "".equals(configurations.getHostName()))
            errorMessage += "HostName, ";

        if (configurations.getPassword() == null || "".equals(configurations.getPassword()))
            errorMessage += "Password, ";

        if (configurations.getPort() == null || "".equals(configurations.getPort()))
            errorMessage += "Port, ";

        if (configurations.getSSLPort() == null || "".equals(configurations.getSSLPort()))
            errorMessage += "SSLPort, ";

        if (configurations.getUserDN() == null || "".equals(configurations.getUserDN()))
            errorMessage += "UserDN, ";

        if (!"".equals(errorMessage))
        {
            errorMessage = errorMessage.substring(0, errorMessage.length() - 2);
            testResult = new TestResult(ExecutionResult.FAILED);
            testResult.setErrorMessage(errorMessage);
        }
        else
        {
            testResult = new TestResult(ExecutionResult.PASSED);
        }

        return testResult;
    }

    /**
     * Test sanity for <i>Ldap DIF2 Connection</i> Group
     *
     * @return <code>SanityUnitResult</code>
     *
     * @exception IdentityManagerException
     * @exception ConfigurationException
     */
    @SanityCheckTest
    public TestResult test2LdapDIF2Connection() throws IdentityManagerException, ConfigurationException
    {
        TestResult testResult;

        try
        {
            identityManagerImpl.getLDAPUtils()
                    .findUserByDistinguishedName(identityManagerImpl.getLDAPUtils().getConfigurations().getUserDN());
            testResult = new TestResult(ExecutionResult.PASSED);
        }
        catch (LDAPOperationException e)
        {
            e.printStackTrace();

            testResult = new TestResult(ExecutionResult.FAILED);
            testResult.setErrorMessage(e.getMessage() + e.getCause());
        }

        return testResult;
    }

    /**
     * Test sanity for <i>Ldap DIF2 Default Group existence</i> Group
     *
     * @return <code>SanityUnitResult</code>
     *
     * @exception IdentityManagerException
     * @exception ConfigurationException
     */
    @SanityCheckTest
    public TestResult test3LdapDefaultGroupDN() throws IdentityManagerException, ConfigurationException
    {

        TestResult testResult;

        try
        {
            identityManagerImpl.getLDAPUtils().findGroupByDistinguishedName(
                    identityManagerImpl.getLDAPUtils().getConfigurations().getDefaultGroupDN());
            testResult = new TestResult(ExecutionResult.PASSED);
        }
        catch (LDAPOperationException e)
        {
            e.printStackTrace();

            testResult = new TestResult(ExecutionResult.FAILED);
            testResult.setErrorMessage(e.getMessage() + e.getCause());
        }

        return testResult;
    }

    /**
     * Test sanity for <i>Ldap DIF2 Default Profile existence</i> Group
     *
     * @return <code>SanityUnitResult</code>
     *
     * @exception IdentityManagerException
     * @exception ConfigurationException
     */
    @SanityCheckTest
    public TestResult test4LdapDefaultprofileDN() throws IdentityManagerException, ConfigurationException
    {
        TestResult testResult;

        try
        {
            identityManagerImpl.getLDAPUtils().findGroupByDistinguishedName(
                    identityManagerImpl.getLDAPUtils().getConfigurations().getDefaultProfileDN());
            testResult = new TestResult(ExecutionResult.PASSED);
        }
        catch (LDAPOperationException e)
        {
            e.printStackTrace();

            testResult = new TestResult(ExecutionResult.FAILED);
            testResult.setErrorMessage(e.getMessage() + e.getCause());
        }

        return testResult;
    }

    /**
     * Test sanity for <i>Ldap DIF2 Group Mappings</i> Group
     *
     * @return <code>SanityUnitResult</code>
     *
     * @exception IdentityManagerException
     * @exception ConfigurationException
     */
    @SanityCheckTest
    public TestResult test5LdapDIF2GroupMappings() throws IdentityManagerException, ConfigurationException
    {
        TestResult testResult;
        String errorMessage = "";

        Map<String, String> groupMappings = identityManagerImpl.getLDAPUtils().getConfigurations().getGroupMappings();

        for (Entry<String, String> entry : groupMappings.entrySet())
        {
            if (entry.getValue() != null && !"".equals(entry.getValue()))
            {
                try
                {
                    identityManagerImpl.getGroup(entry.getValue());
                }
                catch (IdentityManagerException e)
                {
                    errorMessage += entry.getKey() + ", ";
                    e.printStackTrace();
                }
            }
        }

        if (!"".equals(errorMessage))
        {
            errorMessage = errorMessage.substring(0, errorMessage.length() - 2);

            testResult = new TestResult(ExecutionResult.FAILED);
            testResult.setErrorMessage(errorMessage);
        }
        else
        {
            testResult = new TestResult(ExecutionResult.PASSED);
        }

        return testResult;
    }
}