/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dem.managers.impl;

import java.util.List;
import java.util.Map.Entry;

import org.hibernate.Transaction;

import pt.digitalis.dif.dem.Entity;
import pt.digitalis.dif.dem.interfaces.IApplication;
import pt.digitalis.dif.dem.interfaces.IEntity;
import pt.digitalis.dif.dem.interfaces.IProvider;
import pt.digitalis.dif.dem.interfaces.IService;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.impl.model.DIFRepositoryFactory;
import pt.digitalis.dif.dem.managers.impl.model.dao.ICustomMessagesDAO;
import pt.digitalis.dif.dem.managers.impl.model.dao.impl.CustomMessagesDAOImpl;
import pt.digitalis.dif.dem.managers.impl.model.data.CustomMessages;
import pt.digitalis.dif.dem.objects.messages.Message;
import pt.digitalis.dif.dem.objects.messages.MessageList;
import pt.digitalis.dif.dem.objects.messages.MessageTranslations;
import pt.digitalis.utils.common.StringUtils;

/**
 * Extends the MessageManagerImpl class to provide custom messages support.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a><br/>
 * @created 2008/07/30
 */
public class MessagesDatabaseManager extends MessageManagerImpl {

    /** The DAO. */
    ICustomMessagesDAO theDAO = new CustomMessagesDAOImpl();

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#collectEntityMessagesFromRepository(java.lang.Class)
     */
    @Override
    public MessageList collectEntityMessagesFromRepository(Class<?> clazz)
    {
        MessageList messages = super.collectEntityMessagesFromRepository(clazz);
        messages = updateMessageListFromDatabase(messages, clazz.getSimpleName());
        return messages;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#collectEntityMessagesFromRepository(pt.digitalis.dif.dem.Entity,
     *      java.lang.Object)
     */
    @Override
    public MessageList collectEntityMessagesFromRepository(Entity type, Object instance)
    {
        MessageList messages = super.collectEntityMessagesFromRepository(type, instance);
        messages = updateMessageListFromDatabase(messages, ((IEntity) instance).getID());
        return messages;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.impl.MessageManagerImpl#collectEntityMessagesFromRepository(java.lang.String)
     */
    @Override
    public MessageList collectEntityMessagesFromRepository(String messagePath)
    {
        MessageList messages = super.collectEntityMessagesFromRepository(messagePath);
        messages = updateMessageListFromDatabase(messages, messagePath);
        return messages;
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#isPersistent()
     */
    @Override
    public boolean isPersistent()
    {
        return true;
    }

    /**
     * Checks is a given message of a given class can be updated. That is, if it exists on the default message file.
     * 
     * @param clazz
     *            the class to check the messages
     * @param messageID
     *            the message to check
     * @return T if the message can be updated, F otherwise
     */
    private boolean isUpdatableMessage(Class<?> clazz, String messageID)
    {
        return collectEntityMessagesFromRepository(clazz).containsMessageWithID(messageID);
    }

    /**
     * Checks is a given message of a given entity can be updated. That is, if it exists on the default message file.
     * 
     * @param entity
     *            the entity to check the messages
     * @param messageID
     *            the message to check
     * @return T if the message can be updated, F otherwise
     */
    private boolean isUpdatableMessage(IEntity entity, String messageID)
    {
        Entity type = null;

        // Fetch entity interfaces
        Class<?>[] interfaces = entity.getClass().getInterfaces();

        if (interfaces[0].equals(IProvider.class))
            type = Entity.PROVIDER;
        else if (interfaces[0].equals(IApplication.class))
            type = Entity.APPLICATION;
        else if (interfaces[0].equals(IService.class))
            type = Entity.SERVICE;
        else if (interfaces[0].equals(IStage.class))
            type = Entity.STAGE;

        return collectEntityMessagesFromRepository(type, entity).containsMessageWithID(messageID);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#updateMessage(java.lang.Class, java.lang.String,
     *      pt.digitalis.dif.dem.objects.messages.MessageTranslations) Only messages that exist on the default file can
     *      be updated. This implementation enforces this check-up prior to the update.
     */
    @Override
    public void updateMessage(Class<?> clazz, String messageID, MessageTranslations message)
    {
        // Check if the message to update is contained on the entity's default message file and proceed only of so
        if (isUpdatableMessage(clazz, messageID))
            updateMessage(clazz.getSimpleName(), messageID, message);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.IMessageManager#updateMessage(pt.digitalis.dif.dem.interfaces.IEntity,
     *      java.lang.String, pt.digitalis.dif.dem.objects.messages.MessageTranslations) Only messages that exist on the
     *      default file can be updated. This implementation enforces this check-up prior to the update.
     */
    @Override
    public void updateMessage(IEntity entity, String messageID, MessageTranslations message)
    {
        // Check if the message to update is contained on the entity's default message file and proceed only of so
        if (isUpdatableMessage(entity, messageID))
            updateMessage(getEntityID(entity), messageID, message);
    }

    /**
     * @see pt.digitalis.dif.dem.managers.impl.AbstractMessageManager#updateMessage(java.lang.String, java.lang.String,
     *      pt.digitalis.dif.dem.objects.messages.MessageTranslations)
     */
    @Override
    public void updateMessage(String entityID, String messageID, MessageTranslations message)
    {

        for (Entry<String, Message> translation: message.getTranslations().entrySet())
        {
            // Start transaction
            Transaction transaction = DIFRepositoryFactory.getSession().beginTransaction();
            CustomMessages messageOnDB = theDAO.findByEntityAndMessageIDAndLanguage(entityID, messageID,
                    translation.getKey());

            if (messageOnDB == null)
            {
                if (StringUtils.isNotBlank(translation.getValue().getMessage()))
                {
                    // No message on database, insert new record...
                    CustomMessages messageToFind = new CustomMessages();
                    messageToFind.setLanguage(translation.getKey());
                    messageToFind.setEntityId(entityID);
                    messageToFind.setMessageId(messageID);
                    messageToFind.setMessage(translation.getValue().getMessage());
                    theDAO.persist(messageToFind);
                }
            }
            else
            {
                // Update the existing message on the database...
                if (translation.getValue().isCustomized())
                {
                    // Update the existing record...
                    messageOnDB.setMessage(translation.getValue().getMessage());
                    theDAO.merge(messageOnDB);

                }
                else
                {
                    // The message update was a revert to the default message value... delete the customization...
                    theDAO.delete(messageOnDB);
                }
            }
            // End transaction
            transaction.commit();
        }

        // Forces message reloading
        super.invalidateMessages();
    }

    /**
     * Takes a messages list, fetches custom messages from the database and updates the original list with the custom
     * messages.
     * 
     * @param messages
     *            the messages list to update
     * @param entityName
     * @return the updated message list
     */
    private MessageList updateMessageListFromDatabase(MessageList messages, String entityName)
    {

        boolean wasActive = DIFRepositoryFactory.openTransaction();

        // Fetch the custom messages from repository
        List<CustomMessages> customMessages = theDAO.findAllWithEntityName(entityName);

        // Check if the messages exist on the messages list and add just the existing ones
        for (CustomMessages customMessage: customMessages)
        {
            // Fetch message ID
            String messageID = customMessage.getMessageId();
            // Fetch message language
            String language = customMessage.getLanguage();
            Message message;

            if (messages.containsMessageWithID(customMessage.getMessageId())
                    && messages.getMessageTranslations(messageID).getMessage(language) != null)
            {
                // Create new message with original and custom text
                message = new Message(
                        messages.getMessageTranslations(messageID).getMessage(language).getDefaultMessage(),
                        customMessage.getMessage());
            }
            else
            {
                // Create new message from scratch
                message = new Message(customMessage.getMessage());
            }
            // Add it to the list
            messages.addMessageTranslation(messageID, language, message);
        }

        if (!wasActive)
        {
            DIFRepositoryFactory.getSession().getTransaction().commit();
        }

        // Return the updated object
        return messages;
    }
}
