/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.listeners;

import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import pt.digitalis.dif.controller.AbstractDIFDispatcher;
import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.controller.interfaces.IDIFRequest;
import pt.digitalis.dif.controller.interfaces.IDIFResponse;
import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.controller.objects.DIFResponse;
import pt.digitalis.dif.controller.objects.RESTAction;
import pt.digitalis.dif.controller.security.managers.ISessionManager;
import pt.digitalis.dif.exception.controller.ControllerException;
import pt.digitalis.dif.exception.security.IdentityManagerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.presentation.renderers.interfaces.IViewServletRenderer;
import pt.digitalis.dif.utils.extensions.document.IDocumentRepositoryManager;
import pt.digitalis.dif.utils.extensions.document.UnauthorizedDocumentAccess;
import pt.digitalis.dif.utils.http.HttpUtils;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.common.NumericUtils;
import pt.digitalis.utils.common.StringUtils;

/**
 * The listener for Generic Document requests (HTTP channel).
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2008/11/24
 */
public class DocumentListener extends HttpListener {

    /**  */
    private static final long serialVersionUID = 14857394857879979L;

    /**  */
    private IDocumentRepositoryManager documentRepositoryManager = null;

    /**  */
    private IDIFSession session = null;

    /**  */
    private ISessionManager sessionManager = null;

    /**
     * Inspector for the 'documentRepositoryManager' attribute.
     * 
     * @return the documentRepositoryManager value
     */
    protected IDocumentRepositoryManager getDocumentRepositoryManager()
    {
        if (documentRepositoryManager == null)
            documentRepositoryManager = DIFIoCRegistry.getRegistry()
                    .getImplementation(IDocumentRepositoryManager.class);

        return documentRepositoryManager;
    }

    /**
     * Inspector for the 'session' attribute.
     * 
     * @param request
     *            the HTTP servlet Request
     * @return the session value
     */
    protected IDIFSession getSession(HttpServletRequest request)
    {
        if (session == null)
            session = getSessionManager().createSession(HttpUtils.buildSessionId((request).getSession()));

        return session;
    }

    /**
     * Inspector for the 'sessionManager' attribute.
     * 
     * @return the sessionManager value
     */
    protected ISessionManager getSessionManager()
    {
        if (sessionManager == null)
            sessionManager = DIFIoCRegistry.getRegistry().getImplementation(ISessionManager.class);

        return sessionManager;
    }

    /**
     * @param req
     *            the HTTP Servlet Request
     * @return T is authorized acccess
     */
    private boolean isAuthorizedDocument(HttpServletRequest req)
    {
        // If the document is authorized serve it
        return getDocumentRepositoryManager().isDocumentAuthorizedInCurrentSession(getSession(req),
                Long.parseLong(req.getParameter(HTTPConstants.DOCID_PARAMETER)));
    }

    /**
     * @param request
     *            the HTTP request
     * @return F is this request is a component or template request
     * @throws ControllerException
     */
    private boolean isMainRequest(HttpServletRequest request) throws ControllerException
    {
        IDIFRequest difRequest = httpChal.translateRequest(request);
        return !difRequest.isComponentMode() && !difRequest.isHelpMode() && !difRequest.isTemplateMode();
    }

    /**
     * @see pt.digitalis.dif.listeners.HttpListener#processRequest(javax.servlet.http.HttpServletRequest,
     *      javax.servlet.http.HttpServletResponse, pt.digitalis.dif.controller.objects.RESTAction)
     */
    @Override
    protected void processRequest(HttpServletRequest req, HttpServletResponse resp, RESTAction restAction)
            throws ServletException, IOException
    {
        session = null;

        // Document Listener should not validate the client specs
        req.setAttribute(IDIFRequest.CLIENT_VALIDATIONS_ATTRIBUTE_ID, false);

        if (req.getParameter(IDIFRequest.LOG_PARAMETER_ID) != null)
        {
            processServerLog(req, resp, restAction);
        }
        else
        {

            IDIFResponse response = null;
            IViewServletRenderer renderer;
            boolean askedForDocumentInRepository = (req.getParameterMap().get(HTTPConstants.DOCID_PARAMETER) != null);

            try
            {
                // Don't process for templates and inner componentes
                if (!isMainRequest(req))
                    super.processRequest(req, resp, restAction);
                else
                {
                    // Asked for an unauthorized document repository document
                    if (askedForDocumentInRepository && !isAuthorizedDocument(req))
                    {
                        // Report the unauthorized access to log
                        UnauthorizedDocumentAccess unauthorizedException = new UnauthorizedDocumentAccess(
                                getSession(req), NumericUtils.toLong(req.getParameter(HTTPConstants.DOCID_PARAMETER)));
                        DIFLogger.getLogger().warn(unauthorizedException.getRenderedExceptionContext());

                        // Create the response object
                        response = httpChal.serve(req, resp, restAction);
                        // Raise the exception... (will trigger redirect to the exception handler
                        throw unauthorizedException;
                    }
                    // authorized document repository document, or document event generator
                    else if (!askedForDocumentInRepository)
                        response = httpChal.serve(req, resp, restAction);

                    // Get the renderer for the specified view engine and execute it...
                    if (response != null && response.getView() != null)
                        // A view was given... use it to render the result
                        renderer = DIFIoCRegistry.getRegistry().getImplementation(IViewServletRenderer.class,
                                response.getView().getEngine());
                    else
                        // No view... default Document renderer
                        renderer = DIFIoCRegistry.getRegistry().getImplementation(IViewServletRenderer.class, "docs");

                    if (response == null)
                    {
                        response = new DIFResponse();
                        response.setRequest(httpChal.translateRequest(req));
                    }

                    renderer.renderView(response, getServletContext(), req, resp);
                    AbstractDIFDispatcher.performCleanup(null, true);
                }

            }
            catch (Exception exception)
            {
                try
                {
                    AbstractDIFDispatcher.performCleanup(null, false);
                }
                catch (ControllerException e)
                {
                    e.printStackTrace();
                }

                // Safeguard for Runtime exceptions that corrupt the response object
                IDIFRequest difRequest = null;
                if (response != null)
                    difRequest = response.getRequest();

                try
                {
                    response = errorHandler.getDefaultErrorResponse(difRequest, exception);
                }
                catch (Exception e)
                {
                    response = errorHandler.getDefaultErrorResponse(null, exception);
                }

                httpChal.publish(response, req, resp);

                // Get the renderer for the specified view engine and execute it...
                renderer = DIFIoCRegistry.getRegistry().getImplementation(IViewServletRenderer.class,
                        response.getView().getEngine());
                renderer.renderView(response, getServletContext(), req, resp);
            }
        }
    }

    /**
     * Download the server log. If the LOG_PARAMETER_ID parameter is empty, download's the server.log, otherwise try to
     * download a file with the parameter value
     * 
     * @param req
     *            the HTTP request
     * @param resp
     *            the HTTP response
     * @param restAction
     *            the REST action
     * @throws IOException
     */
    private void processServerLog(HttpServletRequest req, HttpServletResponse resp, RESTAction restAction)
            throws IOException
    {
        try
        {
            IDIFSession session = getSession(req);
            if (session != null && session.getUser() != null
                    && session.getUser().getGroupIDs().contains("Administrators"))
            {
                String fileName = "server.log";
                if (StringUtils.isNotBlank(req.getParameter(IDIFRequest.LOG_PARAMETER_ID)))
                {
                    fileName = req.getParameter(IDIFRequest.LOG_PARAMETER_ID);
                }

                String filePath = System.getProperty("jboss.server.log.dir") + "/" + fileName;
                File file = new File(filePath);
                int length = 0;
                ServletOutputStream outStream = resp.getOutputStream();
                ServletContext context = getServletConfig().getServletContext();
                String mimetype = context.getMimeType(filePath);

                // sets response content type
                if (mimetype == null)
                {
                    mimetype = "application/octet-stream";
                }
                resp.setContentType(mimetype);
                resp.setContentLength((int) file.length());

                // sets HTTP header
                resp.setHeader("Content-Length", String.valueOf(file.length()));
                resp.setHeader("Content-Disposition", "attachment; filename=\"" + fileName + "\"");

                byte[] byteBuffer = new byte[4096];
                DataInputStream in = new DataInputStream(new FileInputStream(file));

                // reads the file's bytes and writes them to the response stream
                while ((length = in.read(byteBuffer)) != -1)
                {
                    outStream.write(byteBuffer, 0, length);
                }

                in.close();
                outStream.close();
            }
        }
        catch (IdentityManagerException e)
        {
            e.printStackTrace();
        }
    }
}
