/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.listeners.objects;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;

import net.sf.json.JSONException;
import net.sf.json.JSONObject;

import org.apache.commons.io.IOUtils;

import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.utils.logging.DIFLogger;

/**
 * Overloads the {@link HttpServletRequestWrapper} to perform URL rewriting for REST requests to DIF servlets.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/01/30
 */
public class RESTHttpServletRequestWrapper extends HttpServletRequestWrapper {

    /**  */
    private boolean isAJAX = false;

    /**  */
    private boolean isAsset = false;

    /**  */
    private boolean isDoc = false;

    /**  */
    private boolean isPage = false;

    /**  */
    private boolean isRestCall = false;

    /** The overriden parameters processed */
    @SuppressWarnings({"rawtypes"})
    HashMap overridenParameters = null;

    /**
     * Default constructor
     * 
     * @param request
     */
    public RESTHttpServletRequestWrapper(HttpServletRequest request)
    {
        super(request);
    }

    /**
     * Adds a parameter do the request
     * 
     * @param key
     * @param value
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    public void addParameter(String key, String value)
    {
        if (overridenParameters == null)
            overridenParameters = new HashMap();

        overridenParameters.put(key, value);
    }

    /**
     * @see javax.servlet.ServletRequestWrapper#getParameter(java.lang.String)
     */
    @Override
    public String getParameter(String s)
    {
        if (overridenParameters != null && !requestContainsStage() && overridenParameters.containsKey(s))
        {
            return (String) overridenParameters.get(s);
        }
        return super.getParameter(s);
    }

    /**
     * @see javax.servlet.ServletRequestWrapper#getParameterMap()
     */
    @Override
    @SuppressWarnings({"rawtypes", "unchecked"})
    public Map getParameterMap()
    {
        if (overridenParameters != null && !requestContainsStage())
        {
            Map superMap = new HashMap();
            superMap.putAll(super.getParameterMap());
            superMap.putAll(overridenParameters);
            return superMap;
        }
        return super.getParameterMap();
    }

    /**
     * @see javax.servlet.ServletRequestWrapper#getParameterNames()
     */
    @Override
    @SuppressWarnings("unchecked")
    public Enumeration<?> getParameterNames()
    {
        if (overridenParameters != null && !requestContainsStage())
        {
            List<?> keys = Collections.list(super.getParameterNames());
            keys.addAll(overridenParameters.keySet());
            return Collections.enumeration(keys);
        }
        else
            return super.getParameterNames();
    }

    /**
     * @see javax.servlet.ServletRequestWrapper#getParameterValues(java.lang.String)
     */
    @Override
    public String[] getParameterValues(String s)
    {
        if (overridenParameters != null && !requestContainsStage() && overridenParameters.containsKey(s))
        {
            return new String[] {(String) overridenParameters.get(s)};
        }
        return super.getParameterValues(s);
    }

    /**
     * Inspector for the 'isAJAX' attribute.
     * 
     * @return the isAJAX value
     */
    public boolean isAJAX()
    {
        return isAJAX;
    }

    /**
     * Inspector for the 'isAsset' attribute.
     * 
     * @return the isAsset value
     */
    public boolean isAsset()
    {
        return isAsset;
    }

    /**
     * Inspector for the 'isDoc' attribute.
     * 
     * @return the isDoc value
     */
    public boolean isDoc()
    {
        return isDoc;
    }

    /**
     * Inspector for the 'isPage' attribute.
     * 
     * @return the isPage value
     */
    public boolean isPage()
    {
        return isPage;
    }

    /**
     * Inspector for the 'isRestCall' attribute.
     * 
     * @return the isRestCall value
     */
    public boolean isRestCall()
    {
        return isRestCall;
    }

    /**
     * Processes the REST path to infer all DiF's servlet parameters.<br>
     * Supports the following request types with the given syntax:
     * <ul>
     * <li>Pages: /appContext/page/stageID</li>
     * <li>Documents: /appContext/doc/stageID[/documentID]</li>
     * <li>AJAX: /appContext/ajax/stageID[/eventID[/id]]</li>
     * </ul>
     * The last id parameter for AJAX requests will be passed as an "id" parameter for CRUD actions.<br>
     * All other context will be disregarded.
     */
    public void processRESTParameters()
    {
        String[] args = getRequestURL().toString().split("/");
        List<String> pathTerms = new ArrayList<String>();
        isPage = false;
        isDoc = false;
        isAJAX = false;
        isAsset = false;

        for (int i = 0; i < args.length; i++)
        {
            if (!isAsset)
                isAsset = "asset".equalsIgnoreCase(args[i]);
            if (!isPage)
                isPage = "page".equalsIgnoreCase(args[i]);
            if (!isDoc)
                isDoc = "doc".equalsIgnoreCase(args[i]);
            if (!isAJAX)
                isAJAX = "ajax".equalsIgnoreCase(args[i]);

            if (isPage || isDoc || isAJAX || isAsset)
                pathTerms.add(args[i]);
        }

        isRestCall = pathTerms.size() >= 2;
        addParameter(HTTPConstants.REST_URL_PARAMETER, Boolean.valueOf(isRestCall).toString().toLowerCase());

        // Process request URL...
        if (getRequestURI() != null && super.getParameter(HTTPConstants.STAGE_PARAMETER) == null)
        {
            // for assets infer the asset name
            if (isAsset && pathTerms.size() >= 2)
            {
                pathTerms.remove(0);
                addParameter(HTTPConstants.ASSET_ID_PARAMETER, pathTerms.get(0));
            }

            // for assets infer the asset name
            if (isAsset && pathTerms.size() >= 2)
            {
                pathTerms.remove(0);
                addParameter(HTTPConstants.ASSET_ID_PARAMETER, pathTerms.get(0));
            }

            // Parameters found and a valid path was given...
            if ((isPage || isAJAX || isDoc) && (pathTerms.size() >= 2))
            {
                pathTerms.remove(0);

                // The first is always the stage ID
                addParameter(HTTPConstants.STAGE_PARAMETER, pathTerms.get(0));

                if (pathTerms.size() > 1)
                {
                    if (isDoc)
                    {
                        // The document ID
                        addParameter(HTTPConstants.DOCID_PARAMETER, pathTerms.get(1));
                    }
                    else if (isAJAX)
                    {
                        // The eventID
                        addParameter(HTTPConstants.EVENT_ID, pathTerms.get(1));

                        if (pathTerms.size() > 2)
                            // The instance ID for CRUD actions
                            addParameter("id", pathTerms.get(2));
                    }
                }
            }
        }

        // FIXME: Must test the uncommented part to allow AJAX file uploads. I think this will enabled it
        if (isAJAX) // && !ServletFileUpload.isMultipartContent(this))
        {
            addParameter(HTTPConstants.AJAX_MODE_PARAMETER, "true");

            // Process JSON body if available...
            try
            {
                InputStream is = super.getInputStream();
                StringBuffer buffer = new StringBuffer();
                StringWriter writer = new StringWriter();
                IOUtils.copy(is, writer, "UTF8");

                buffer.append(writer.toString());

                if (buffer.length() > 0)
                {
                    addParameter("json", buffer.toString());

                    // Add all inner JSON request parameter to the current IDIFRequest
                    JSONObject jsonObject;
                    try
                    {
                        jsonObject = JSONObject.fromObject(buffer.toString());

                        for (Object jsonParameterName: jsonObject.keySet())
                            if (!"result".equalsIgnoreCase(jsonParameterName.toString()))
                                addParameter(jsonParameterName.toString(),
                                        jsonObject.getString(jsonParameterName.toString()));
                    }
                    catch (JSONException e)
                    {
                        DIFLogger.getLogger().warn("Was not possible to read JSON parameters from request!");
                        e.printStackTrace();
                    }
                }

            }
            catch (IOException e)
            {
                // Will discard parameters.
            }
        }
    }

    /**
     * @return T if the request contains
     */
    private boolean requestContainsStage()
    {
        return super.getParameter(HTTPConstants.STAGE_PARAMETER) != null;
    }
}
