/**
 * 2014, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.presentation.ajax;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import net.sf.json.JSONSerializer;
import net.sf.json.JsonConfig;
import net.sf.json.processors.JsonBeanProcessor;

import org.apache.commons.lang.StringUtils;

import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.utils.common.BeanInspector;
import pt.digitalis.utils.common.CollectionUtils;
import pt.digitalis.utils.common.IBeanAttributes;
import pt.digitalis.utils.config.annotations.ConfigIgnore;

/**
 * Builds a JSON representation of a given bean and the desired relations
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @param <T>
 *            the bean to translate
 * @created 06/03/2014
 */
public class JSONResponseBean<T extends IBeanAttributes> extends AbstractJSONResponseCommon {

    /**
     * A custom class to customize the bean JSON processor to add some custom fields to the JSON result. These field
     * will be existing attributes in the bean.
     * 
     * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
     * @created 14/03/2014
     */
    private class JsonBeanCustomProcessor implements JsonBeanProcessor {

        /** the custom fields and their attribute path */
        private Map<String, String> customFieldDefs;

        /** The {@link JSONResponseBean} instance */
        private JSONResponseBean<?> jsonResponseBean;

        /**
         * @param jsonResponseBean
         *            the {@link JSONResponseBean} class instance
         * @param customFieldDefs
         *            the custom fields and their attribute path
         */
        public JsonBeanCustomProcessor(JSONResponseBean<?> jsonResponseBean, Map<String, String> customFieldDefs)
        {
            this.jsonResponseBean = jsonResponseBean;
            this.customFieldDefs = customFieldDefs;
        }

        /**
         * @param bean
         * @param config
         * @return the converted object to JSONObject instance
         */
        public JSONObject processBean(Object bean, JsonConfig config)
        {
            JSONObject result = ((JSONArray) JSONSerializer.toJSON(bean,
                    jsonResponseBean.getJsonConfigForCurrentBean(bean.getClass()))).getJSONObject(0);

            for (Entry<String, String> additionalField: customFieldDefs.entrySet())
                result.put(additionalField.getKey(), BeanInspector.getValue(bean, additionalField.getValue()));

            return result;
        }
    }

    /** the bean instance to translate */
    private T bean;

    /** the relations to include in the generated JSON response */
    private Map<Class<?>, Map<String, String>> fieldsToAdd = new HashMap<Class<?>, Map<String, String>>();

    /** the relations to include in the generated JSON response */
    private Map<Class<?>, List<String>> fieldsToExclude = new HashMap<Class<?>, List<String>>();

    /**
     * Default constructor
     * 
     * @param bean
     *            the bean to translate
     */
    public JSONResponseBean(T bean)
    {
        super();

        this.bean = bean;
    }

    /**
     * Declares a relation to be excluded from the JSON object representation.
     * 
     * @param objectClass
     *            the object class to filter properties
     * @param fieldPairs
     *            the list of attributes to exclude
     */
    public void addClassAditionalAttributes(Class<?> objectClass, String fieldPairs)
    {
        this.fieldsToAdd.put(objectClass, CollectionUtils.keyValueStringToMap(fieldPairs));
    }

    /**
     * Declares a relation to be excluded from the JSON object representation.
     * 
     * @param objectClass
     *            the object class to filter properties
     * @param propertyList
     *            the list of attributes to exclude
     */
    public void addClassExclusions(Class<?> objectClass, List<String> propertyList)
    {
        this.fieldsToExclude.put(objectClass, propertyList);
    }

    /**
     * Declares a relation to be excluded from the JSON object representation.
     * 
     * @param objectClass
     *            the object class to filter properties
     * @param propertyList
     *            the list of attributes to exclude in a comma separated format
     */
    public void addClassExclusions(Class<?> objectClass, String propertyList)
    {
        if (StringUtils.isNotBlank(propertyList))
            this.fieldsToExclude.put(objectClass, Arrays.asList(propertyList.split(",")));
    }

    /**
     * Inspector for the 'bean' attribute.
     * 
     * @return the bean value
     */
    public T getBean()
    {
        return bean;
    }

    /**
     * Inspector for the 'fieldsToExclude' attribute.
     * 
     * @return the fieldsToExclude value
     */
    public Map<Class<?>, List<String>> getFieldsToExclude()
    {
        return fieldsToExclude;
    }

    /**
     * @return the {@link JsonConfig} object with the current {@link JSONResponseBean} configuration defined
     */
    JsonConfig getJsonConfigForCurrentBean()
    {
        return this.getJsonConfigForCurrentBean(null);
    }

    /**
     * Excludes BeanProcessor for given a given class to avoid recursive calls
     * 
     * @param clazzToExclude
     *            the class that will have its bean processor excluded from the configuration
     * @return the {@link JsonConfig} object with the current {@link JSONResponseBean} configuration defined
     */
    JsonConfig getJsonConfigForCurrentBean(Class<?> clazzToExclude)
    {
        String[] attributes = {};

        JsonConfig jsonConfig = new JsonConfig();
        jsonConfig.addIgnoreFieldAnnotation(ConfigIgnore.class);

        for (Entry<Class<?>, Map<String, String>> addFieldConfig: this.fieldsToAdd.entrySet())
            // Adds a new class (bean processor) for every class that we want to add custom fields
            if (clazzToExclude != addFieldConfig.getKey())
                jsonConfig.registerJsonBeanProcessor(addFieldConfig.getKey(), new JsonBeanCustomProcessor(this,
                        addFieldConfig.getValue()));

        for (Entry<Class<?>, List<String>> entry: this.fieldsToExclude.entrySet())
            jsonConfig.registerPropertyExclusions(entry.getKey(), entry.getValue().toArray(attributes));

        return jsonConfig;
    }

    /**
     * @see pt.digitalis.dif.presentation.ajax.IJSONRawResponse#getResponse(pt.digitalis.dif.controller.interfaces.IDIFContext)
     */
    public Map<String, Object> getResponse(IDIFContext context)
    {
        Map<String, Object> map = new HashMap<String, Object>();

        if (this.bean != null)
            map.put("result", JSONObject.fromObject(this.bean, this.getJsonConfigForCurrentBean()));

        return map;
    }
}
