/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.controller.http;

import java.util.Enumeration;
import java.util.List;
import java.util.Map;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.FilenameUtils;
import org.owasp.validator.html.AntiSamy;
import org.owasp.validator.html.CleanResults;
import org.owasp.validator.html.Policy;
import org.owasp.validator.html.PolicyException;
import org.owasp.validator.html.ScanException;

import pt.digitalis.dif.controller.AbstractChAL;
import pt.digitalis.dif.controller.interfaces.IDIFRequest;
import pt.digitalis.dif.controller.interfaces.IDIFResponse;
import pt.digitalis.dif.controller.interfaces.IDIFSession;
import pt.digitalis.dif.controller.objects.ClientDescriptor;
import pt.digitalis.dif.controller.objects.ControllerExecutionStep;
import pt.digitalis.dif.controller.objects.DIFContext;
import pt.digitalis.dif.controller.objects.DIFRequest;
import pt.digitalis.dif.controller.objects.DIFSessionConstants;
import pt.digitalis.dif.controller.security.managers.ISessionManager;
import pt.digitalis.dif.dem.annotations.controller.Channel;
import pt.digitalis.dif.exception.BusinessException;
import pt.digitalis.dif.exception.controller.ControllerException;
import pt.digitalis.dif.exception.security.IllegalFileUploadException;
import pt.digitalis.dif.presentation.config.PresentationConfiguration;
import pt.digitalis.dif.startup.DIFGeneralConfigurationParameters;
import pt.digitalis.dif.utils.extensions.document.DocumentRepositoryEntry;
import pt.digitalis.dif.utils.http.HttpUtils;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.dif.utils.security.HTTPSecurityConfiguration;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.config.IConfigurations;

import com.google.inject.Inject;
import com.newrelic.api.agent.Trace;

/**
 * This class is an AbstractChAL's specialization for the HTTP channel. It defines a dispatcher suited to deal with the
 * HTTP-proceeding requests and overrides the base class defined abstract methods to handle both the HTTP requests and
 * responses.
 * 
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @author Luis Pinto <a href="mailto:lpinto@digitalis.pt">lpinto@digitalis.pt</a>
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created 2007/03/27
 * @see AbstractChAL
 */
@Channel(HTTPConstants.HTTP_CHANNEL_ID)
public class ChALHTTPImpl extends AbstractChAL<HttpServletRequest, HttpServletResponse> {

    /** The tag used to save the Browser Language in the Client Descriptor */
    private final static String BROWSER_LANGUAGE_TAG = "Language";

    /** The tag used to save the Browser Locale in the Client Descriptor */
    private final static String BROWSER_LOCALE_TAG = "Locale";

    /** The tag used to save the Browser Main Version in the Client Descriptor */
    private final static String BROWSER_MAIN_VERSION_TAG = "MainVersion";

    /** The tag used to save the Browser Minor Version in the Client Descriptor */
    private final static String BROWSER_MINOR_VERSION_TAG = "MinorVersion";

    /** The tag used to save the client remote address in the Client Descriptor */
    private final static String CLIENT_REMOTE_ADDR = "ClientRemoteAddress";

    /** The tag used to save the Session ID in the Client Descriptor */
    private final static String JSESSIONID_TAG = "JSESSIONID";

    /**
     * Malicious XSS content parser
     * 
     * @param antiSamy
     *            the antiSamy instance
     * @param difRequest
     *            the current DIF request
     * @param parameterName
     *            the name of the parameter to parse
     * @param parameterValue
     *            the value to parse
     * @return the parsed value (free of malicious XSS content)
     * @throws Exception
     */
    public static String getSanitizedParameterValue(AntiSamy antiSamy, DIFRequest difRequest, String parameterName,
            String parameterValue) throws Exception
    {
        return getSanitizedParameterValue(antiSamy, difRequest, parameterName, parameterValue, true);
    }

    /**
     * Malicious XSS content parser
     * 
     * @param antiSamy
     *            the antiSamy instance
     * @param difRequest
     *            the current DIF request
     * @param parameterName
     *            the name of the parameter to parse
     * @param parameterValue
     *            the value to parse
     * @param reportErrors
     *            if T will report errors to the log
     * @return the parsed value (free of malicious XSS content)
     * @throws Exception
     */
    public static String getSanitizedParameterValue(AntiSamy antiSamy, DIFRequest difRequest, String parameterName,
            String parameterValue, boolean reportErrors) throws Exception
    {
        return getSanitizedParameterValue(antiSamy, difRequest, parameterName, parameterValue, true, null);
    }

    /**
     * Malicious XSS content parser
     * 
     * @param antiSamy
     *            the antiSamy instance
     * @param difRequest
     *            the current DIF request
     * @param parameterName
     *            the name of the parameter to parse
     * @param parameterValue
     *            the value to parse
     * @param reportErrors
     *            if T will report errors to the log
     * @param contextParams
     *            aditional context parameters to add to the issue report
     * @return the parsed value (free of malicious XSS content)
     * @throws Exception
     */
    public static String getSanitizedParameterValue(AntiSamy antiSamy, DIFRequest difRequest, String parameterName,
            String parameterValue, boolean reportErrors, Map<String, Object> contextParams) throws Exception
    {
        if (HTTPSecurityConfiguration.getInstance().getXssParameterSanitization() && antiSamy != null)
        {
            // XSS malicious content parser
            CleanResults scanResult = null;
            String result = parameterValue;

            DIFContext context = null;

            if (difRequest != null)
            {
                context = new DIFContext();
                context.setRequest(difRequest);
            }

            try
            {
                scanResult = antiSamy.scan(parameterValue);
            }
            catch (ScanException e)
            {
                if (reportErrors)
                    DIFLogger.getLogger().warn(
                            new BusinessException(e).addToExceptionContext(context)
                                    .addToExceptionContext(contextParams)
                                    .addToExceptionContext("Parameter ID", parameterName)
                                    .addToExceptionContext("Parameter value", parameterValue)
                                    .getRenderedExceptionContext());
            }
            catch (PolicyException e)
            {
                if (reportErrors)
                    DIFLogger.getLogger().warn(
                            new BusinessException(e).addToExceptionContext(context)
                                    .addToExceptionContext(contextParams)
                                    .addToExceptionContext("Parameter ID", parameterName)
                                    .addToExceptionContext("Original parameter value", parameterValue)
                                    .getRenderedExceptionContext());
            }

            if (scanResult != null && scanResult.getNumberOfErrors() > 0)
            {
                if (reportErrors)
                    DIFLogger.getLogger().warn(
                            new BusinessException("Errors detected on parameter parsing")
                                    .addToExceptionContext(context).addToExceptionContext(contextParams)
                                    .addToExceptionContext("Parameter ID", parameterName)
                                    .addToExceptionContext("Parameter value", parameterValue)
                                    .addToExceptionContext("Sanitized parameter value", result)
                                    .addToExceptionContext("Errors", scanResult.getErrorMessages())
                                    .getRenderedExceptionContext());
                result = scanResult.getCleanHTML();
            }

            return result;
        }
        else
            return parameterValue;
    }

    /** The HTTP configurations record */
    private final HTTPControllerConfiguration config;

    /** The authentication manager */
    ISessionManager sessionManager;

    /**
     * Class default constructor. Instantiates a dispatcher suited to handle the HTTP channel characteristics.
     * 
     * @param sessionManager
     *            the session manager to use
     * @param configurations
     * @throws Exception
     *             if the configurations could not be read
     */
    @Inject
    public ChALHTTPImpl(ISessionManager sessionManager, IConfigurations configurations) throws Exception
    {
        this.sessionManager = sessionManager;
        this.config = configurations.readConfiguration(HTTPControllerConfiguration.class);
    }

    /**
     * @return return the single instance used for this
     * @throws PolicyException
     */
    private AntiSamy getAntiSamyInstance() throws PolicyException
    {
        String POLICY_FILE_LOCATION = "antisamyDIF.xml"; // Path to policy file
        Policy policy = Policy.getInstance(Thread.currentThread().getContextClassLoader()
                .getResourceAsStream(POLICY_FILE_LOCATION)); // Create Policy object
        AntiSamy antiSamy = new AntiSamy(policy);

        return antiSamy;
    }

    /**
     * @see pt.digitalis.dif.controller.AbstractChAL#getClientDescriptor(java.lang.Object)
     * @param originalRequest
     *            the original request
     * @return the client descriptor
     * @throws ControllerException
     *             when any runtime exception is thrown
     */
    @Override
    protected ClientDescriptor getClientDescriptor(HttpServletRequest originalRequest) throws ControllerException
    {
        ClientDescriptor client = null;
        WebBrowserInfo browserInfo = null;

        try
        {
            client = new ClientDescriptor();
            browserInfo = new WebBrowserInfo(originalRequest);
            client.setHardware(browserInfo.getOS());
            client.setName(browserInfo.getName());
            client.setSoftware(browserInfo.getName());
            client.setVendor(browserInfo.getCompany());
            client.setVersion(browserInfo.getVersion());

            client.addAttribute(BROWSER_MAIN_VERSION_TAG, browserInfo.getMainVersion());
            client.addAttribute(BROWSER_MINOR_VERSION_TAG, browserInfo.getMinorVersion());
            client.addAttribute(BROWSER_LANGUAGE_TAG, browserInfo.getLanguage());
            client.addAttribute(BROWSER_LOCALE_TAG, browserInfo.getLocale());
            client.addAttribute(CLIENT_REMOTE_ADDR, browserInfo.getRemoteAddr());
            client.setClientSupportedLanguages(browserInfo.getClientSupportedLanguages());

            // In case client browser validation, verify if browser is supported. Otherwise, will set has valid.
            if (HTTPControllerConfiguration.getInstance().getCompatibleBrowserValidation())
                client.setSupportedBrowser(browserInfo.isSupportedBrowser());
            else
                client.setSupportedBrowser(true);

            /* Saves all the cookies in the client descriptor attributes */
            if (originalRequest.getCookies() != null)
            {
                for (Cookie cookie: originalRequest.getCookies())
                {
                    client.addAttribute(cookie.getName(), cookie.getValue());
                }
            }

            return client;

        }
        catch (RuntimeException runtimeException)
        {
            ControllerException controllerException = new ControllerException(
                    ControllerExecutionStep.CHAL_CLIENT_AGENT_IDENTIFICATION, runtimeException);

            controllerException.addToExceptionContext("Original Request", originalRequest);
            controllerException.addToExceptionContext("Client Agent", client);
            controllerException.addToExceptionContext("Web Browser Info", browserInfo);

            throw controllerException;
        }
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IChAL#publish(pt.digitalis.dif.controller.interfaces.IDIFResponse,
     *      java.lang.Object, java.lang.Object)
     * @param difResponse
     *            the framework's response
     * @param originalRequest
     *            the original request
     * @param finalResponse
     *            the response to send back to the Listener
     */
    @Trace(metricName = "DIF:HTTPChAL:Publish", dispatcher = true)
    public void publish(IDIFResponse difResponse, HttpServletRequest originalRequest, HttpServletResponse finalResponse)
    {
        if (difResponse.getRequest() != null)
        {
            Map<String, Object> clientAttributes = difResponse.getRequest().getClient().getAttributes();
            for (String key: clientAttributes.keySet())
            {
                if (!key.equals(BROWSER_MAIN_VERSION_TAG) && !key.equals(BROWSER_MINOR_VERSION_TAG)
                        && !key.equals(BROWSER_LANGUAGE_TAG) && !key.equals(BROWSER_LOCALE_TAG)
                        && !key.equals(JSESSIONID_TAG))
                {
                    Object cookieValue = clientAttributes.get(key);
                    if (cookieValue == null)
                    {
                        cookieValue = "";
                    }
                    Cookie cookie = new Cookie(key, cookieValue.toString());
                    cookie.setMaxAge(31536000);
                    finalResponse.addCookie(cookie);
                }

            }
        }

        originalRequest.setAttribute(HTTPConstants.RESPONSE_ATTRIBUTE, difResponse);
    }

    /**
     * Selects the active language for the session based on the client and framework's supported languages.
     * 
     * @param clientSupportedLanguages
     *            the client supported languages
     * @return the active language
     */
    private String selectActiveLanguage(List<String> clientSupportedLanguages)
    {
        // Init return
        String result = null;

        // Check the client supported languages
        for (String language: clientSupportedLanguages)
        {
            // The first language supported by DIF will be the active language
            if (messageManager.isLanguageSupported(language))
            {
                result = language;
                break;
            }
        }

        // If none of the client supported languages is supported by the framework, use the default
        if (result == null)
            result = DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage();

        // Return the active language
        return result;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IChAL#translateRequest(java.lang.Object)
     */
    @Trace(metricName = "DIF:HTTPChAL:Translate", dispatcher = true)
    public DIFRequest translateRequest(HttpServletRequest originalRequest) throws ControllerException
    {
        final String COOKIE_LANGUAGE_NAME = "ApplicationLanguageCookie";

        DIFRequest difRequest = null;
        IDIFSession session = null;

        try
        {
            AntiSamy antiSamy = null;

            if (HTTPSecurityConfiguration.getInstance().getXssParameterSanitization())
                antiSamy = this.getAntiSamyInstance();

            // Build a new DIFRequest
            difRequest = new DIFRequest();

            // Fetches the stage from the original request
            String stageID = originalRequest.getParameter(HTTPConstants.STAGE_PARAMETER);

            // Fetches the format from the original request
            String format = originalRequest.getParameter(HTTPConstants.FORMAT_PARAMETER);

            // Pass the default stage if this is null
            if (stageID == null || "".equals(stageID))
                stageID = config.getHomeStageID();
            else
                stageID = getSanitizedParameterValue(antiSamy, difRequest, HTTPConstants.STAGE_PARAMETER, stageID,
                        false);

            difRequest.setStage(stageID);
            difRequest.setFormat(format);

            // Determines if in component mode
            difRequest.setComponentMode(Boolean.parseBoolean(originalRequest
                    .getParameter(HTTPConstants.COMPONENT_MODE_PARAMETER)));

            // Determines if in ajax mode
            difRequest
                    .setAjaxMode(Boolean.parseBoolean(originalRequest.getParameter(HTTPConstants.AJAX_MODE_PARAMETER)));

            // Determines is the request was made in REST URL mode
            difRequest
                    .setRestCall(Boolean.parseBoolean(originalRequest.getParameter(HTTPConstants.REST_URL_PARAMETER)));

            // Determines if in popUp mode
            difRequest.setPopupMode(Boolean.parseBoolean(originalRequest
                    .getParameter(HTTPConstants.POPUP_MODE_PARAMETER)));

            // Determines if in help mode
            difRequest
                    .setHelpMode(Boolean.parseBoolean(originalRequest.getParameter(HTTPConstants.HELP_MODE_PARAMETER)));

            // Determines if in template mode
            difRequest.setTemplateMode(Boolean.parseBoolean(originalRequest
                    .getParameter(HTTPConstants.TEMPLATE_MODE_PARAMETER)));

            difRequest.addAttribute(IDIFRequest.CLIENT_VALIDATIONS_ATTRIBUTE_ID,
                    originalRequest.getAttribute(IDIFRequest.CLIENT_VALIDATIONS_ATTRIBUTE_ID));

            // Save the original HTTP Request inside the DIF request (non advertised functionality, should not be
            // accessed)
            difRequest.addAttribute(DIFRequest.ORIGINAL_REQUEST, originalRequest);

            // Create the client descriptor
            ClientDescriptor clientDescriptor = getClientDescriptor(originalRequest);
            difRequest.setClient(clientDescriptor);

            // Creates a new session or reuses the existent one from the session manager
            session = sessionManager.createSession(HttpUtils.buildSessionId(originalRequest.getSession()));

            // If is a new session (language is null!) choose language, else use the previously defined language
            if (session.getLanguage() == null)
            {
                // If exists a cookie with the language choose this
                if (difRequest.getClient().getAttribute(COOKIE_LANGUAGE_NAME) != null
                        && !PresentationConfiguration.getInstance().getIgnoreBrowserDefaultLanguage())
                    session.setLanguage(difRequest.getClient().getAttribute(COOKIE_LANGUAGE_NAME).toString());
                // If cookie not exist, check the language parameter override. If parameter different null, set this
                // language, else choose client language
                else if (PresentationConfiguration.getInstance().getIgnoreBrowserDefaultLanguage() != null
                        && PresentationConfiguration.getInstance().getIgnoreBrowserDefaultLanguage())
                    session.setLanguage(DIFGeneralConfigurationParameters.getInstance().getDefaultLanguage());
                else
                    session.setLanguage(selectActiveLanguage(clientDescriptor.getClientSupportedLanguages()));
            }

            // Check if a given language was forced
            String language = originalRequest.getParameter(HTTPConstants.LANGUAGE);

            // If a language was forced and the new language is supported, set it on session (otherwise use the
            // previously defined language)
            if (language != null && messageManager.isLanguageSupported(language))
            {
                // Set's the language cookie
                difRequest.getClient().addAttribute(COOKIE_LANGUAGE_NAME, language);
                session.setLanguage(language);
            }

            // Check if a given WebUIMode has been forced
            String webUIMode = originalRequest.getParameter(HTTPConstants.WEBUI_MODE);
            String webUIDebugMode = originalRequest.getParameter(HTTPConstants.WEBUI_DEBUG_MODE);
            String webUICompatMode = originalRequest.getParameter(HTTPConstants.WEBUI_COMPAT_MODE);

            if (webUIMode != null)
                session.addAttribute(HTTPConstants.WEBUI_MODE, webUIMode);
            if (webUIDebugMode != null)
                session.addAttribute(HTTPConstants.WEBUI_DEBUG_MODE, webUIDebugMode);
            if (webUICompatMode != null)
                session.addAttribute(HTTPConstants.WEBUI_COMPAT_MODE, webUICompatMode);

            // Set the session on the request
            difRequest.setSession(session);

            // See if a dif form has been submited to place all it's fields in the request
            String allFormFields = originalRequest.getParameter(HTTPConstants.FORM_FIELD_NAMES);

            if (StringUtils.isNotBlank(allFormFields))
            {
                String[] fields = allFormFields.split(",");
                for (String fieldName: fields)
                    difRequest.addParameter(fieldName.toLowerCase(), null);
            }

            // Handle file upload forms...
            if (ServletFileUpload.isMultipartContent(originalRequest))
            {
                Integer maxUploadFileSize = (Integer) session.getAttribute(DIFSessionConstants.MAX_DOCUMENT_SIZE);

                if (maxUploadFileSize == null)
                {
                    throw new IllegalFileUploadException(ControllerExecutionStep.CHAL_TRANSLATE_REQUEST);
                }

                // Get the multi-part attributes and files...
                DiskFileItemFactory diskFileItemFactory = new DiskFileItemFactory();
                ServletFileUpload servletFileUpload = new ServletFileUpload(diskFileItemFactory);

                try
                {
                    /*
                     * All the file upload inputs register in the session parameter MAX_DOCUMENT_SIZE the value for
                     * maximum file size. At this point we read the session parameter and establish a limit for the
                     * request parser. If one of the files exceed the size, a exceptions thrown. The problem with this
                     * solution is that the request isn't parsed and so we do not have the request parameters. This
                     * prevents the use of the parameter errors because it needs the form parameter and other stuff.
                     */

                    // * Had to comment this previous implementation. The stage froze while rendering the JSP document
                    // template.
                    // if (maxUploadFileSize != null)
                    // /* The size is in kilobytes and must be converted to bites */
                    // servletFileUpload.setFileSizeMax(maxUploadFileSize * 1024);

                    @SuppressWarnings("unchecked")
                    List<FileItem> fileItemsList = servletFileUpload.parseRequest(originalRequest);

                    String parameterName = null;
                    String parameterValue = null;

                    String originalFileName = null;
                    // String contentType = null;
                    byte[] content = null;

                    for (FileItem fileItem: fileItemsList)
                    {
                        if (fileItem.isFormField())
                        {
                            // If it is a normal form field...
                            parameterName = fileItem.getFieldName();
                            parameterValue = fileItem.getString(HTTPControllerConfiguration.getInstance().getCharset());

                            if (parameterValue != null && !"".equals(parameterValue))
                                difRequest
                                        .addParameter(
                                                parameterName,
                                                getSanitizedParameterValue(antiSamy, difRequest, parameterName,
                                                        parameterValue));
                        }
                        else
                        {
                            // Workarround for commented* block above.
                            // /* The size is in kilobytes and must be converted to bites */
                            if (fileItem.getSize() > maxUploadFileSize * 1024)
                                throw new FileUploadException("The file " + fileItem.getName()
                                        + " exceeds its maximum permitted size of " + maxUploadFileSize * 1024
                                        + " characters");

                            // If it is a file field...
                            else if (fileItem.getSize() != 0)
                            {
                                parameterName = fileItem.getFieldName();
                                content = fileItem.get();
                                originalFileName = FilenameUtils.getName(fileItem.getName());
                                // contentType = fileItem.getContentType();
                                DocumentRepositoryEntry doc = new DocumentRepositoryEntry();

                                doc.setName(parameterName);
                                doc.setFileName(originalFileName);
                                doc.calculateMimeType();
                                doc.setBytes(content);

                                if (session.isLogged())
                                    doc.setCreatorID(session.getUser().getID());

                                difRequest.addParameter(parameterName, doc);
                            }
                        }
                    }
                }
                catch (FileUploadException e)
                {
                    e.printStackTrace();
                    difRequest.addAttribute(HTTPConstants.UPLOAD_FILE_SIZE_ERROR, true);
                }
            }

            /* Add original request parameters from HTTPRequest to DIFRequest */
            Enumeration<?> parameterNames = originalRequest.getParameterNames();

            // All parameters
            if (parameterNames != null)
            {
                while (parameterNames.hasMoreElements())
                {
                    String parameterName = (String) parameterNames.nextElement();
                    String parameterValue = originalRequest.getParameter(parameterName);

                    // Must set value as null for (null,"","null") in order for the new value to override previous
                    // values in session or declared default values in @Parameter
                    if (StringUtils.isNotEmpty(parameterValue) && !"null".equalsIgnoreCase(parameterValue))
                        difRequest.addParameter(parameterName,
                                getSanitizedParameterValue(antiSamy, difRequest, parameterName, parameterValue));
                    else
                        difRequest.addParameter(parameterName, null);
                }
            }

            /* Add original request attributes from to DIFRequest */
            Enumeration<?> attributeNames = originalRequest.getAttributeNames();

            if (attributeNames != null)
            {
                while (attributeNames.hasMoreElements())
                {
                    String attributeName = (String) attributeNames.nextElement();
                    difRequest.addAttribute(attributeName, originalRequest.getAttribute(attributeName));
                }
            }

            // TODO: To verify the best place to put this code
            if ("true".equals(difRequest.getParameter(IDIFRequest.LOGOUT_PARAMETER_ID)))
            {
                originalRequest.getSession().removeAttribute("NetpaUserPreferencesSession");
            }

            return difRequest;

        }
        catch (Exception exception)
        {
            ControllerException controllerException = new ControllerException(
                    ControllerExecutionStep.CHAL_TRANSLATE_REQUEST, exception);

            controllerException.addToExceptionContext("Original Request", originalRequest);
            controllerException.addToExceptionContext("DIF request", difRequest);
            controllerException.addToExceptionContext("DIF Session", session);

            throw controllerException;
        }
    }

    /**
     * @see pt.digitalis.dif.controller.AbstractChAL#validateRequest(java.lang.Object) for further details
     * @param originalRequest
     *            The original request specialization received from the client.
     * @return T case the request is well formed, F otherwise
     * @throws ControllerException
     *             when any runtime exception is thrown
     */
    @Override
    protected boolean validateRequest(HttpServletRequest originalRequest) throws ControllerException
    {
        return true;
    }
}
