/**
 * 2007, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */
package pt.digitalis.dif.controller.http;

import pt.digitalis.dif.controller.AbstractControllerErrorHandler;
import pt.digitalis.dif.controller.AbstractDIFDispatcher;
import pt.digitalis.dif.controller.ExceptionHandlers;
import pt.digitalis.dif.controller.interfaces.IDIFDispatcher;
import pt.digitalis.dif.controller.interfaces.IDIFRequest;
import pt.digitalis.dif.controller.interfaces.IDIFResponse;
import pt.digitalis.dif.controller.objects.DIFResponse;
import pt.digitalis.dif.dem.objects.ViewObject;
import pt.digitalis.dif.dem.objects.ViewType;
import pt.digitalis.dif.exception.controller.ControllerException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.utils.config.IConfigurations;

import com.google.inject.Inject;

/**
 * HTTP Requests Controller Error Handler.
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @created Dec 21, 2007
 */
public class DispatcherErrorHandlerHTTPImpl extends AbstractControllerErrorHandler {

    /** The HTTP configurations record */
    private final HTTPControllerConfiguration config;

    /** Http AbstractChAL instance */
    private final IDIFDispatcher httpDispatcher;

    /**
     * Default constructor
     * 
     * @param configurations
     * @throws Exception
     *             if the configuration could not be read
     */
    @Inject
    public DispatcherErrorHandlerHTTPImpl(IConfigurations configurations) throws Exception
    {
        this.httpDispatcher = DIFIoCRegistry.getRegistry().getImplementation(IDIFDispatcher.class, "http");
        this.config = configurations.readConfiguration(HTTPControllerConfiguration.class);
    }

    /**
     * @see pt.digitalis.dif.controller.AbstractControllerErrorHandler#buildDynamicErrorResponse(pt.digitalis.dif.controller.interfaces.IDIFRequest,
     *      java.lang.String, java.lang.Exception)
     */
    @Override
    protected IDIFResponse buildDynamicErrorResponse(IDIFRequest originalrequest, String errorStageID,
            Exception exception)
    {
        // Use the original request and change the stage from the original to the error stage.
        IDIFRequest request = originalrequest;
        IDIFResponse response = null;

        if (errorStageID == null)
            errorStageID = config.getDynamicErrorStageID();

        request.setStage(errorStageID);

        if (exception instanceof ControllerException)
        {
            ControllerException controllerException = (ControllerException) exception;

            if (controllerException.getMessage().equals(AbstractDIFDispatcher.INVALID_STAGE_ERROR_MESSAGE))
                request.setStage(config.getBadStageStageID());
        }

        try
        {
            response = httpDispatcher.dispatch(request);
        }
        catch (Exception e)
        {
            // If DIF could not generate a dynamic error page try a static one
            response = buildStaticErrorResponse(originalrequest, exception);
        }

        return responseWithContext(response, originalrequest, exception);
    }

    /**
     * @see pt.digitalis.dif.controller.AbstractControllerErrorHandler#buildStaticErrorResponse(pt.digitalis.dif.controller.interfaces.IDIFRequest,
     *      java.lang.Exception)
     */
    @Override
    protected IDIFResponse buildStaticErrorResponse(IDIFRequest originalRequest, Exception exception)
    {

        ViewObject view = new ViewObject("jsp", ViewType.ERROR, config.getStaticErrorPage(), false);

        IDIFResponse response = new DIFResponse();
        response.setView(view);
        response.addStageResult(ORIGINAL_REQUEST, originalRequest);
        response.addStageResult(EXCEPTION, exception);

        return response;
    }

    /**
     * @see pt.digitalis.dif.controller.interfaces.IDispatcherErrorHandler#getDefaultErrorResponse(pt.digitalis.dif.controller.interfaces.IDIFRequest,
     *      java.lang.Exception)
     */
    public IDIFResponse getDefaultErrorResponse(IDIFRequest originalrequest, Exception exception)
    {

        // Get the error handler registered in the system
        String errorStageID = ExceptionHandlers.getExceptionHandler(exception);

        if (originalrequest != null)
            originalrequest.addAttribute(ExceptionHandlers.RAISED_EXCEPTION_ATTRIBUTE, exception);

        // No handler, render the default error page
        if (errorStageID == null)
            errorStageID = config.getDynamicErrorStageID();

        if (originalrequest == null || errorStageID.equals(originalrequest.getStage()))
            return buildStaticErrorResponse(null, exception);
        else
            return buildDynamicErrorResponse(originalrequest, errorStageID, exception);
    }
}
