/**
 * 2008, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.dem.annotations.presentation;

import java.util.Collection;
import java.util.Map;

import pt.digitalis.dif.codegen.CGAncillaries;
import pt.digitalis.dif.codegen.util.ClassEnhancementContext;
import pt.digitalis.dif.codegen.util.ClassMethodEnhancement;
import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.dem.DEMAnnotationLogic;
import pt.digitalis.dif.dem.annotations.AnnotationTags;
import pt.digitalis.dif.dem.managers.IMessageManager;
import pt.digitalis.dif.dem.managers.impl.UsageIssuesManagerImpl;
import pt.digitalis.dif.dem.objects.EventType;
import pt.digitalis.dif.dem.objects.issues.IssueScope;
import pt.digitalis.dif.dem.objects.issues.IssueType;
import pt.digitalis.dif.dem.objects.messages.Message;
import pt.digitalis.dif.dem.objects.parameters.ParameterScope;
import pt.digitalis.dif.dem.objects.parameters.types.StringParameter;
import pt.digitalis.dif.exception.codegen.DIFCodeGenerationException;
import pt.digitalis.dif.ioc.DIFIoCRegistry;
import pt.digitalis.dif.presentation.ajax.IJSONRawResponse;
import pt.digitalis.dif.utils.extensions.ICaptcha;
import pt.digitalis.utils.bytecode.holders.AnnotationHolder;
import pt.digitalis.utils.bytecode.holders.AnnotationMemberValueHolder;
import pt.digitalis.utils.bytecode.holders.MethodHolder;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.utils.inspection.exception.ResourceNotFoundException;

/**
 * DiF Presentation Layer Annotation Logic
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2008/09/18
 */
public class PresentationAnnotationLogic extends DEMAnnotationLogic {

    /**
     * Constructor
     * 
     * @param annotationFQName
     *            the annotation fully qualified name
     * @param annotationName
     *            the annotation simple name
     * @param isPrimary
     *            if the annotation is primary
     */
    public PresentationAnnotationLogic(String annotationFQName, String annotationName, boolean isPrimary)
    {
        super(annotationFQName, annotationName, isPrimary);
    }

    /**
     * @see pt.digitalis.dif.dem.DEMAnnotationLogic#addSourceCodeForAnnotation(pt.digitalis.dif.codegen.util.ClassEnhancementContext,
     *      pt.digitalis.utils.bytecode.holders.AnnotationHolder, pt.digitalis.utils.bytecode.holders.MethodHolder)
     */
    @Override
    public void addSourceCodeForAnnotation(ClassEnhancementContext classEnhancementContext,
            AnnotationHolder annotation, MethodHolder method) throws ResourceNotFoundException,
            DIFCodeGenerationException
    {
        try
        {
            // Get annotation and method info
            String annotationName = annotation.getName();
            String methodName = method.getName();
            String signature = method.getSignature();

            if (annotationName.equals(OnSubmitSaveAction.class.getCanonicalName()))
            {
                // Register the new EventHandler for the passed form name
                String formName = annotation.getMembers().get("value").toString();
                String code = getCodeForExecutionMethod(signature, methodName);

                // Add the call in the "handler switch" method
                if (code != null)
                {
                    Collection<Message> saveTranslations = DIFIoCRegistry.getRegistry()
                            .getImplementation(IMessageManager.class).collectEntityMessagesFromRepository("Form")
                            .getMessageTranslations("save").getTranslations().values();
                    StringBuffer saveMessageList = new StringBuffer();
                    for (Message message: saveTranslations)
                    {
                        saveMessageList.append(message.getMessage());
                        saveMessageList.append("|");
                    }

                    classEnhancementContext.addEnhancement(
                            CGAncillaries.STAGE_PROXY_ID,
                            CGAncillaries.STAGE_EVENT_HANDLERS_BUILDER,
                            "addEvent(getTemplateResources().getEventType(\""
                                    + EventType.FORM_SUBMIT_SAVE_ACTION.toString() + "\"), \"" + formName + "\");");

                    // Must ensure that the SUBMIT_SAVE_ACTION is always validated before the general SUBMIT
                    ClassMethodEnhancement enhancement = classEnhancementContext.getClassEnhancements()
                            .get(CGAncillaries.STAGE_INSTANCE_ID).getMethodEnhancements()
                            .get(CGAncillaries.CALL_EXEC_ONEVENT_METHOD);

                    String codeLine = "if ((\"" + EventType.FORM_SUBMIT.toString() + "\".equals($2.toString())) && (\""
                            + formName + "\".equals($3)) && (\"" + saveMessageList
                            + "\".contains(getTemplateResources().getRequestParameter($1,\""
                            + HTTPConstants.SUBMIT_ACTION + "\") + \"|\"))) " + code;
                    int pos = enhancement.getSource().indexOf("if ((\"" + EventType.FORM_SUBMIT.toString());

                    if (pos != -1)
                    {
                        StringBuilder buffer = new StringBuilder();
                        buffer.append(enhancement.getSource().substring(0, pos));
                        buffer.append(codeLine);
                        buffer.append("\n");
                        buffer.append(enhancement.getSource().substring(pos));

                        enhancement.setSource(buffer);
                        classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID, enhancement);
                    }
                    else
                        classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                                CGAncillaries.CALL_EXEC_ONEVENT_METHOD, codeLine);

                    classEnhancementContext.addEnhancementTerminatorCode(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EXEC_ONEVENT_METHOD, "return null;");
                }
            }
            else if (annotationName.equals(Captcha.class.getCanonicalName()))
            {
                // Register the new EventHandler for the passed form name
                String formName = StringUtils.toStringOrNull(annotation.getMembers().get("formName"));
                Map<String, AnnotationHolder> methodAnnotations = method.getAnnotations();
                AnnotationHolder onSubmitAnnotation = methodAnnotations.get(OnSubmit.class.getName());

                String conditionBegin = "";
                String conditionEnd = "";

                if ((formName == null || AnnotationTags.NONE.equalsIgnoreCase(formName)) && onSubmitAnnotation != null)
                    formName = StringUtils.toStringOrNull(onSubmitAnnotation.getMembers().get("value"));

                if (onSubmitAnnotation == null)
                {
                    // if a validation function. Must guard the ICaptcha usage on this function result
                    conditionBegin = "if (" + methodName + "()) {\n";
                    conditionEnd = "\n}";
                }

                if (formName != null)
                {

                    String code = conditionBegin + "this.declareFeatureActive(\"" + ICaptcha.CAPTCHA_PRESENT + "\");\n"
                            + "this.declareFeatureActive(\"" + ICaptcha.CAPTCHA_PRESENT + ":" + formName + "\");"
                            + conditionEnd;

                    // TODO: Viegas: CSSnet: This must be the last thing to execute. _CG_Finalize?
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, code);
                }

                // Declare parameter...
                String attrInitializer = conditionBegin + "getParameters().addStageParameter(Class.forName(\""
                        + StringParameter.class.getCanonicalName() + "\"), \"" + ICaptcha.CAPTCHA_INPUT_ID
                        + "\", getTemplateResources().getParameterScope(\"" + ParameterScope.REQUEST.toString()
                        + "\"), null, \"required\").setFormLinked(\"" + formName + "\");" + conditionEnd;

                classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                        CGAncillaries.STAGE_INJECTED_ATTRIBUTES_INIT_METHOD_NAME, attrInitializer);
            }
            else if (annotationName.equals(OnSubmit.class.getCanonicalName()))
            {
                // Register the new EventHandler for the passed form name
                String formName = annotation.getMembers().get("value").toString();
                String code = getCodeForExecutionMethod(signature, methodName);

                // Add the call in the "handler switch" method
                if (code != null)
                {
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_PROXY_ID,
                            CGAncillaries.STAGE_EVENT_HANDLERS_BUILDER,
                            "addEvent(getTemplateResources().getEventType(\"" + EventType.FORM_SUBMIT.toString()
                                    + "\"), \"" + formName + "\");");
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EXEC_ONEVENT_METHOD, "if ((\"" + EventType.FORM_SUBMIT.toString()
                                    + "\".equals($2.toString())) && (\"" + formName + "\".equals($3))) " + code);
                    classEnhancementContext.addEnhancementTerminatorCode(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EXEC_ONEVENT_METHOD, "return null;");
                }
            }
            else if (annotationName.equals(OnAJAXSubmit.class.getCanonicalName()))
            {
                // Register the new EventHandler for the passed form name
                String formName = annotation.getMembers().get("value").toString();
                String callCode = getCodeForEventMethodCall(signature, methodName);
                String translateCode = getCodeForAJAXEventMethodTranslation(signature, methodName);

                // Add the call in the "handler switch" method
                if (callCode != null)
                {
                    classEnhancementContext.addEnhancement(
                            CGAncillaries.STAGE_PROXY_ID,
                            CGAncillaries.STAGE_EVENT_HANDLERS_BUILDER,
                            "addEvent(getTemplateResources().getEventType(\""
                                    + EventType.FORM_SUBMIT_AJAX_REQUEST.toString() + "\"), \"" + formName + "\");");

                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EXEC_ONEVENT_METHOD,
                            "if ((\"" + EventType.FORM_SUBMIT_AJAX_REQUEST.toString()
                                    + "\".equals($2.toString())) && (\"" + formName + "\".equals($3))) "
                                    + translateCode);
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EVENT_METHOD, "if ((\"" + EventType.FORM_SUBMIT_AJAX_REQUEST.toString()
                                    + "\".equals($2.toString())) && (\"" + formName + "\".equals($3))) "
                                    + translateCode);

                    classEnhancementContext.addEnhancementTerminatorCode(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EXEC_ONEVENT_METHOD,
                            "System.out.println($1);System.out.println($2);System.out.println($3);return null;");
                    classEnhancementContext.addEnhancementTerminatorCode(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EVENT_METHOD, "return null;");

                }
            }
            else if (annotationName.equals(OnSubmitValidationLogic.class.getCanonicalName()))
            {
                // Register the new EventHandler for the passed form name
                String formName = annotation.getMembers().get("value").toString();
                String callCode = getCodeForEventMethodCall(signature, methodName);
                String translateCode = getCodeForAJAXEventMethodTranslation(signature, methodName);

                // Add the call in the "handler switch" method
                if (callCode != null)
                {
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_PROXY_ID,
                            CGAncillaries.STAGE_EVENT_HANDLERS_BUILDER,
                            "addEvent(getTemplateResources().getEventType(\"" + EventType.FORM_VALIDATION.toString()
                                    + "\"), \"" + formName + "\");");

                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EXEC_ONEVENT_METHOD, "if ((\"" + EventType.FORM_VALIDATION.toString()
                                    + "\".equals($2.toString())) && (\"" + formName + "\".equals($3))) "
                                    + translateCode);
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EVENT_METHOD, "if ((\"" + EventType.FORM_VALIDATION.toString()
                                    + "\".equals($2.toString())) && (\"" + formName + "\".equals($3))) "
                                    + translateCode);

                    classEnhancementContext.addEnhancementTerminatorCode(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EXEC_ONEVENT_METHOD,
                            "System.out.println($1);System.out.println($2);System.out.println($3);return null;");
                    classEnhancementContext.addEnhancementTerminatorCode(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EVENT_METHOD, "return null;");

                }
            }
            else if (annotationName.equals(OnAJAX.class.getCanonicalName()))
            {
                // Register the new EventHandler for the passed AJAX event id
                String eventID = annotation.getMembers().get("value").toString();
                String callCode = getCodeForEventMethodCall(signature, methodName);
                String translateCode = getCodeForAJAXEventMethodTranslation(signature, methodName);

                // Add the call in the "handler switch" method
                if (callCode != null)
                {
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_PROXY_ID,
                            CGAncillaries.STAGE_EVENT_HANDLERS_BUILDER,
                            "addEvent(getTemplateResources().getEventType(\"" + EventType.AJAX_REQUEST.toString()
                                    + "\"), \"" + eventID + "\");");
                    classEnhancementContext
                            .addEnhancement(CGAncillaries.STAGE_INSTANCE_ID, CGAncillaries.CALL_EXEC_ONEVENT_METHOD,
                                    "if ((\"" + EventType.AJAX_REQUEST.toString() + "\".equals($2.toString())) && (\""
                                            + eventID + "\".equals($3))) " + translateCode);
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EVENT_METHOD, "if ((\"" + EventType.AJAX_REQUEST.toString()
                                    + "\".equals($2.toString())) && (\"" + eventID + "\".equals($3))) return "
                                    + callCode + ";");

                    classEnhancementContext.addEnhancementTerminatorCode(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EXEC_ONEVENT_METHOD,
                            "System.out.println($1);System.out.println($2);System.out.println($3);return null;");
                    classEnhancementContext.addEnhancementTerminatorCode(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EVENT_METHOD, "return null;");
                }
            }
            else if (annotationName.equals(OnDocument.class.getCanonicalName()))
            {
                // Register the new EventHandler for the passed Document generation event id
                String eventID = annotation.getMembers().get("value").toString();
                String callCode = getCodeForEventMethodCall(signature, methodName);
                String translateCode = getCodeForDocumentEventMethodTranslation(signature, methodName);

                // Add the call in the "handler switch" method
                if (callCode != null)
                {
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_PROXY_ID,
                            CGAncillaries.STAGE_EVENT_HANDLERS_BUILDER,
                            "addEvent(getTemplateResources().getEventType(\"" + EventType.DOCUMENT_TYPE.toString()
                                    + "\"), \"" + eventID + "\");");
                    classEnhancementContext
                            .addEnhancement(CGAncillaries.STAGE_INSTANCE_ID, CGAncillaries.CALL_EXEC_ONEVENT_METHOD,
                                    "if ((\"" + EventType.DOCUMENT_TYPE.toString() + "\".equals($2.toString())) && (\""
                                            + eventID + "\".equals($3))) " + translateCode);
                    classEnhancementContext.addEnhancement(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EVENT_METHOD, "if ((\"" + EventType.DOCUMENT_TYPE.toString()
                                    + "\".equals($2.toString())) && (\"" + eventID + "\".equals($3))) return "
                                    + callCode + ";");

                    classEnhancementContext.addEnhancementTerminatorCode(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EXEC_ONEVENT_METHOD, "return null;");
                    classEnhancementContext.addEnhancementTerminatorCode(CGAncillaries.STAGE_INSTANCE_ID,
                            CGAncillaries.CALL_EVENT_METHOD, "return null;");
                }
            }
        }
        catch (ResourceNotFoundException e)
        {
            // Rethrow for outer handling
            throw e;
        }
        catch (Exception e)
        {
            DIFCodeGenerationException codeGenException;

            if (e instanceof DIFCodeGenerationException)
            {
                codeGenException = (DIFCodeGenerationException) e;
            }
            else
            {

                codeGenException = new DIFCodeGenerationException(e);
                codeGenException.addToExceptionContext("Original Class Name", classEnhancementContext
                        .getOriginalClassObject().getFQName());
            }

            codeGenException.addToExceptionContext("Annotation", annotation.getName());

            throw codeGenException;
        }
    }

    /**
     * Return the appropriated execute method call for AJAX events, injecting the result in the stages response output
     * node. Also passes the appropriated parameters
     * 
     * @param signature
     *            the method to call signature
     * @param methodName
     *            the method name to call
     * @return the line of code for the calling of the method
     */

    protected String getCodeForAJAXEventMethodTranslation(String signature, String methodName)
    {

        /*
         * IMPLEMENTATION NOTE: Although the annotated method's return is void, IStage#execute(DIFContext) returns a
         * ViewObject, so the code generation must inject a ViewObject return. It returns the default view.
         */
        String result = getCodeForEventMethodCall(signature, methodName);

        // Add return to the result for AJAX rendering (JSON/XML)
        if (signature.endsWith(CGAncillaries.VOID_RETURN))
            result += ";";

        else if (signature.endsWith(CGAncillaries.BOOLEAN_RETURN))
            result = "($1).addStageResult(\"result\", " + Boolean.class.getCanonicalName() + ".valueOf(" + result
                    + "));";

        else if (signature.endsWith(CGAncillaries.MAP_RETURN))
            result = "($1).addStageResult(\"result\", " + result + ");";

        else if (signature.endsWith(CGAncillaries.LIST_RETURN))
            result = "($1).addStageResult(\"result\", " + result + ");";

        else
        {
            String[] signatureSplit = signature.split("\\)L");
            String returnObject = signatureSplit[signatureSplit.length - 1].replaceAll("/", ".").replaceAll(";", "");
            Class<?> classToValidate = null;
            try
            {
                classToValidate = Class.forName(returnObject);
            }
            catch (ClassNotFoundException e)
            {
                e.printStackTrace();
            }

            if (classToValidate != null && IJSONRawResponse.class.isAssignableFrom(classToValidate))
            {
                StringBuffer buffer = new StringBuffer();
                buffer.append(IJSONRawResponse.class.getCanonicalName() + " resultTemp = " + result + "; ");
                buffer.append("if (resultTemp != null) ");

                result = buffer.toString() + "($1).addStageResult(\"JSON\", resultTemp.getResponse(($1)));";
            }
            else
            {
                // Any other object will be added link a result. A string will be added normally, a bean will be parsed
                // for attributes
                result = "($1).addStageResult(\"result\", " + result + ");";
            }
        }
        return "{" + result + " return this.getDefaultView();}";
    }

    /**
     * Return the appropriated execute method call for document events, injecting the result in the stages response
     * output node. Also passes the appropriated parameters
     * 
     * @param signature
     *            the method to call signature
     * @param methodName
     *            the method name to call
     * @return the line of code for the calling of the method
     */

    protected String getCodeForDocumentEventMethodTranslation(String signature, String methodName)
    {

        String result = getCodeForEventMethodCall(signature, methodName);

        // Add return to the result for Document type specific rendering
        result = "($1).addStageResult(\"DIF-DOC\", " + result + ");";

        return "{" + result + " return null;}";
    }

    /**
     * Return the appropriated execute method call. Also passes the appropriated parameters
     * 
     * @param signature
     *            the method to call signature
     * @param methodName
     *            the method name to call
     * @return the line of code for the calling of the method without the enclosing brackets and line terminator
     */

    protected String getCodeForEventMethodCall(String signature, String methodName)
    {

        String result = "";

        // Check method signature
        if (signature.startsWith(CGAncillaries.VOID_ARGS))
            result = "this." + methodName + "()";

        else if (signature.startsWith(CGAncillaries.IDIF_CONTEXT_ARGS))
            result = "this." + methodName + "($1)";

        else
            return "";

        return result;
    }

    /**
     * @see pt.digitalis.dif.dem.DEMAnnotationLogic#validateSpecificRules(java.lang.String,
     *      pt.digitalis.utils.bytecode.holders.MethodHolder)
     */
    @Override
    protected boolean validateSpecificRules(String annotationName, MethodHolder method)
            throws ResourceNotFoundException
    {
        Map<String, AnnotationHolder> methodAnnotations = method.getAnnotations();

        if (Captcha.class.getCanonicalName().contains(annotationName)
                && methodAnnotations.containsKey(Captcha.class.getCanonicalName()))
        {
            final String LINKED_FORM_ID = "formName";

            // Check linkedToForm
            AnnotationMemberValueHolder linkedToForm = null;
            AnnotationHolder captchaAnnotation = methodAnnotations.get(Captcha.class.getName());
            if (captchaAnnotation != null)
                linkedToForm = captchaAnnotation.getMembers().get(LINKED_FORM_ID);

            // Check parent onSubmit
            AnnotationHolder onSubmitAnnotation = methodAnnotations.get(OnSubmit.class.getName());

            if (onSubmitAnnotation == null
                    && (linkedToForm == null || AnnotationTags.NONE.equals(linkedToForm.toString())))
            {
                UsageIssuesManagerImpl.getInstance().addIssue(
                        IssueType.WARN,
                        IssueScope.LOADTIME,
                        method.getParentClassName(),
                        "To use @" + Captcha.class.getSimpleName() + " for method: " + method.getName()
                                + " you must specify the associated form using the " + LINKED_FORM_ID
                                + " attribute or using this annotation in the @" + OnSubmit.class.getSimpleName()
                                + " annotated method", null);
                return false;

            }
            else if (onSubmitAnnotation != null && linkedToForm != null)
            {
                UsageIssuesManagerImpl.getInstance().addIssue(
                        IssueType.WARN,
                        IssueScope.LOADTIME,
                        method.getParentClassName(),
                        "Using @" + Captcha.class.getSimpleName() + " for method: " + method.getName() + ", with the "
                                + LINKED_FORM_ID + " attribute defined, for an @" + OnSubmit.class.getSimpleName()
                                + " annotated method. Form is already defined by the @"
                                + OnSubmit.class.getSimpleName() + ". Remove the " + LINKED_FORM_ID + " attribute",
                        null);
                return false;

            }

            // Validate the signature of the annotated method
            if (onSubmitAnnotation == null && !"()Z".equals(method.getSignature()))
            {
                UsageIssuesManagerImpl
                        .getInstance()
                        .addIssue(
                                IssueType.WARN,
                                IssueScope.LOADTIME,
                                method.getParentClassName(),
                                "@"
                                        + Captcha.class.getSimpleName()
                                        + " annotated methods must have no parameters and a primitive boolean return type. Please refactor the "
                                        + method.getName() + " method", null);
                return false;
            }
        }

        return super.validateSpecificRules(annotationName, method);
    }
}
