package pt.digitalis.dif.controller.http;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.StringTokenizer;

import javax.servlet.http.HttpServletRequest;

import cz.mallat.uasparser.OnlineUpdater;
import cz.mallat.uasparser.UASparser;
import cz.mallat.uasparser.UserAgentInfo;
import pt.digitalis.dif.controller.objects.ControllerExecutionStep;
import pt.digitalis.dif.exception.controller.ControllerException;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.common.StringUtils;
import pt.digitalis.web.ancillaries.WebConstants;

/**
 * Creates an HTTP-request browser info record.
 * 
 * @author Pedro Viegas * href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a>
 * @author Rodrigo Gonalves <a href="mailto:rgoncalves@digitalis.pt">rgoncalves@digitalis.pt</a>
 * @created Dec 17, 2007
 */
public class WebBrowserInfo {

    /** The 'Accept Language' header. */
    final static public String ACCEPT_LANGUAGE_HEADER = "Accept-Language";

    /** UAsparser */
    private static UASparser parser = null;

    /** User Agent Detector Session */
    final static public String USER_AGENT_DETECTOR_SESSION = "User-Agent-Detector-Session";

    static
    {
        try
        {
            parser = new UASparser(OnlineUpdater.getVendoredInputStream());
        }
        catch (IOException e)
        {
            ControllerException controllerException = new ControllerException(
                    ControllerExecutionStep.CHAL_CLIENT_AGENT_IDENTIFICATION, e);

            controllerException.addToExceptionContext("Original Request", e);
            controllerException.addToExceptionContext("Client Agent", e);
            controllerException.addToExceptionContext("Web Browser Info", e);
        }
    }

    /** The default language. */
    private String language = WebConstants.LANG_EN;

    /** The default locale. */
    private Locale locale;

    /** the client remote address */
    private String remoteAddr;

    /** The associated HTTP request. */
    private final HttpServletRequest request;
    /** The supported languages. */
    private List<String> supportedLanguages;

    /** The user agent header value */
    private UserAgentInfo userAgent;

    /**
     * Constructor. Parses the given servlet request to extract the browser info.
     * 
     * @param request
     *            the servlet request to parse
     */
    public WebBrowserInfo(HttpServletRequest request)
    {
        this.request = request;
        this.initialize();
        this.setUserAgent();
        this.setLanguage();
        this.setLocale();

        remoteAddr = StringUtils.nvl(request.getHeader("x-forwarded-for"), request.getRemoteAddr());
    }

    /**
     * Returns the list of supported languages.
     * 
     * @return the list of supported languages
     */
    public List<String> getClientSupportedLanguages()
    {
        return this.supportedLanguages;
    }

    /**
     * @return the company
     */
    public String getCompany()
    {
        if (this.userAgent != null)
        {
            return this.userAgent.getUaCompany();
        }
        else
        {
            return WebConstants.UNKNOWN;
        }
    }

    /**
     * @return the browser language
     */
    public String getLanguage()
    {
        return this.language;
    }

    /**
     * @return the locales
     */
    public Locale getLocale()
    {
        return this.locale;
    }

    /**
     * @return main browser version
     */
    public String getMainVersion()
    {
        String result = WebConstants.UNKNOWN;
        if (this.userAgent != null)
        {
            String version = this.getVersion();
            if (StringUtils.isNotBlank(version))
            {
                String[] splitVersion = version.split("\\.");
                result = splitVersion[0];
            }
        }

        return result;
    }

    /**
     * @return minor version
     */
    public String getMinorVersion()
    {
        String result = WebConstants.UNKNOWN;
        if (this.userAgent != null)
        {
            String version = this.getVersion();
            if (StringUtils.isNotBlank(version))
            {
                String[] splitVersion = version.split("\\.");
                if (splitVersion.length > 1)
                {
                    result = splitVersion[1] + version.substring(version.indexOf("."), version.length());
                }
                else
                {
                    result = "0";
                }
            }
        }

        return result;
    }

    /**
     * @return the browser name
     */
    public String getName()
    {
        if (this.userAgent != null)
            return this.userAgent.getUaFamily();
        else
            return WebConstants.UNKNOWN;
    }

    /**
     * @return the browser OS
     */
    public String getOS()
    {
        if (this.userAgent != null)
            return this.userAgent.getOsName();
        else
            return WebConstants.UNKNOWN;
    }

    /**
     * Inspector for the 'remoteAddr' attribute.
     * 
     * @return the remoteAddr value
     */
    public String getRemoteAddr()
    {
        return remoteAddr;
    }

    /**
     * @return the browser version
     */
    public String getVersion()
    {
        if (this.userAgent != null)
            return this.userAgent.getBrowserVersionInfo();
        else
            return WebConstants.UNKNOWN;
    }

    /**
     *
     */
    public void initialize()
    {
        this.supportedLanguages = new ArrayList<String>();

        String acceptedLanguages = this.request.getHeader(ACCEPT_LANGUAGE_HEADER);

        if (acceptedLanguages != null)
        {

            String[] languages = null;

            // If 'quality factor' is present, replace it for a comma
            if (acceptedLanguages.contains(";q="))
            {
                acceptedLanguages = acceptedLanguages.replaceAll(";q=[0-9].[0-9]", ",");
            }

            // Split string around commas
            languages = acceptedLanguages.split(",");

            // Add each language to the supported languages list
            for (int i = 0; i < languages.length; i++)
            {
                this.supportedLanguages.add(languages[i]);
            }
        }
        else
        {
            this.supportedLanguages.add("pt");
            this.supportedLanguages.add("en");
        }
    }

    /**
     * @return T if the browser is Google Chrome
     */
    public boolean isChrome()
    {
        return this.userAgent != null && this.getName().equals("Chrome");
    }

    /**
     * @return if the browser is google chrome 7 or upper
     */
    private boolean isChrome7More()
    {
        if (isChrome())
        {
            try
            {

                return new Integer(this.getMainVersion()).intValue() >= 7;
            }
            catch (NumberFormatException e)
            {
                return false;
            }
        }
        else
        {
            return false;
        }
    }

    /**
     * @return T if the browser is FF
     */
    public boolean isFF()
    {
        return this.userAgent != null && this.getName().equals("Firefox");
    }

    /**
     * @return if the browser is google firefox 3 or upper
     */
    public boolean isFF3More()
    {
        if (isFF())
        {
            try
            {
                return new Integer(this.getMainVersion()).intValue() >= 3;
            }
            catch (NumberFormatException e)
            {
                return false;
            }
        }
        else
        {
            return false;
        }
    }

    /**
     * @return T if the browser is IE
     */
    public boolean isIE()
    {
        return this.userAgent != null && this.getName().equals("IE");
    }

    /**
     * @return if the browser is Microsoft IE 6
     */
    public boolean isIE6()
    {
        try
        {
            return (isIE() && (Integer.parseInt(getMainVersion()) == 6));
        }
        catch (NumberFormatException e)
        {
            return false;
        }
    }

    /**
     * @return if the browser is Microsoft IE 7
     */
    public boolean isIE7()
    {
        try
        {
            return (isIE() && (Integer.parseInt(getMainVersion()) == 7));
        }
        catch (NumberFormatException e)
        {
            return false;
        }
    }

    /**
     * @return if the browser is Microsoft IE 7
     */
    public boolean isIE8()
    {
        try
        {
            return (isIE() && (Integer.parseInt(getMainVersion()) == 8));
        }
        catch (NumberFormatException e)
        {
            return false;
        }
    }

    /**
     * @return if the browser is Microsoft IE and the version is in between 8 and 10
     */
    public boolean isIEBetween8And11()
    {
        if (isIE())
        {
            try
            {
                Integer verInt = new Integer(this.getMainVersion());
                return (verInt >= 8 && verInt <= 11);
            }
            catch (NumberFormatException e)
            {
                return false;
            }
        }
        else
        {
            return false;
        }
    }

    /**
     * @return T if the browser is Opera
     */
    public boolean isOpera()
    {
        return this.userAgent != null && this.getName().equals("Opera");
    }

    /**
     * @return T if the browser is Safari
     */
    public boolean isSafari()
    {
        return this.userAgent != null && this.getName().equals("Safari");
    }

    /**
     * @return if the browser is safari 5 or upper
     */
    private boolean isSafari5More()
    {
        if (isSafari())
        {
            try
            {
                return new Integer(this.getMainVersion()).intValue() >= 5;
            }
            catch (NumberFormatException e)
            {
                return false;
            }
        }
        else
        {
            return false;
        }
    }

    /**
     * @return T if the browser is supported
     */
    public boolean isSupportedBrowser()
    {
        return (isIEBetween8And11() || isFF3More() || isChrome7More() || isSafari5More());
    }

    /**
     *
     */
    private void setLanguage()
    {
        String prefLanguage = this.request.getHeader(ACCEPT_LANGUAGE_HEADER);

        if (prefLanguage != null)
        {
            String language = null;
            StringTokenizer st = new StringTokenizer(prefLanguage, WebConstants.COMMA);

            int elements = st.countTokens();

            for (int idx = 0; idx < elements; idx++)
            {
                if (this.supportedLanguages.contains((language = st.nextToken())))
                {
                    this.language = language;
                }
            }
        }
        else
        {
            this.language = "en";
        }
    }

    /**
     *
     */
    private void setLocale()
    {
        this.locale = new Locale(this.language, "");
    }

    /**
     * Modifier for the 'remoteAddr' attribute.
     * 
     * @param remoteAddr
     *            the new remoteAddr value to set
     */
    public void setRemoteAddr(String remoteAddr)
    {
        this.remoteAddr = remoteAddr;
    }

    /**
     */
    private void setUserAgent()
    {

        if (this.request.getHeader(WebConstants.USER_AGENT_HEADER) != null)
        {
            try
            {
                this.userAgent = parser.parse(this.request.getHeader(WebConstants.USER_AGENT_HEADER));
            }
            catch (IOException e)
            {
                DIFLogger.getLogger().info(e);
            }
        }
    }
}
