/**
 * - Digitalis Internal Framework v2.0 - (C) 2007, Digitalis Informatica. Distribuicao e Gestao de Informatica, Lda.
 * Estrada de Paco de Arcos num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999
 * http://www.digitalis.pt
 */
package pt.digitalis.dif.listeners;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Calendar;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import pt.digitalis.dif.controller.AbstractDIFDispatcher;
import pt.digitalis.dif.controller.http.HTTPConstants;
import pt.digitalis.dif.controller.objects.RESTAction;
import pt.digitalis.dif.exception.controller.ControllerException;
import pt.digitalis.dif.presentation.assets.AssetManager;
import pt.digitalis.dif.presentation.assets.IAsset;
import pt.digitalis.dif.utils.logging.DIFLogger;

/**
 * The listener for Generic Document requests (HTTP channel).
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2008/11/24
 */
public class AssetListener extends HttpListener {

    /** File read buffer cache size */
    private static final int READ_CACHE = 102400;

    /**  */
    private static final long serialVersionUID = -5129451303399518776L;

    /**
     * @see pt.digitalis.dif.listeners.HttpListener#processRequest(javax.servlet.http.HttpServletRequest,
     *      javax.servlet.http.HttpServletResponse, pt.digitalis.dif.controller.objects.RESTAction)
     */
    @Override
    protected void processRequest(HttpServletRequest req, HttpServletResponse resp, RESTAction restAction)
            throws ServletException, IOException
    {
        String assetID = req.getParameter(HTTPConstants.ASSET_ID_PARAMETER);

        // If a valid asset was passed
        if (assetID != null)
            returnAsset(assetID, req, resp);
    }

    /**
     * Builds the response with the requested asset
     * 
     * @param assetID
     *            the requested asset ID
     * @param req
     *            the HTTP request
     * @param resp
     *            the HTTP response
     */
    private void returnAsset(String assetID, HttpServletRequest req, HttpServletResponse resp)
    {

        IAsset asset = AssetManager.getInstance().getAsset(assetID);

        if (asset == null)
            DIFLogger.getLogger().warn("Requested asset does not exist: " + assetID);
        else
        {
            Calendar expires = Calendar.getInstance();
            expires.add(Calendar.SECOND, asset.getExpirationInSeconds().intValue());
            expires.set(Calendar.MILLISECOND, 0);

            resp.setContentType(asset.getContentType());
            resp.setHeader("Cache-Control", String.format("max-age=%d", asset.getExpirationInSeconds()));
            resp.setDateHeader("Expires", expires.getTime().getTime());
            // resp.setHeader("Vary", "Accept-Encoding");
            resp.setHeader("ETag", assetID);

            // Previous content compare
            String previousToken = req.getHeader("If-None-Match");

            if (previousToken != null && previousToken.equals(assetID))
            {
                DIFLogger.getLogger().debug("Not modified: " + assetID);
                resp.setStatus(HttpServletResponse.SC_NOT_MODIFIED);
                resp.setHeader("Last-Modified", req.getHeader("If-Modified-Since"));
            }
            else
            { // first time through or previous version
                DIFLogger.getLogger().debug("Serving new version of: " + assetID);

                Calendar cal = Calendar.getInstance();
                cal.set(Calendar.MILLISECOND, 0);
                resp.setDateHeader("Last-Modified", cal.getTime().getTime());
            }

            try
            {
                OutputStream out = null;

                if (asset.getCompressContent())
                {
                    // Select the appropriate content encoding based on the
                    // client's Accept-Encoding header. Choose GZIP if the header
                    // includes "gzip". Choose ZIP if the header includes "compress".
                    // Choose no compression otherwise.
                    String encodings = req.getHeader("Accept-Encoding");
                    if (encodings != null && encodings.indexOf("gzip") != -1)
                    {
                        // Go with GZIP
                        resp.setHeader("Content-Encoding", "gzip");
                        out = new GZIPOutputStream(resp.getOutputStream());
                    }
                    else if (encodings != null && encodings.indexOf("compress") != -1)
                    {
                        // Go with ZIP
                        resp.setHeader("Content-Encoding", "x-compress");
                        out = new ZipOutputStream(resp.getOutputStream());
                        ((ZipOutputStream) out).putNextEntry(new ZipEntry("dummy name"));
                    }
                    else
                    {
                        // No compression
                        out = resp.getOutputStream();
                    }

                }
                else
                    out = resp.getOutputStream();

                InputStream input = asset.getResourceStream(req, this.getServletContext());
                Long contentLength = 0L;

                if (input != null)
                {
                    byte[] buf = new byte[READ_CACHE];

                    while (true)
                    {
                        int length = input.read(buf);
                        contentLength += length;

                        if (length < 0)
                            break;
                        out.write(buf, 0, length);
                    }
                }

                out.close();

                // Set content length
                if (contentLength > Integer.MAX_VALUE)
                    resp.setHeader("content-length", "" + contentLength);
                else
                    resp.setContentLength(contentLength.intValue());

                AbstractDIFDispatcher.performCleanup(null, true);

            }
            catch (Exception exception)
            {
                try
                {
                    AbstractDIFDispatcher.performCleanup(null, false);
                }
                catch (ControllerException controllerException)
                {
                    controllerException.printStackTrace();
                }

                exception.printStackTrace();
            }
        }
    }
}
