/**
 * 2010, Digitalis Informatica. All rights reserved. Distribuicao e Gestao de Informatica, Lda. Estrada de Paco de Arcos
 * num.9 - Piso -1 2780-666 Paco de Arcos Telefone: (351) 21 4408990 Fax: (351) 21 4408999 http://www.digitalis.pt
 */

package pt.digitalis.dif.presentation.ajax;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringEscapeUtils;

import net.sf.ezmorph.bean.MorphDynaBean;
import net.sf.json.JSONArray;
import net.sf.json.JSONException;
import net.sf.json.JSONObject;
import pt.digitalis.dif.controller.interfaces.IDIFContext;

/**
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 2010/04/28
 */
public abstract class AbstractJSONResponseCommon implements IJSONRawResponse {

    /** the list of fields that we block change requests. These will be filtered out on JSON input conversion */
    protected List<String> fieldsToBlockChanges = new ArrayList<String>();

    /** the list of fields that to ignore the security validation */
    protected List<String> fieldsToIgnoreValidation = new ArrayList<String>();

    /** the list of fields that we accept change requests. Others will be filtered out on JSON input conversion */
    protected List<String> fieldToAllowChanges = new ArrayList<String>();

    /** Cache for body read operation */
    private Map<String, String> jsonBodyCache = null;

    /** the list of fields that have been rejected by the security validation */
    private List<String> rejectedFieldsFromAuthorizationValidation = null;

    /**
     * Adds the field to ignore validation.
     * 
     * @param fieldName
     *            the field name
     */
    protected void addFieldToIgnoreValidation(String fieldName)
    {
        fieldsToIgnoreValidation.add(fieldName);
    }

    /**
     * Adds a list of fields to a white list of allowed fields in auto DML operations.<br/>
     * If no white or black lists exists, all supported fields are processed (the default behavior).
     * 
     * @param fieldNames
     *            the field names, comma separated
     */
    public void allowChangesToFields(String fieldNames)
    {
        this.fieldToAllowChanges.addAll(Arrays.asList(fieldNames.split(",")));
    }

    /**
     * Adds a list of fields to a black list of blocked fields in auto DML operations.<br/>
     * If no white or black lists exists, all supported fields are processed (the default behavior).
     * 
     * @param fieldNames
     *            the field names, comma separated
     */
    public void blockChangesToFields(String fieldNames)
    {
        this.fieldsToBlockChanges.addAll(Arrays.asList(fieldNames.split(",")));
    }

    /**
     * @see pt.digitalis.dif.presentation.ajax.IJSONRawResponse#getBeanAttributesFromJSONRequestBody(pt.digitalis.dif.controller.interfaces.IDIFContext)
     */
    public Map<String, String> getBeanAttributesFromJSONRequestBody(IDIFContext context)
    {
        return this.getBeanAttributesFromJSONRequestBody(context, new ArrayList<String>());
    }

    /**
     * Parses the request JSON message body to get the data attributes to process
     * 
     * @param context
     *            the current execution context
     * @param declaredFields
     *            the list of declared fields that will not have their name changed according to the notation (aaa_id ->
     *            aaa.id)
     * @return the attributes map
     */
    protected Map<String, String> getBeanAttributesFromJSONRequestBody(IDIFContext context, List<String> declaredFields)
    {
        if (jsonBodyCache == null)
        {
            jsonBodyCache = new HashMap<String, String>();

            Object jsonAttribute = context.getRequest().getParameter("json");

            if (jsonAttribute != null)
            {
                try
                {
                    JSONObject jsonObject = JSONObject.fromObject(jsonAttribute.toString());

                    // If is valid JSON object...
                    // Get the result node from the JSON request
                    if (jsonObject.has("result"))
                        jsonObject = jsonObject.getJSONObject("result");

                    // If there is a result node...
                    if (jsonObject != null)
                    {
                        int totalAttributes = jsonObject.size();
                        JSONArray arrayNames = jsonObject.names();
                        this.rejectedFieldsFromAuthorizationValidation = new ArrayList<String>();

                        MorphDynaBean bean = (MorphDynaBean) JSONObject.toBean(jsonObject);
                        String parsedName;

                        // Add all attributes inside the JSON object
                        for (int i = 0; i < totalAttributes; i++)
                        {
                            if (declaredFields.contains(arrayNames.getString(i)))
                                parsedName = arrayNames.getString(i);
                            else
                                parsedName = arrayNames.getString(i).replaceAll("_", ".");

                            // Empty lists allow or block all fields
                            boolean allow = this.fieldsToIgnoreValidation.contains(parsedName)
                                    || this.fieldToAllowChanges.isEmpty()
                                    || this.fieldToAllowChanges.contains(parsedName);
                            boolean block = !this.fieldsToBlockChanges.isEmpty()
                                    || this.fieldsToBlockChanges.contains(parsedName);

                            // If present in the allow list and not present in the block list...
                            if (allow && !block)
                            {
                                if (bean.get(arrayNames.getString(i)) == null)
                                    jsonBodyCache.put(parsedName, null);
                                else
                                    jsonBodyCache.put(parsedName, StringEscapeUtils
                                            .unescapeHtml(jsonObject.getString(arrayNames.getString(i))));
                            }
                            else
                                this.rejectedFieldsFromAuthorizationValidation.add(parsedName);
                        }
                    }
                }
                catch (JSONException e)
                {
                    e.printStackTrace();
                }
            }
        }

        return jsonBodyCache;
    }

    /**
     * Inspector for the 'rejectedFieldsFromAuthorizationValidation' attribute.
     * 
     * @return the rejectedFieldsFromAuthorizationValidation value
     */
    protected List<String> getRejectedFieldsFromAuthorizationValidation()
    {
        return rejectedFieldsFromAuthorizationValidation;
    }

    /**
     * @return T if any unauthorized field name has been submited in the request and thus rejected
     */
    protected boolean hasSecurityRejectedFieldsInRequest()
    {
        return this.rejectedFieldsFromAuthorizationValidation != null
                && !this.rejectedFieldsFromAuthorizationValidation.isEmpty();

    }
}
