package pt.digitalis.dif.presentation.entities.system.difsso;

import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import com.google.inject.Inject;

import pt.digitalis.dif.controller.http.HTTPControllerConfiguration;
import pt.digitalis.dif.controller.interfaces.IDIFContext;
import pt.digitalis.dif.controller.objects.DIFRequest;
import pt.digitalis.dif.controller.security.managers.IIdentityManager;
import pt.digitalis.dif.controller.security.managers.ISessionManager;
import pt.digitalis.dif.controller.security.managers.ISessionManagerInternal;
import pt.digitalis.dif.controller.security.objects.IDIFUser;
import pt.digitalis.dif.dem.annotations.entities.StageDefinition;
import pt.digitalis.dif.dem.annotations.parameter.Parameter;
import pt.digitalis.dif.dem.annotations.presentation.OnAJAX;
import pt.digitalis.dif.dem.annotations.stage.Context;
import pt.digitalis.dif.dem.annotations.stage.Execute;
import pt.digitalis.dif.dem.annotations.stage.InjectMessages;
import pt.digitalis.dif.dem.interfaces.IStage;
import pt.digitalis.dif.dem.managers.IDEMManager;
import pt.digitalis.dif.utils.ObjectFormatter;
import pt.digitalis.dif.utils.logging.DIFLogger;
import pt.digitalis.utils.common.StringUtils;

/**
 * Receives an SSO request from another DIF application and returns the generated token ID
 * 
 * @author Pedro Viegas <a href="mailto:pviegas@digitalis.pt">pviegas@digitalis.pt</a><br/>
 * @created 02/09/2015
 */
@StageDefinition(name = "Data Provider", service = "ssoservice")
public class SSORequestReceiver {

    /**  */
    public final static String SSO_REQUEST_AJAX_HANDLER_ID = "ssoRequestAjaxHandlerId";

    /**  */
    @Context
    IDIFContext context;

    /**  */
    @Inject
    IDEMManager demManager;

    /** the desired destination stage ID */
    @Parameter(constraints = "required")
    protected String destinationStageID;

    /**  */
    @Inject
    IIdentityManager identityManager;

    /**  */
    @InjectMessages
    Map<String, String> messages;

    /**  */
    @Inject
    protected ISessionManager sessionManager;

    /**  */
    @Parameter
    protected String tokenID;

    /** the user to auto login */
    @Parameter(constraints = "required")
    protected String userID;

    /**
     * Will perform redirection if the received token exists and is valid
     * 
     * @throws Exception
     */
    @Execute
    public void execute() throws Exception
    {
        SSOToken token = SSOManager.getInstance().getTokenByID(tokenID);
        boolean redirectSuccess = false;

        if (!SSOSecurityConfiguration.getInstance().getActive())
        {
            context.addResultMessage("WARN", messages.get("ssoFailed"), messages.get("SSODisabled"), true, true);
        }
        else if (destinationStageID != null && userID != null && token != null)
        {
            if (!token.isExpired() && token.getStageID().equals(destinationStageID) && token.getUserID().equals(userID))
            {
                // Login without password! Trusting token...
                ((ISessionManagerInternal) sessionManager)
                        .logInNoPasswordValidation(context.getSession().getSessionID(), userID, null);

                context.redirectTo(destinationStageID);
                redirectSuccess = true;
            }
            else
            {
                context.addResultMessage("WARN", messages.get("ssoFailed"), messages.get("tokenInvalidOrExpired"), true,
                        true);
            }
        }
        else
        {
            context.addResultMessage("WARN", messages.get("ssoFailed"), messages.get("incorrectOrMissingParameters"),
                    true, true);
        }

        if (!redirectSuccess)
            context.redirectTo(HTTPControllerConfiguration.getInstance().getHomeStageID());
    }

    /**
     * Processes an SSO request from another DIF application
     * 
     * @return the result of the SSO request
     * @throws Exception
     */
    @OnAJAX(SSO_REQUEST_AJAX_HANDLER_ID)
    public SSOResponse processSSORequest() throws Exception
    {
        if (!SSOSecurityConfiguration.getInstance().getActive())
            return new SSOResponse(false, null, messages.get("SSODisabled"));

        HttpServletRequest request = (HttpServletRequest) context.getRequest()
                .getAttribute(DIFRequest.ORIGINAL_REQUEST);

        // Will ignore headers that may contain original IP if accessed through proxies since they are easily faked and
        // could constitute an easy backdoor for hacking. Must have the proxy in the whitelist if other DIF servers are
        // behind a such a proxy. Be careful since this will open up to all proxy users this service
        String callerIP = request.getRemoteAddr();
        DIFLogger.getLogger().debug(
                "SSO Request: [Stage: " + destinationStageID + "; User: " + userID + "] received from " + callerIP);

        // Build the caller IP white list
        List<String> whiteListIPs;

        if (StringUtils.isNotBlank(SSOSecurityConfiguration.getInstance().getServerIPWhiteList()))
            // If configured use the given white list
            whiteListIPs = Arrays.asList(SSOSecurityConfiguration.getInstance().getServerIPWhiteList().split(","));
        else
        {
            // If not configured will accept requests current server IP addresses (all existing network interfaces)
            whiteListIPs = new ArrayList<String>();

            Enumeration<NetworkInterface> e = NetworkInterface.getNetworkInterfaces();
            while (e.hasMoreElements())
            {
                NetworkInterface n = e.nextElement();
                Enumeration<InetAddress> ee = n.getInetAddresses();
                while (ee.hasMoreElements())
                {
                    InetAddress i = ee.nextElement();
                    whiteListIPs.add(i.getHostAddress());
                }
            }
        }
        DIFLogger.getLogger().debug("WhiteList infered: " + whiteListIPs.toString());

        SSOResponse response;
        SSOToken generatedToken = null;

        // Validate the request
        if (whiteListIPs.contains(callerIP))
        {
            IDIFUser requestedUser = identityManager.getUser(userID);
            IStage stage = demManager.getStage(destinationStageID);

            if (requestedUser == null)
            {
                // Given user does not exist
                response = new SSOResponse(false, null, messages.get("userDoesNotExist"));
            }
            else if (stage == null)
            {
                // Requested stage does not exist
                response = new SSOResponse(false, null, messages.get("stageDoesNotExist"));
            }
            else if (!requestedUser.canAccess(stage))
            {
                // Requested stage does not exist
                response = new SSOResponse(false, null, messages.get("noAccessToStage"));
            }
            else
            {
                // Passed all tests, get SSO Token...
                generatedToken = SSOManager.getInstance().newToken(destinationStageID, userID);
                response = new SSOResponse(true, generatedToken.getId());
            }
        }
        else
        {
            // Server not in white IP list
            response = new SSOResponse(false, null, messages.get("notInIPWhiteList"));
        }

        DIFLogger.getLogger()
                .debug("SSO result: " + new ObjectFormatter().addItemIfNotNull("generatedToken", generatedToken)
                        .addItemIfNotNull("response", response).getFormatedObject());

        return response;
    }
}
